#!/usr/bin/python
# Copyright (C) 2008-2015 by LMI Technologies Inc.  All rights reserved.
# Distributed under the terms of the MIT License.
# Redistributed files must retain the above copyright notice.

import argparse
import io
import os
import xml.etree.ElementTree
import struct

class AisCommand:
    aisMagic          = 0x41504954
    spiI2cMasterMagic = 0x41504954
    i2cSlaveXmtStart  = 0x00000058
    spiSlaveXmtStart  = 0x00005853
    spiI2cXmitSync    = 0x58535900
    spiI2cPingDevice  = 0x5853590B
    sectionLoad       = 0x58535901
    requestCRC        = 0x58535902
    enableCRC         = 0x58535903
    disableCRC        = 0x58535904
    jump              = 0x58535905
    jumpAndClose      = 0x58535906
    bootTable         = 0x58535907
    startOver         = 0x58535908
    reserved          = 0x58535909
    sectionFill       = 0x5853590A
    ping              = 0x5853590B
    deviceCfgCommand  = 0x585359F0
    pageSwapCommand   = 0x585359F1
    latchAddress      = 0x585359F2

class AisSection:
    def __init__(self, name, sectionSize, sectionOffset, sectionAddress):
        self.name = name
        self.sectionSize = sectionSize
        self.sectionOffset = sectionOffset
        self.sectionAddress = sectionAddress

def WriteWord(stream, word):
    stream.write(struct.pack('I', word))

def ElfToAis(ofdFileName, inFileName, outFileName, emifBoot = False, configFileName = None):

    if configFileName is not None:
        cfgFile = io.open(configFileName, 'r')
        # one hex command per line (0x12345678), ignoring trailing chars
        commands = [line[0:10] for line in cfgFile.readlines()]
    else:
        commands = []

    inFile = io.open(inFileName, 'rb')
    outFile = io.open(outFileName, 'wb')

    doc = xml.etree.ElementTree.parse(ofdFileName)
    root = doc.getroot()

    objectFileItem = root.find('object_file')
    elfItem = objectFileItem.find('elf')
    sectionTableItem  = elfItem.find('section_table')
    elf32EhdrItem = elfItem.find('elf32_ehdr')

    entryPoint = int(elf32EhdrItem.find('e_entry').text, 16)

    sections = []
    totalDataSize = 0
    for sectionItem in sectionTableItem.findall('section'):

        elf32ShdrItem = sectionItem.find('elf32_shdr')

        sectionNameItem = elf32ShdrItem.find('sh_name_string')
        shTypeItem = elf32ShdrItem.find('sh_type')
        shSizeItem = elf32ShdrItem.find('sh_size')
        shOffsetItem = elf32ShdrItem.find('sh_offset')
        shAddrItem = elf32ShdrItem.find('sh_addr')

        sectionName = sectionNameItem.text
        sectionSize = int(shSizeItem.text, 16)
        sectionOffset = int(shOffsetItem.text, 16)
        sectionAddress = int(shAddrItem.text, 16)

        if ((sectionOffset != 0) and
            (sectionSize != 0) and
            (sectionAddress != 0) and
            (shTypeItem.text != 'SHT_NOBITS')):

                sections.append(AisSection(sectionName, sectionSize, sectionOffset, sectionAddress))
                totalDataSize += sectionSize

    if emifBoot:
        WriteWord(outFile, 0)

    WriteWord(outFile, AisCommand.aisMagic)

    for command in commands:
        WriteWord(outFile, int(command, 16))

    for section in sections:
        sectionSize = section.sectionSize

        wordCount = (sectionSize + 3) >> 2
        sectionSize  = wordCount << 2

        WriteWord(outFile, AisCommand.sectionLoad)
        WriteWord(outFile, section.sectionAddress)
        WriteWord(outFile, sectionSize)

        inFile.seek(section.sectionOffset)
        outFile.write(inFile.read(4 * wordCount))

    WriteWord(outFile, AisCommand.jumpAndClose)
    WriteWord(outFile, entryPoint)
    WriteWord(outFile, len(sections))
    WriteWord(outFile, totalDataSize)

if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='Convert C64x ELF to DM647 AIS.')

    parser.add_argument('ofd', help='OFD input file.')
    parser.add_argument('input', help='ELF input file.')
    parser.add_argument('output', help='AIS output file.')
    parser.add_argument('--emif', action='store_true', help='Specifies whether code is loaded from emif by ROM, to be used for boot loader images.')
    parser.add_argument('--cfg', help='AIS command input file.')

    args = parser.parse_args()
    ElfToAis(args.ofd, args.input, args.output, args.emif, args.cfg)
