/** 
 * @file    kVsModelDefStore.h
 * @brief   Declares the kVsModelDefStore class. 
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_MODEL_DEF_STORE_H
#define K_VISION_MODEL_DEF_STORE_H

#include <kVision/Common/kVision.h>
#include <kVision/Vs/kVsModelDef.h>
#include <kFireSync/Utils/kSensorIdentity.h>

/**
 * @class       kVsModelDefStore
 * @extends     kObject
 * @ingroup     kVision-Vs
 * @brief       
 */
typedef kObject kVsModelDefStore;


/** 
 * Constructs a kVsModelDefStore object.
 *
 * @public                  @memberof kVsModelDefStore
 * @param   initializer     Destination for the constructed object handle.
 * @param   path            Path to the ModelDefs xml file.
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status. 
 */
kVsFx(kStatus) kVsModelDefStore_Construct(kVsModelDefStore* modelDefStore, const kChar* path, kAlloc allocator);

/**
* Returns a list of all models matching the provided kVsModelDef object.
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @param    filter          Object to match against.
* @param    models          Handle for constructed list of matching models as kVsModelDef objects.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsModelDefStore_FilterModels(kVsModelDefStore modelDefStore, kVsModelDef filter, kArrayList* models);

/**
* Returns a list of all models matching a specific key-value.
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @param    fieldName       Name of the key (ie. PartNumber).
* @param    fieldValue      Value to match against.
* @param    models          Handle for constructed list of matching models as kVsModelDef objects.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsModelDefStore_FilterModelsByField(kVsModelDefStore modelDefStore, const kChar* fieldName, const kChar* fieldValue, kArrayList* models);

/**
* Attempts to filter down to one and only one unique matching model.
*
* This function will return kERROR_NOT_FOUND if no matching schema is found.
* This function will return kERROR_PARAMETER if more than 1 matching schema is found.
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @param    fieldName       Name of the key (ie. PartNumber).
* @param    fieldValue      Value to match against.
* @param    model           Destination for constructed kVsModelDef object.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsModelDefStore_FilterSingleModelByField(kVsModelDefStore modelDefStore, const kChar* fieldName, const kChar* fieldValue, kVsModelDef* model);

/**
* Attempts to filter down to one and only one unique matching model.
*
* This function will return kERROR_NOT_FOUND if no matching schema is found.
* This function will return kERROR_PARAMETER if more than 1 matching schema is found.
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @param    filter          Filter to match against.
* @param    model           Destination for constructed kVsModelDef object.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsModelDefStore_FilterSingleModel(kVsModelDefStore modelDefStore, kVsModelDef filter, kVsModelDef* model);

/**
* Returns the number of models stored.
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @return                   Number of stored models.
*/
kVsFx(kSize) kVsModelDefStore_ModelCount(kVsModelDefStore modelDefStore);

/**
* Returns the the model at an index. Note that this is not a cloned model, so changing its fields will
* modify the stored entries. This should be avoided, and it is up to the user to clone the object.
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @param    index           Index to access.
* @return                   kVsModelDef object.
*/
kVsFx(kVsModelDef) kVsModelDefStore_ModelAt(kVsModelDefStore modelDefStore, kSize index);

/**
* Helper function for filtering a model by PartNumber. Tries to filter against Hardware Part Number first, and by Part Number
* if no match found. 
*
* If no match is found for either Hardware Part Number or Part Number, returns a kERROR_NOT_FOUND
*
* @public                   @memberof kVsModelDefStore
* @param    modelDefStore   kVsModelDefStore object.
* @param    partNumber      PartNumber value to search for.
* @param    model           Destination for constructed kVsModelDef object.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsModelDefStore_FilterSingleModelByPartNumber(kVsModelDefStore store, const kChar* partNumber, kVsModelDef* model);

/**
* Helper function for filtering a model by PartNumber and Hardware Partnumber. Tries to filter against Part Number and Hardware Part Number.
*
* If no match is found for the Hardware Part Number and Part Number, returns a kERROR_NOT_FOUND. If more than one match is found, returns a kERROR_PARAMETER.
*
* @public                       @memberof kVsModelDefStore
* @param    modelDefStore       kVsModelDefStore object.
* @param    partNumber          PartNumber value to search for.
* @param    hardwarePartNumber  Hardware PartNumber value to search for.
* @param    model               Destination for constructed kVsModelDef object.
* @return                       Operation status.
*/
kVsFx(kStatus) kVsModelDefStore_FilterSingleModelByPartNumberAndHardwarePartNumber(kVsModelDefStore store, const kChar* partNumber, const kChar* hardwarePartNumber, kVsModelDef* model);

#include <kVision/Vs/kVsModelDefStore.x.h>

#endif  /* K_VISION_MODEL_DEF_STORE_H */
