/**
* @file     kVsImageStatsAlg.h
* @brief    Declares the kVsImageStats class
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef K_VISION_IMAGE_STATS_ALG_H
#define K_VISION_IMAGE_STATS_ALG_H

#include <kVision/Common/kVision.h>
#include <kVision/Vs/kVsImageStats.h>

/**
* @class       kVsImageStatsAlg
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       image statistics algorithms class, this class is the algorithm that will generate all image
*              statistics for DynamicExposure and TrackingWindow behavior. The algorithm itself requires raw
*              image from the sensor camera, and constructed array list of spots and kVsImageStats object.
*              The output of this algorithm will be the kVsImageStats object that contains all image statistics
*              contains all image statistics. 
*
*              Parameters Need to be Set:
*              AverageWindow: averageWindow size of image statistics algorithm, this is one parameter used by sobel ACCG or ARCG
*              EdgeWindow: edgeWindow size of image statistics algorithm, this is one parameter of sobel ACCG or ARCG
*              Threshold: threshold of image statistics algorithm, this is one parameter of ACCG or ARCG
*              WidthThreshold: spot widthThreshold of image statistics algorithm, this is one parameter of sobel ACCG or ARCG
*              MinWidth: minWidth of image statistic algorithms
*              MaxWidth: maxWidth of image statistic algorithms
*              MinSum: the minSum of image statistics algorithm, this is minimum pixel value sum for a spot of sobel ACCG or ARCG
*              IsSobel: isSobel flag for determining the algorithms using sobel algorithms or non-sobel algorithms
*              RowBased: the rowBased flag for determining ACCG or ARCG
*              
*              
*/
typedef kObject kVsImageStatsAlg;

/**
* Constructs a kVsImageStatsAlg object.
*
* @public                  @memberof kVsImageStatsAlg
* @param    imageStatsAlg  Destination for the constructed object handle
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_Construct(kVsImageStatsAlg* imageStatsAlg, kSize spotsCapacity, kAlloc allocator);

/**
* Execute the Image status algorithms to generate image status for dynamic exposure and tracking window simulation, 
* by default, isSobel and rowBase flags are false. To change these two flags, use corresponding getter and setters. 
* Image status algorithm follows Gocator PL algorithm equivalent to current FSS implementation.
* It output all needed data into kVsImageStats obj, which is the container for all image statistics
* and can be used for dynamic exposure and tracking window simulation.
* Before executing the algorithms, please make sure all parameters for ACCG (ARCG) or sobel ACCG ARCG are all set
* 
*
* @public                @memberof kVsImageStatsAlg
* @param   imageStatsAlg ImageStatsAlg object
* @param   imageStats    ImageStats object
* @param   image         Raw image from sensor camera
* @return                Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_Execute(kVsImageStatsAlg imageStatsAlg, kVsImageStats imageStats,kImage image);

/**
* Sets the averageWindow size of image statistics algorithm, this is one parameter used by sobel ACCG or ARCG
*
* @public                @memberof kVsImageStatsAlg
* @param   imageStatsAlg ImageStatsAlg object
* @param   averageWindow AverageWindow size of sobel ACCG or ARCG
* @return                Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetAverageWindow(kVsImageStatsAlg imageStatsAlg, kSize averageWindow);

/**
* Gets the averageWindow of image statistic algorithm
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 AverageWindow.
*/
kVsFx(kSize) kVsImageStatsAlg_AverageWindow(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the edgeWindow size of image statistics algorithm, this is one parameter of sobel ACCG or ARCG
*
* @public                @memberof kVsImageStatsAlg
* @param   imageStatsAlg ImageStatsAlg object
* @param   edgeWindow    EdgeWindow size of sobel ACCG or ARCG 
* @return                Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetEdgeWindow(kVsImageStatsAlg imageStatsAlg, kSize edgeWindow);

/**
* Gets the edgeWindow of image statistic algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 EdgeWindow.
*/
kVsFx(kSize) kVsImageStatsAlg_EdgeWindow(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the threshold of image statistics algorithm, this is one parameter of ACCG or ARCG
*
* @public                @memberof kVsImageStatsAlg
* @param   imageStatsAlg ImageStatsAlg object
* @param   threshold     Threshold of ACCG or ARCG 
* @return                Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetThreshold(kVsImageStatsAlg imageStatsAlg, kSize threshold);

/**
* Gets the threshold of image statistic algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 Threshold.
*/
kVsFx(kSize) kVsImageStatsAlg_Threshold(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the spot widthThreshold of image statistics algorithm, this is one parameter of sobel ACCG or ARCG
*
* @public                  @memberof kVsImageStatsAlg
* @param   imageStatsAlg   ImageStatsAlg object
* @param   widthThreshold  WidthThreshold of sobel ACCG or ARCG 
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetWidthThreshold(kVsImageStatsAlg imageStatsAlg, kSize widthThreshold);

/**
* Gets the widthThreshold of image statistic algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 WidthThreshold.
*/
kVsFx(kSize) kVsImageStatsAlg_WidthThreshold(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the minWidth of image statistics algorithm, this is minimum spot width of ACCG or ARCG
*
* @public                  @memberof kVsImageStatsAlg
* @param   imageStatsAlg   ImageStatsAlg object
* @param   minWidth        MinWidth of ACCG or ARCG 
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetMinWidth(kVsImageStatsAlg imageStatsAlg, kSize minWidth);

/**
* Gets the minWidth of image statistic algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 MinWidth.
*/
kVsFx(kSize) kVsImageStatsAlg_MinWidth(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the maxWidth of image statistics algorithm, this is maximum spot width of ACCG or ARCG
*
* @public                  @memberof kVsImageStatsAlg
* @param   imageStatsAlg   ImageStatsAlg object
* @param   maxWidth        MaxWidth of ACCG or ARCG 
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetMaxWidth(kVsImageStatsAlg imageStatsAlg, kSize maxWidth);

/**
* Gets the maxWidth of image statistic algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 MaxWidth.
*/
kVsFx(kSize) kVsImageStatsAlg_MaxWidth(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the minSum of image statistics algorithm, this is minimum pixel value sum for a spot of sobel ACCG or ARCG
*
* @public                  @memberof kVsImageStatsAlg
* @param   imageStatsAlg   ImageStatsAlg object
* @param   minSum          MinSum of sobel ACCG or ARCG 
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetMinSum(kVsImageStatsAlg imageStatsAlg, kSize minSum);

/**
* Gets the minSum of image statistic algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 MinSum.
*/
kVsFx(kSize) kVsImageStatsAlg_MinSum(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the isSobel flag for determining the algorithms using sobel algorithms or non-sobel algorithms
*
* @public                  @memberof kVsImageStatsAlg
* @param   imageStatsAlg   ImageStatsAlg object
* @param   isSobel         IsSobel flag
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetIsSobel(kVsImageStatsAlg imageStatsAlg, kBool isSobel);

/**
* Gets the isSobel flag for determining the algorithms using sobel algorithms or non-sobel algorithms
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 IsSobel.
*/
kVsFx(kBool) kVsImageStatsAlg_IsSobel(kVsImageStatsAlg imageStatsAlg);

/**
* Sets the rowBased flag for determining ACCG or ARCG
*
* @public                  @memberof kVsImageStatsAlg
* @param   imageStatsAlg   ImageStatsAlg object
* @param   rowBased        RowBased flag for determining ACCG or ARCG
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStatsAlg_SetRowBased(kVsImageStatsAlg imageStatsAlg, kBool rowBased);

/**
* Gets the rowBased flag for determining ACCG or ARCG
*
* @public                 @memberof kVsImageStatsAlg
* @param   imageStatsAlg  ImageStatsAlg object
* @return                 RowBased.
*/
kVsFx(kBool) kVsImageStatsAlg_RowBased(kVsImageStatsAlg imageStatsAlg);

#include <kVision/Vs/kVsImageStatsAlg.x.h>

#endif
