/**
 * @file    kS3dSurfaceResampler.h
 * @brief   Declares the kS3dSurfaceResampler class.
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef LMITECH_KVISION_S3D_SURFACERESAMPLER_INCLUDED
#define LMITECH_KVISION_S3D_SURFACERESAMPLER_INCLUDED

#include <kVision/Common/kVision.h>

typedef kObject kS3dSurfaceResampler;

kDeclareClassEx(kVs, kS3dSurfaceResampler, kObject)

/*
    At least for now assume Y coordinate increases along the array rows, while X increases along the columns
*/

typedef struct
{
    kBool useCuda;

    k32u colCountIn; // number of elements in a column
    k32u rowCountIn; // number of elements in a row
    kBool yRows;     // True - rows greatest change in y.  False - rows greatest change in x.
    kBool xDecreasing;
    kBool yDecreasing;  
    k32s outHeight;
    k32s outWidth;
    k32s threadCount;

    kRect3d64f fov;
    kPoint3d64f dataOrigin;
    k64f xResolution;
    k64f yResolution;
    k64f zResolution;
    k64f xStepOut;
    k64f yStepOut;
    k64f xThreshold;
    k64f yThreshold;
    k64f zThreshold;

    kBool intensity;

} kS3dSurfaceResamplerParams;

/** 
 * Constructs a kS3dSurfaceResampler object.
 *
 * @public              @memberof kS3dSurfaceResampler
 * @param   resampler   Destination for the constructed object handle. 
 * @param   params      Parameters.  
 * @param   alloc       Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kVsFx(kStatus) kS3dSurfaceResampler_Construct(kS3dSurfaceResampler* resampler, const kS3dSurfaceResamplerParams* params, kAlloc alloc);

/** 
 * Processes the inputs.
 * 
 * If CUDA is enabled, this function will take care of moving data between the host and the device.
 *
 * @public                  @memberof kS3dSurfaceResampler
 * @param   resampler       kS3dSurfaceResampler object. 
 * @param   inSurface       input surface -- kArray2<kPoint3d16s>.
 * @param   inIntensity     input intensities -- kArray2<k8u>.
 * @param   outSurface      resampled output surface -- kArray2<kPoint3d16s> or kArray2<k16s>
 * @param   outIntensity    resampled output intensities -- kArray2<k8u>.
 * @return                  Operation status. 
 */
kVsFx(kStatus) kS3dSurfaceResampler_Process(kS3dSurfaceResampler resampler, kArray2 inSurface, kArray2 inIntensity, kArray2 outSurface, kArray2 outIntensity);

/** 
 * Processes the inputs.
 * 
 * If CUDA is enabled, this function assumes that the provided arguments have their buffers in device memory space.
 *
 * @public                  @memberof kS3dSurfaceResampler
 * @param   resampler       kS3dSurfaceResampler object. 
 * @param   inSurface       input surface -- kArray2<kPoint3d16s>.
 * @param   inIntensity     input intensities -- kArray2<k8u>.
 * @param   outSurface      resampled output surface -- kArray2<kPoint3d16s> or kArray2<k16s>
 * @param   outIntensity    resampled output intensities -- kArray2<k8u>.
 * @return                  Operation status. 
 */
kVsFx(kStatus) kS3dSurfaceResampler_ProcessEx(kS3dSurfaceResampler resampler, kArray2 inSurface, kArray2 inIntensity, kArray2 outSurface, kArray2 outIntensity);

#endif
