/**
* @file    kS3dStereoProfilerBase.x.h
*
* @internal
* Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef K_VISION_S3D_STEREO_PROFILER_BASE_X_H
#define K_VISION_S3D_STEREO_PROFILER_BASE_X_H

//////////////////////////////////////////////////////////////////////////
// 2D LUT params
// Rectification
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dStereoProfilerLutParams2d
{
    kBool invertX[kS3D_STEREO_CAL_VIEW_COUNT];
    kBool invertY[kS3D_STEREO_CAL_VIEW_COUNT];
    kBool transpose[kS3D_STEREO_CAL_VIEW_COUNT];

    k32u fullImageWidth;
    k32u fullImageHeight;

    kPoint32s spotOffset[kS3D_STEREO_CAL_VIEW_COUNT];
    k32s spotXShift;
    k32s spotYShift;

    k32s projWidth;
    k32s projHeight;

    k32s projXStep;
    k32s projYStep;

    k32s projXMask;
    k32s projYMask;

    k32s projXMax;
    k32s projYMax;

    k32s projXShift;
    k32s projYShift;

    k32s projXRound;
    k32s projYRound;

} kS3dStereoProfilerLutParams2d;

//////////////////////////////////////////////////////////////////////////
// 3D LUT params
// Disparity / Phase
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dStereoProfilerLutParams3d
{
    k32s width;
    k32s height;
    k32s depth;

    k32s xBegin;
    k32s yBegin;
    k32s dBegin;

    k32u xShift;
    k32u yShift;
    k32u dShift;

    k32u xMask;
    k32u yMask;
    k32u dMask;

    k32u xRound;
    k32u yRound;
    k32u dRound;

    // possibly deprecate //
    k32s xStep;
    k32u yStep;
    k32u dStep; // only this step is used and its local to Init ???

} kS3dStereoProfilerLutParams3d; 

//////////////////////////////////////////////////////////////////////////
//
// Hardware specific context object 
// Allocated on host or device
//
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dStereoProfilerHwContextClass
{
    // Lookup //
    kS3dStereoProfilerLutParams3d* disparityParams;
    kPoint3d16s* disparityLut;

    kS3dStereoProfilerLutParams3d* phaseParams;
    kPoint3d16s* phaseLut[kS3D_STEREO_CAL_VIEW_COUNT]; // phase0, phase1

    // Rectification //
    kS3dStereoProfilerLutParams2d* rectParams;
    kPoint16s* rectLut[kS3D_STEREO_CAL_VIEW_COUNT];

} kS3dStereoProfilerHwContextClass;

//////////////////////////////////////////////////////////////////////////
// Class
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dStereoProfilerBaseClass
{
    kObjectClass base;


    // Lookup interpolation //////////////////////////////////////////////
    kS3dStereoProfilerLutParams3d disparityLutParams;
    kArray3 disparityLut;
    
    kS3dStereoProfilerLutParams3d phaseLutParams;
    kArray3 phaseLut[2];
    
    // Rectification / Projection ////////////////////////////////////////    
    kS3dStereoProfilerLutParams2d rectLutParams;
    kArray2 rectLut[2];

    // Hardware context for device or host executions ////////////////////
    kS3dStereoProfilerHwContext hwContext;



    // Generics //////////////////////////////////////////////////////////
    kS3dStripeLookupType lookupType;

    k64f resolution;
    k64f projectionResolution; 

    kRect3d64f lutLimits;
    kRect3d64f activeAreaRoi;
    kL3dTransform3d transform;

    kSize leftViewIndex;

    // Temperature expansion correction //////////////////////////////////
    k64f referenceTemperature;
    kBool correctExpansion;
    k64f temperature;
    
    k64f viewDepth;         //Distance from the FOV midrange to the baseline. Should be negative, mm.
    k64f opticalCentre;     //Distance from the FOV midrange to the optical center (center ray convergence) Negative is closer to sensor.
    k64f expansionCoeff;    //Expansion coefficient (CTE) of the spine material.
      
} kS3dStereoProfilerBaseClass;

kDeclareClassEx(kVs, kS3dStereoProfilerBase, kObject)

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

// virtual //
kVsFx(kStatus) xkS3dStereoProfilerBase_VInitClone(kS3dStereoProfilerBase profiler, kS3dStereoProfilerBase source, kAlloc allocator);
kVsFx(kStatus) xkS3dStereoProfilerBase_VRelease(kS3dStereoProfilerBase profiler);
kVsFx(kSize) xkS3dStereoProfilerBase_VSize(kS3dStereoProfilerBase profiler);

// helpers //
kVsFx(kStatus) xkS3dStereoProfilerBase_InitFields(kS3dStereoProfilerBase profiler, const kS3dStereoProfilerParams* params, kS3dStereoCal cal, kAlloc allocator);

kVsFx(kStatus) xkS3dStereoProfilerBase_FindLutBounds(kS3dStereoProfilerBase profiler, kArray2 map, k32s step, k32s* begin, k32s* count);
kVsFx(kStatus) xkS3dStereoProfilerBase_UpdateLut(kS3dStereoProfilerBase profiler, kArray2 map, kArray1 x, kArray1 y, kArray1 z, kSSize vBegin, kSSize vStep, kArray3 lut);

kVsFx(kStatus) xkS3dStereoProfilerBase_UpdateViewOrientation(kS3dStereoProfilerBase profiler, kArray2 disparityMap, kArray1 disparityToZ);

kVsFx(kStatus) xkS3dStereoProfilerBase_LutLimits(kS3dStereoProfilerBase profiler, kArray3 lut, kPoint3d16s* xyzMin, kPoint3d16s* xyzMax);
kVsFx(kStatus) xkS3dStereoProfilerBase_UpdateLimits(kS3dStereoProfilerBase profiler);

//////////////////////////////////////////////////////////////////////////
//
// These can be potentially separated into a LUT class
// together with kS3dStereoProfilerLutParams3d
//
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(k16s) xkS3dStereoProfilerBase_Interpolate(k16s v0, k16s v1, k32s fract, k32u shift, k32u round)
{
    return (k16s)(v0 +(((v1 - v0) * fract + round) >> shift));
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) xkS3dStereoProfilerBase_InterpolateLinear(
    const kPoint3d16s* p0, const kPoint3d16s* p1, 
    k32s fract, k32u shift, k32u round, kPoint3d16s* out)
{
    if (p0->x != k16S_NULL && p1->x != k16S_NULL)
    {
        out->x = xkS3dStereoProfilerBase_Interpolate(p0->x, p1->x, fract, shift, round);
        out->y = xkS3dStereoProfilerBase_Interpolate(p0->y, p1->y, fract, shift, round);
        out->z = xkS3dStereoProfilerBase_Interpolate(p0->z, p1->z, fract, shift, round);
    }
    else
    {
        out->x = out->y = out->z = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
// can't use kCheck() inside cuda functions 
// since its calling some host only functions
// need to change FS, a.t.m. its not necessary as none of those functions can fail
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) xkS3dStereoProfilerBase_InterpolateBilinear(
    const kS3dStereoProfilerLutParams3d* params,
    const kPoint3d16s* p00, const kPoint3d16s* p01, 
    const kPoint3d16s* p10, const kPoint3d16s* p11, 
    k32s x, k32s y, kPoint3d16s* out)
{
    kPoint3d16s bilinTemp[2];

    xkS3dStereoProfilerBase_InterpolateLinear(p00, p01, x, params->xShift, params->xRound, &bilinTemp[0]);
    xkS3dStereoProfilerBase_InterpolateLinear(p10, p11, x, params->xShift, params->xRound, &bilinTemp[1]);

    if (bilinTemp[0].x != k16S_NULL && bilinTemp[1].x != k16S_NULL)
    {
        xkS3dStereoProfilerBase_InterpolateLinear(&bilinTemp[0], &bilinTemp[1], y, params->yShift, params->yRound, out);
    }
    else
    {
        out->x = out->y = out->z = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) xkS3dStereoProfilerBase_InterpolateTrilinear(
    const kS3dStereoProfilerLutParams3d* params,
    const kPoint3d16s* p000, const kPoint3d16s* p001, const kPoint3d16s* p010, const kPoint3d16s* p011,
    const kPoint3d16s* p100, const kPoint3d16s* p101, const kPoint3d16s* p110, const kPoint3d16s* p111,
    k32s x, k32s y, k32s d, kPoint3d16s* out)
{
    kPoint3d16s trilinTemp[2];

    xkS3dStereoProfilerBase_InterpolateBilinear(params, p000, p001, p010, p011, x, y, &trilinTemp[0]);
    xkS3dStereoProfilerBase_InterpolateBilinear(params, p100, p101, p110, p111, x, y, &trilinTemp[1]);

    if (trilinTemp[0].x != k16S_NULL && trilinTemp[1].x != k16S_NULL)
    {
        xkS3dStereoProfilerBase_InterpolateLinear(&trilinTemp[0], &trilinTemp[1], d, params->dShift, params->dRound, out);
    }
    else
    {
        out->x = out->y = out->z = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
// LUT Phase
//////////////////////////////////////////////////////////////////////////

//kCudaInlineFx(kStatus) xkS3dStereoProfilerBase_LookupPhase(
//    const kS3dStereoProfilerLutParams3d* params, const kPoint3d16s* disparityLut, 
//    k16s xProj, k16s yProj, k32s phase, 
//    kPoint3d16s* result)
//{
//    if(xProj != k16S_NULL && yProj != k16S_NULL)
//    {
//        const k32s dStep = params->height * params->width;
//        const k32s rStep = params->width;
//        const k32s rcStep = rStep + 1;
//        
//        const k32s x = xProj - params->xBegin;
//        const k32s y = yProj - params->yBegin;
//        
//        const k32s pIndex = (phase - params->dBegin) >> params->dShift;
//        const k32s pFract = (phase - params->dBegin) & params->dMask;
//        
//        const k32s xIndex = x >> params->xShift;
//        const k32s yIndex = y >> params->yShift;
//        
//        const k32s xFract = x & params->xMask;
//        const k32s yFract = y & params->yMask;       
//
//        if(xIndex >= 0 && xIndex < (params->width - 1) && yIndex >= 0 && yIndex < (params->height - 1) && pIndex >= 0 && (pIndex < (params->depth - 1)))
//        {
//            //basePtD0 = kArray3_At(params->phaseLut[VIEWINDEX], pIndex, yIndex, xIndex);
//            const kSSize index = (pIndex * params->height + yIndex) * params->width + xIndex;
//
//            const kPoint3d16s* basePtD0 = disparityLut + index; // access kArray3(pIndex, yIndex, xIndex)
//            const kPoint3d16s* basePtD1 = basePtD0 + dStep;
//
//            xkS3dStereoProfilerBase_InterpolateTrilinear(params, 
//                basePtD0, basePtD0 + 1, basePtD0 + rStep, basePtD0 + rcStep,
//                basePtD1, basePtD1 + 1, basePtD1 + rStep, basePtD1 + rcStep,
//                xFract, yFract, pFract, result);
//        }
//        else
//        {
//            result->x = result->y = result->z = k16S_NULL;
//        }
//    }
//    else
//    {
//        result->x = result->y = result->z = k16S_NULL;
//    }
//
//    return kOK;
//}

//////////////////////////////////////////////////////////////////////////
//
// Rectification / Projection
//
// Templetize xkS3dStereoProfilerBase_InterpolateLinear() on point (2D or 3D) and nuke the following
// this is a 2D interpolation vs 3D above (z is skipped)
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) xkS3dStereoProfilerBase_InterpolateLinear2d(
    const kPoint16s* p0, const kPoint16s* p1,
    k32s fract, k32u shift, k32u round, 
    kPoint16s* out)
{
    if (p0->x != k16S_NULL && p1->x != k16S_NULL)
    {
        out->x = xkS3dStereoProfilerBase_Interpolate(p0->x, p1->x, fract, shift, round);
        out->y = xkS3dStereoProfilerBase_Interpolate(p0->y, p1->y, fract, shift, round);
    }
    else
    {
        out->x = out->y = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) xkS3dStereoProfilerBase_InterpolateBilinear2d(
    const kS3dStereoProfilerLutParams2d* params,
    const kPoint16s* p00, const kPoint16s* p01,
    const kPoint16s* p10, const kPoint16s* p11,
    k32s x, k32s y, 
    kPoint16s* out)
{
    kPoint16s bilinTemp[2];

    xkS3dStereoProfilerBase_InterpolateLinear2d(p00, p01, x, params->projXShift, params->projXRound, &bilinTemp[0]);
    xkS3dStereoProfilerBase_InterpolateLinear2d(p10, p11, x, params->projXShift, params->projXRound, &bilinTemp[1]);

    if (bilinTemp[0].x != k16S_NULL && bilinTemp[1].x != k16S_NULL)
    {
        xkS3dStereoProfilerBase_InterpolateLinear2d(&bilinTemp[0], &bilinTemp[1], y, params->projYShift, params->projYRound, out);
    }
    else
    {
        out->x = out->y = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
//
//
// Hardware context 
// Host or Device functions
//
//
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// Bilinear
///@brief Rectifies fractional image coordinates. Both coordinates must be scaled to kSPOT_CENTRE_SCALE
//////////////////////////////////////////////////////////////////////////


kCudaInlineFx(void) kS3dStereoProfilerHwContext_Rectify(kS3dStereoProfilerHwContext hwContext, kSize viewIndex, k32s x, k32s y, k16s* xProj, k16s* yProj)
{
    kS3dStereoProfilerHwContextClass* obj = (kS3dStereoProfilerHwContextClass*)hwContext;

    const kS3dStereoProfilerLutParams2d* params = obj->rectParams;
    const kPoint16s* rectLut = obj->rectLut[viewIndex];

    const k32s rStep = params->projWidth;
    const k32s rcStep = rStep + 1;    

    k32s xv = (x << params->spotXShift) + (params->spotOffset[viewIndex].x << kSPOT_CENTRE_SHIFT);
    k32s yv = (y << params->spotYShift) + (params->spotOffset[viewIndex].y << kSPOT_CENTRE_SHIFT);

    if (xv < params->projXMax && yv < params->projYMax)
    {
        const k32s xIndex = xv >> params->projXShift;
        const k32s yIndex = yv >> params->projYShift;
        const k32s xFract = xv & params->projXMask;
        const k32s yFract = yv & params->projYMask;

        //kPoint16s* basePt = (kPoint16s*)kArray2_At(obj->rectLut[viewIndex], yIndex, xIndex);
        const kPoint16s* basePt = rectLut + yIndex * params->projWidth + xIndex;

        kPoint16s out;
        xkS3dStereoProfilerBase_InterpolateBilinear2d(params, basePt, basePt + 1, basePt + rStep, basePt + rcStep, xFract, yFract, &out);

        *xProj = out.x;
        *yProj = out.y;
    }
    else
    {
        *xProj = *yProj = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
// Linear
///@brief Rectifies fractional image coordinates. Both coordinates must be scaled to kSPOT_CENTRE_SCALE, however, Y values must fall on integer rows. 
///No interpolation across rows is performed
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) kS3dStereoProfilerHwContext_RectifyLinear(kS3dStereoProfilerHwContext hwContext, kSize viewIndex, k32s x, k32s y, k16s* xProj, k16s* yProj)
{
    kS3dStereoProfilerHwContextClass* obj = (kS3dStereoProfilerHwContextClass*)hwContext;

    const kS3dStereoProfilerLutParams2d* params = obj->rectParams;
    const kPoint16s* rectLut = obj->rectLut[viewIndex];
    
    const k32s xv = (x << params->spotXShift) + (params->spotOffset[viewIndex].x << kSPOT_CENTRE_SHIFT);
    const k32s yv = (y << params->spotYShift) + (params->spotOffset[viewIndex].y << kSPOT_CENTRE_SHIFT);

    if (xv >= 0 && xv < params->projXMax && yv >= 0 && yv < params->projYMax)
    {
        const k32s xIndex = xv >> params->projXShift;
        const k32s yIndex = yv >> params->projYShift;
        const k32s xFract = xv & params->projXMask;

        // kPoint16s* basePt = (kPoint16s*)kArray2_At(obj->rectLut[viewIndex], yIndex, xIndex);
        // kPoint16s* basePt = rectLut[yIndex][xIndex] 
        const kPoint16s* basePt = rectLut + yIndex * params->projWidth + xIndex;

        kPoint16s out;
        xkS3dStereoProfilerBase_InterpolateLinear2d(basePt, basePt + 1, xFract, params->projXShift, params->projXRound, &out);

        *xProj = out.x;
        *yProj = out.y;
    }
    else
    {
        *xProj = *yProj = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) kS3dStereoProfilerHwContext_LookupDisparity(kS3dStereoProfilerHwContext hwContext, k16s xProj0, k16s xProj1, k16s yProj, kPoint3d16s* result)
{
    kS3dStereoProfilerHwContextClass* obj = (kS3dStereoProfilerHwContextClass*)hwContext;

    //xkS3dStereoProfilerBase_LookupDisparity(obj->disparityParams, obj->disparityLut, xProj0, xProj1, yProj, result);

    const kS3dStereoProfilerLutParams3d* params = obj->disparityParams;
    const kPoint3d16s* disparityLut = obj->disparityLut;

    const k32s dStep = params->height * params->width;
    const k32s rStep = params->width;
    const k32s rcStep = rStep + 1;

    if (xProj0 != k16S_NULL && xProj1 != k16S_NULL && yProj != k16S_NULL)
    {
        const k32s x = xProj0 - params->xBegin;
        const k32s y = yProj - params->yBegin;
        const k32s d = (xProj1 - (k32s)xProj0) - params->dBegin;

        const k32s xIndex = x >> params->xShift;
        const k32s yIndex = y >> params->yShift;
        const k32s dIndex = d >> params->dShift;

        const k32s xFract = x & params->xMask;
        const k32s yFract = y & params->yMask;
        const k32s dFract = d & params->dMask;

        if (xIndex >= 0 && xIndex < (params->width - 1) && yIndex >= 0 && yIndex < (params->height - 1) && dIndex >= 0 && (dIndex < (params->depth - 1)))
        {
            //basePtD0 = (kPoint3d16s*)kArray3_At(params->disparityLut, dIndex, yIndex, xIndex);
            const kSSize index = (dIndex * params->height + yIndex) * params->width + xIndex;

            const kPoint3d16s* basePtD0 = disparityLut + index; // access kArray3(dIndex, yIndex, xIndex)
            const kPoint3d16s* basePtD1 = basePtD0 + dStep;

            xkS3dStereoProfilerBase_InterpolateTrilinear(params,
                //kxS3D_STEREO_PROFILER_INTERP_TRILIN(params,
                basePtD0, basePtD0 + 1, basePtD0 + rStep, basePtD0 + rcStep,
                basePtD1, basePtD1 + 1, basePtD1 + rStep, basePtD1 + rcStep,
                xFract, yFract, dFract, result);
        }
        else
        {
            result->x = result->y = result->z = k16S_NULL;
        }
    }
    else
    {
        result->x = result->y = result->z = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kCudaInlineFx(void) kS3dStereoProfilerHwContext_LookupPhase(kS3dStereoProfilerHwContext hwContext, kSize viewIndex, k16s xProj, k16s yProj, k32s phase, kPoint3d16s* result)
{
    kS3dStereoProfilerHwContextClass* obj = (kS3dStereoProfilerHwContextClass*)hwContext;

    const kS3dStereoProfilerLutParams3d* params = obj->phaseParams;
    const kPoint3d16s* disparityLut = obj->phaseLut[viewIndex];

    if (xProj != k16S_NULL && yProj != k16S_NULL)
    {
        const k32s dStep = params->height * params->width;
        const k32s rStep = params->width;
        const k32s rcStep = rStep + 1;

        const k32s x = xProj - params->xBegin;
        const k32s y = yProj - params->yBegin;

        const k32s pIndex = (phase - params->dBegin) >> params->dShift;
        const k32s pFract = (phase - params->dBegin) & params->dMask;

        const k32s xIndex = x >> params->xShift;
        const k32s yIndex = y >> params->yShift;

        const k32s xFract = x & params->xMask;
        const k32s yFract = y & params->yMask;

        if (xIndex >= 0 && xIndex < (params->width - 1) && yIndex >= 0 && yIndex < (params->height - 1) && pIndex >= 0 && (pIndex < (params->depth - 1)))
        {
            //basePtD0 = kArray3_At(params->phaseLut[VIEWINDEX], pIndex, yIndex, xIndex);
            const kSSize index = (pIndex * params->height + yIndex) * params->width + xIndex;

            const kPoint3d16s* basePtD0 = disparityLut + index; // access kArray3(pIndex, yIndex, xIndex)
            const kPoint3d16s* basePtD1 = basePtD0 + dStep;

            xkS3dStereoProfilerBase_InterpolateTrilinear(params,
                basePtD0, basePtD0 + 1, basePtD0 + rStep, basePtD0 + rcStep,
                basePtD1, basePtD1 + 1, basePtD1 + rStep, basePtD1 + rcStep,
                xFract, yFract, pFract, result);
        }
        else
        {
            result->x = result->y = result->z = k16S_NULL;
        }
    }
    else
    {
        result->x = result->y = result->z = k16S_NULL;
    }
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////


#endif // K_VISION_S3D_STEREO_PROFILER_BASE_X_H
