/** 
 * @file    kS3dRegistratorLocalAlg.h
 * @brief   Declares the kS3dBallArrayAlg class. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kVS_REGISTRATOR_LOCAL_ALG_H
#define kVS_REGISTRATOR_LOCAL_ALG_H

#include <kVision/Common/kVision.h>
#include <kVision/L3d/kL3dTransform3d.h>

#include <kApi/Data/kArray2.h>

/**
 * @class       kS3dRegistratorLocalAlg
 * @extends     kObject
 * @ingroup     kVision-S3d
 * @brief       Local Height map to height map alignment
 *
 */

typedef kObject kS3dRegistratorLocalAlg;

/** 
 * Constructs a kS3dRegistratorLocalAlg object.
 *
 * @public              @memberof kS3dRegistratorLocalAlg
 * @param   alg         Destination for the constructed object handle.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status. 
 */

kVsFx(kStatus) kS3dRegistratorLocalAlg_Construct(kS3dRegistratorLocalAlg* alg, kAlloc allocator);

/**
 * Assume we want to align two height maps (sourceMap to targetMap) 
 * Alignment is a two step process:
 *    1) Call kS3dRegistratorLocalAlg_SetTarget(targetMap) 
 *        // where targetMap is the height map to align to
 *    2) Call kS3dRegistratorLocalAlg_Align(sourceMapm, outTransformation) 
 *        // where sourceMap is the height map to align and outTransformation is the calculated alignment transformation
 *        // kS3dRegistratorLocalAlg_Align() can be called multiple times to align to the same targetMap
 * 
 * kS3dRegistratorLocalAlg_SetTarget() sets the targetMap as a new reference mesh to align to
 * 
 * @public                @memberof kS3dRegistratorLocalAlg
 * @param   alg           kS3dRegistratorLocalAlg object handle
 * @param   target        HeightMap of the traget mesh to align to (kArray2 of k16s values)
 * @param   targetOffset  Offset of the target height map in 3D space (kPoint3d64f)
 * @param   targetScale   Scale factors in XYZ of the target height map (kPoint3d64f)
 * @return                Operation status.
 */

kVsFx(kStatus) kS3dRegistratorLocalAlg_SetTarget(
    kS3dRegistratorLocalAlg alg, 
    kArray2 target, const kPoint3d64f* targetOffset, const kPoint3d64f* targetScale);

/**
*
* kS3dRegistratorLocalAlg_Align() generated a transformation required to align sourceMap to targetMat
*
* @public                @memberof kS3dRegistratorLocalAlg
* @param   alg           kS3dRegistratorLocalAlg object handle
* @param   source        HeightMap of the source mesh to align (kArray2 of k16s values)
* @param   sourceOffset  Offset of the source height map in 3D space (kPoint3d64f)
* @param   sourceScale   Scale factors in XYZ of the source height map (kPoint3d64f)
* @param   inTransform   Initial transformation of the sourceMap to bring it closer for initial alignment (Can be identity)
* @param   outTransform  Output transformation of the alignment alg
* @return                Operation status.
*/

kVsFx(kStatus) kS3dRegistratorLocalAlg_Align(
    kS3dRegistratorLocalAlg alg, 
    kArray2 source, const kPoint3d64f* sourceOffset, const kPoint3d64f* sourceScale,
    const kL3dTransform3d* inTransform, kL3dTransform3d* outTransform);

/*********************************** 
 * Params (optional)
 ***********************************/

/**
 * Set or get the number of trace steps to use for finding the corresponding matching point. (defaults to 30)
 * Determines how far the alg has to go to find a corresponding closest point to pull to on the other mesh.
 * Decreasing this number will improve performance but will decrease how far the initial position 
 * of meshes from each other can be.
 *
 *
 * @public                   @memberof kS3dRegistratorLocalAlg
 * @param   alg              kS3dRegistratorLocalAlg object handle
 * @param   steps            Number of voxels(steps) to use to find a corresponding point to pull to.
 * @return                   Operation status.
 */

kStatus kS3dRegistratorLocalAlg_TraceStepCount(kS3dRegistratorLocalAlg alg, kSize* steps);
kStatus kS3dRegistratorLocalAlg_SetTraceStepCount(kS3dRegistratorLocalAlg alg, kSize steps);

/**
 * Set or get camera direction, defaults to Point(0,0,1)
 * Should not really be changed.
 *
 * @public                   @memberof kS3dRegistratorLocalAlg
 * @param   alg              kS3dRegistratorLocalAlg object handle
 * @param   dir              Camera direction.
 * @return                   Operation status.
*/

kStatus kS3dRegistratorLocalAlg_TraceDirection(kS3dRegistratorLocalAlg alg, kPoint3d64f* dir);
kStatus kS3dRegistratorLocalAlg_SetTraceDirection(kS3dRegistratorLocalAlg alg, const kPoint3d64f* dir);

/**
 * Set or get number of iterations to run the alg to align the meshes. (defaults to 300)
 * Algorithm will use max this number.
 * It will terminate before reaching this number if the accuracy tolerance is achived.
 *
 * @public                   @memberof kS3dRegistratorLocalAlg
 * @param   alg              kS3dRegistratorLocalAlg object handle
 * @param   count            Number of alg iterations.
 * @return                   Operation status.
 */

kStatus kS3dRegistratorLocalAlg_IterationCount(kS3dRegistratorLocalAlg alg, kSize* count);
kStatus kS3dRegistratorLocalAlg_SetIterationCount(kS3dRegistratorLocalAlg alg, kSize count);

/**
 * Set or get number of sample point to use. (defaults to 300)
 * It will randomly subsample the meshes each iteration and only use this number.
 * Decreasing this number may improve performance but can lead to bad alignment.
 * Increasing this number may imrouve alignment quality.
 * 
 * @public                   @memberof kS3dRegistratorLocalAlg
 * @param   alg              kS3dRegistratorLocalAlg object handle
 * @param   count            Number of alg samples.
 * @return                   Operation status.
 */

kStatus kS3dRegistratorLocalAlg_SampleCount(kS3dRegistratorLocalAlg alg, kSize* count);
kStatus kS3dRegistratorLocalAlg_SetSamplesCount(kS3dRegistratorLocalAlg alg, kSize count);

/**
 * Set or get number of voxels per maximal height map side (per dimention XYZ). (defaults to 100)
 * The algorithm uses internal space partitioning grid for fast point access.
 * This number defines the grid resolution.
 * Increasing it should generally improuve performance but will allso increase memory consumtion.
 * Also after a cirtain threshhold (one point per voxel) the performance will level out or even drop.
 *
 * @public                   @memberof kS3dRegistratorLocalAlg
 * @param   alg              kS3dRegistratorLocalAlg object handle
 * @param   count            Number of alg voxels per longest height map side.
 * @return                   Operation status.
 */
  
kStatus kS3dRegistratorLocalAlg_MaxAxisVoxelCapacity(kS3dRegistratorLocalAlg alg, kSize* count);
kStatus kS3dRegistratorLocalAlg_SetMaxAxisVoxelCapacity(kS3dRegistratorLocalAlg alg, kSize count);

/**
 * Set or get main termination criteria for the algorithm. (defaults to NULL, autogenerated based on grid point density)
 * Determines what the average mesh to mesh distance should be for the alg to terminate, tol = 0.01f (10 microns) 
 *
 * @public                   @memberof kS3dRegistratorLocalAlg
 * @param   alg              kS3dRegistratorLocalAlg object handle
 * @param   tol              Average mesh to mesh distance.
 * @return                   Operation status.
 */

kStatus kS3dRegistratorLocalAlg_ToleranceEvaluationDistance(kS3dRegistratorLocalAlg alg, k64f* tol);
kStatus kS3dRegistratorLocalAlg_SetToleranceEvaluationDistance(kS3dRegistratorLocalAlg alg, k64f tol);

/**
* Set or get the treshhold for the minimal mesh overlap to consider the alignment legal.
* Determines what the mesh overlap should be, tol = 0.1f (10% mesh overlap) to satisfy alignment
*
* @public                   @memberof kS3dRegistratorLocalAlg
* @param   alg              kS3dRegistratorLocalAlg object handle
* @param   tol              Overlap in %.
* @return                   Operation status.
*/

kStatus kS3dRegistratorLocalAlg_ToleranceEvaluationIntersection(kS3dRegistratorLocalAlg alg, k64f* tol);
kStatus kS3dRegistratorLocalAlg_SetToleranceEvaluationIntersection(kS3dRegistratorLocalAlg alg, k64f tol);


//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dRegistratorLocalAlg.x.h>

#endif  /* kVS_REGISTRATOR_LOCAL_ALG_H */
