/**
* @file    kS3dPointsSorter.h
* @brief   Declares the kS3dPointsSorter class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_S3D_POINTSSORTER_H
#define KVISION_S3D_POINTSSORTER_H

#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dPolynomial.h>
#include <kVision/S3d/kS3dPolynomialMap.h>
#include <kVision/L3d/kL3dLattice2d.h>
#include <stdio.h>

/**
* @class       kS3dPointsSorter
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Class sorting the corners to their grid locations by using polynomial reconstruction.
*/
typedef kObject kS3dPointsSorter;

/**
* Constructs a kS3dPointsSorter object
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPointsSorter_Construct(kS3dPointsSorter* sorter, kAlloc allocator);

/**
*  Start the calculation to sort the corners to their grid locations.
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @param   lattice     Output kL3dLattice2d object. The sorted corner points are stored in the kArray2 object. 
* @param   corner      The input corner points array. kArrayList <kPoint64f> 
* @param   center      Marker position. Object handle of kPoint64f. 
* @param   imageSizeX  Width of camera image.
* @param   imageSizeY  Height of camera image.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPointsSorter_Run(kS3dPointsSorter sorter, kL3dLattice2d  lattice, kArrayList corner, const kPoint64f* center, k32s imageSizeX, k32s imageSizeY);

/**
*  Set order of the polynomial.
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @param   order       Order of the polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPointsSorter_SetOrder(kS3dPointsSorter sorter, k32s order);

/**
*  Get order of the polynomial.
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @return              Order of the polynomial.
*/
kVsFx(k32s) kS3dPointsSorter_Order(kS3dPointsSorter sorter);

/**
*  Set tolerance for coarse eliminating outliers
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @param   tol         Tolerance for coarse eliminating outliers in lattice spacing unit. Default value is 0.2 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPointsSorter_SetStartTolerance(kS3dPointsSorter sorter, k64f tol);

/**
*  Get tolerance for coarse eliminating outliers
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @return              Tolerance for coarse eliminating outliers in lattice spacing unit.
*/
kVsFx(k64f) kS3dPointsSorter_StartTolerance(kS3dPointsSorter sorter);

/**
*  Set final tolerance for eliminating outliers
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @param   tol         Tolerance for eliminating outliers in lattice spacing unit. Default value is 0.05
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPointsSorter_SetTolerance(kS3dPointsSorter sorter, k64f tol);

/**
*  Get final tolerance for eliminating outliers.
*
* @public              @memberof kS3dPointsSorter
* @param   sorter      kS3dPointsSorter object.
* @return              Tolerance for eliminating outliers in lattice spacing unit.
*/
kVsFx(k64f) kS3dPointsSorter_Tolerance(kS3dPointsSorter sorter);

#include <kVision/S3d/kS3dPointsSorter.x.h>

#endif /* #ifndef KVISION_S3D_POINTSSORTER_H */
