/**
* @file    kS3dPhaseViewDecompressor.h
* @brief   Declares the kS3dPhaseViewDecompressor class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_VIEW_DECOMPRESSOR_H
#define kS3D_PHASE_VIEW_DECOMPRESSOR_H

#include <kVision/S3d/kS3dDecompressor.h>

/**
* @class       kS3dPhaseViewDecompressor
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       A simple kS3dDecompressor wrapper for decompressing a set of kCompressedPhase subframes.
*/
typedef kObject kS3dPhaseViewDecompressor;

/**
* Constructs a kS3dPhaseViewDecompressor object
*
* @public                    @memberof kS3dPhaseViewDecompressor
* @param   decompressor      Destination for the constructed object handle.
* @param   allocator         Memory allocator (or kNULL for default).
* @return                    Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewDecompressor_Construct(kS3dPhaseViewDecompressor* decompressor, kAlloc alloc);

/**
* Initiates subframe decompression sequence and initializes the output buffer.
* Should be called prior to calling kS3dPhaseViewDecompressor_Update().
* The output buffer handle is retained by the decompressor object and is used to populate the output with decompressed phase and intensity maps.
*
* Upon completion of the sequence (kS3dPhaseViewDecompressor_IsComplete() returns kTRUE).
*
* @public                     @memberof kS3dPhaseViewDecompressor
* @param   decompressor       Decompressor object.
* @param   intensityEnabled   Flag specifying if intensity should be decompressed.
* @param   outputPhasePixel2  Output PhasePixel2 buffer (intensity and phase).
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewDecompressor_Begin(kS3dPhaseViewDecompressor decompressor, kBool intensityEnabled, kArray2 outputPhasePixel2);

/**
* Returns kTRUE when all subframes have been processed and the outputs supplied to kS3dPhaseViewDecompressor_Begin() have been filled.
*
* @public                     @memberof kS3dPhaseViewDecompressor
* @param   decompressor       Decompressor object.
* @return                     Operation status.
*/
kVsFx(kBool) kS3dPhaseViewDecompressor_IsComplete(kS3dPhaseViewDecompressor decompressor);

/**
* Update the decompressor with a new subframe.
* A total number of subframes should be the same as specified by the kS3dPhaseViewDecompressor_SetCompressionSubframeCount().
*
* Call kS3dPhaseViewDecompressor_Begin() to supply output buffers prior to calling this function.
* The function kS3dPhaseViewDecompressor_IsComplete() will return kTRUE once all of the expected subframes have been processed.
*
* @public                     @memberof kS3dPhaseViewDecompressor
* @param   decompressor       Decompressor object.
* @param   subframe           Compressed subframe containing phase and optionally intensity (kCompressedPhase).
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewDecompressor_Update(kS3dPhaseViewDecompressor decompressor, kCompressedPhase subframe);

/**
* Sets the number of subframes in the compression. (Number of parts)
*
* @public                     @memberof kS3dPhaseViewDecompressor
* @param   decompressor       Decompressor object.
* @param   subframeCount      Number of subframes
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewDecompressor_SetCompressionSubframeCount(kS3dPhaseViewDecompressor decompressor, kSize subframeCount);

/**
* Gets the number of subframes in the compression. (Number of parts)
*
* @public                     @memberof kS3dPhaseViewDecompressor
* @param   decompressor       Decompressor object.
* @return                     Number of subframes
*/
kVsFx(kSize) kS3dPhaseViewDecompressor_CompressionSubframeCount(kS3dPhaseViewDecompressor decompressor);

#include <kVision/S3d/kS3dPhaseViewDecompressor.x.h>

#endif  /* kS3D_PHASE_VIEW_DECOMPRESSOR_H */
