/**
* @file    kS3dPhaseCombiner.h
* @brief   Declares the kS3dPhaseCombiner class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_COMBINER_H
#define kS3D_PHASE_COMBINER_H

#include <kFireSync/Data/kSpot.h>
#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/Vs/kVsJobQueue.h>

/**
* @class       kS3dPhaseCombiner
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Combines results of multiple exposure stripe/phase decoding into a single output
*/
typedef kObject kS3dPhaseCombiner;

/**
* Constructs a kS3dPhaseCombiner object
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_Construct(kS3dPhaseCombiner* combiner, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dPhaseCombiner_Begin, kS3dPhaseCombiner_Update).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dPhaseCombiner_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner     Combiner object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_Setup(kS3dPhaseCombiner combiner);

/**
* Initiates combining sequence and initializes the output buffer. Should be called prior to calling 
* kS3dPhaseCombiner_Update. The output buffer handle is retained by the combiner
* object and is used to populate the output map. Upon completion of the sequence (kS3dPhaseCombiner_IsComplete returns kTRUE)
* each entry in the output buffer is populated with the phase, stripe, intensity values from the sequence exhibiting highest contrast.
* Invalid pixels are identified by having the phase value set to k16S_NULL and code value set to k8U_NULL
* 
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner    Combiner object
* @param   output      Output buffer. Expected kArray2<kPhasePixel2> equal in size to the image data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_Begin(kS3dPhaseCombiner combiner, kArray2 output);

/**
* Returns true when all sequences have been processed
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner     Combiner object
* @return              Operation status.
*/
kVsFx(kBool) kS3dPhaseCombiner_IsComplete(kS3dPhaseCombiner combiner);

/**
* Returns true when all sequences have been processed
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner    Combiner object
* @return              Number of phase maps received 
*/
kVsFx(kSize) kS3dPhaseCombiner_AcquiredCount(kS3dPhaseCombiner combiner);

/**
* Update result of a single image sequence. The order in which maps (different exposure sequences) are processed does not affect the result of the combining algorithm
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner    Combiner object
* @param   stripeMap   Result of stripe decoding for the sequence (kArray2<k8u>)
* @param   phaseMap    Result of phase decoding for the sequence (kArray2<kPhasePixel>)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_Update(kS3dPhaseCombiner combiner, kArray2 stripeMap, kArray2 phaseMap);

/**
* Update result of a single image sequence. The order in which maps (different exposure sequences) are processed does not affect the result of the combining algorithm
*
* @public               @memberof kS3dPhaseCombiner
* @param   combiner     Combiner object
* @param   phaseMap     Result of phase decoding for the sequence (kArray2<kPhasePixel2>)
* @return               Operation status.
*/

kVsFx(kStatus) kS3dPhaseCombiner_UpdatePixel2(kS3dPhaseCombiner combiner, kArray2 phaseMap);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner      Combiner object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_SetImageSize(kS3dPhaseCombiner combiner, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner      Combiner object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_ImageSize(kS3dPhaseCombiner combiner, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the total number of decoding sequences
*
* @public                @memberof kS3dPhaseCombiner
* @param   combiner       Combiner object
* @param   sequenceCount Number of expected decoding sequences
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_SetSequenceCount(kS3dPhaseCombiner combiner, kSize sequenceCount);

/**
* Gets the total number of decoding sequences
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner     Combiner object
* @return              Number of expected decoding sequences
*/
kVsFx(kSize) kS3dPhaseCombiner_SequenceCount(kS3dPhaseCombiner combiner);

/**
* Sets the index of the sequence to be used for intensity output
*
* @public                @memberof kS3dPhaseCombiner
* @param   combiner       Combiner object
* @param   index         Index of the sequence to be used for intensity output
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_SetIntensitySequenceIndex(kS3dPhaseCombiner combiner, kSize index);

/**
* Gets the index of the sequence to be used for intensity output
*
* @public              @memberof kS3dPhaseCombiner
* @param   combiner     Combiner object
* @return              Index of the sequence to be used for intensity output
*/
kVsFx(kSize) kS3dPhaseCombiner_IntensitySequenceIndex(kS3dPhaseCombiner combiner);

/**
* Sets the scale of the phase period. The phase values expected by the kS3dPhaseViewProcessor_Run function must be scaled such that
* 0 to 2Pi phase offset is mapped to 0 to period
*
* @public              @memberof kS3dPhaseCombiner
* @param   processor   Processor object
* @param   period      Phase period scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCombiner_SetPhasePeriod(kS3dPhaseCombiner processor, kSize period);

/**
* Gets the scale of the phase period.
*
* @public              @memberof kS3dPhaseCombiner
* @param   processor   Processor object
* @return              Phase period scale
*/
kVsFx(kSize) kS3dPhaseCombiner_PhasePeriod(kS3dPhaseCombiner processor);


#include <kVision/S3d/kS3dPhaseCombiner.x.h>

#endif  /* kS3D_PHASE_COMBINER_H */
