/**
* @file    kS3dHuffmanCode.h
* @brief   Declares the kS3dHuffmanCode Class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_M3D_HUFFMANN_H
#define KVISION_M3D_HUFFMANN_H

#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

/**
* @class       kS3dHuffmanCode
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Routines for encoding and decoding of data. They are used in BCD and OMC File.
*/
typedef kObject kS3dHuffmanCode;

typedef struct kM3dHuffmanItemHeader
{
    k8u ident;
    k32s  allocSize;
    k32f scale;
    k32s  count;
    k16s level;
}kM3dHuffmanItemHeader;

/**
* Constructs a kS3dHuffmanCode object
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_Construct(kS3dHuffmanCode* huffmanCode, kAlloc allocator);

/**
* Get the ID number of the current data set.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @return               ID number of the current data set.
*/
kVsFx(k8u) kS3dHuffmanCode_Ident(kS3dHuffmanCode huffmanCode);

/**
* Set the ID number for the current data set.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @param   ident        ID number of the current data set
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_SetIdent(kS3dHuffmanCode huffmanCode, k8u ident);

/**
* Get item count of the current data set.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @return               item count of the current data set.
*/
kVsFx(k32s) kS3dHuffmanCode_Count(kS3dHuffmanCode huffmanCode);

/**
* Set item count of the current data set.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @param   count        item count of the current data set.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_SetCount(kS3dHuffmanCode huffmanCode, k32s count);

/**
* Get digital resolution of the current data set.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @return               digital resolution of the current data set.
*/
kVsFx(k32f) kS3dHuffmanCode_Scale(kS3dHuffmanCode huffmanCode);


/**
* Set digital resolution of the current data set.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @param   scale        digital resolution of the current data set.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_SetScale(kS3dHuffmanCode huffmanCode, k32f scale);

/**
* Updates working buffers of the height map with current configuration.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_Setup(kS3dHuffmanCode huffmanCode);

/**
* Get decoded output array of k32s.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object
* @return               decoded output array of k32s.
*/
kVsFx(kArray1) kS3dHuffmanCode_Data(kS3dHuffmanCode huffmanCode);

/**
* Read a data block from a file.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object.
* @param   file         kFile object.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_Read(kS3dHuffmanCode huffmanCode, kFile file);

/**
* Write a data block into a file.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object.
* @param   file         kFile object.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_Write(kS3dHuffmanCode huffmanCode, kFile file);

/**
* Write a texture image into a file.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object.
* @param   file         kFile object.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_WriteImage(kS3dHuffmanCode huffmanCode, kImage image, kFile file);

/**
* Read a texture image from a file.
*
* @public               @memberof kS3dHuffmanCode
* @param   huffmanCode  kS3dHuffmanCode object.
* @param   file         kFile object.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dHuffmanCode_ReadImage(kS3dHuffmanCode huffmanCode, kImage image, kFile file);

#include <kVision/S3d/kS3dHuffmanCode.x.h>

#endif /* #ifndef KVISION_M3D_HUFFMANN_H */
