#ifndef LMITECH_KVISION_L3D_SENSORCAL_H_INCLUDED
#define LMITECH_KVISION_L3D_SENSORCAL_H_INCLUDED

#include <kVision/L3d/kL3dCommon.h>
#include <kVision/L3d/kL3dCalData.h>
#include <kVision/L3d/kL3dRangeCalData.h>
#include <kApi/Data/kArray1.h>

/**
* @class   kL3dSensorCal
* @extends kObject
* @ingroup kVision-kL3d
* @brief
*/

typedef kObject kL3dSensorCal;

#define kL3D_SENSOR_CAL_FILE_NAME_FORMAT    "Cal%u.kdat"
#define kL3D_SENSOR_QC_CAL_FILE_NAME_FORMAT "QcCal%u.kdat"

typedef struct kL3dSensorCalParams 
{
    kL3dCalDataType type;

    k32u lineCount;
    k32u sliceCount;

} kL3dSensorCalParams;

/**
* Constructs a kL3dSensorCal object.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   Receives the constructed object.
* @param   params      Struct containing construction parameters.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_Construct(kL3dSensorCal* sensorCal, const kL3dSensorCalParams* params, kAlloc allocator);

/**
* Constructs a kL3dSensorCal object from a file.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   Receives the constructed object.
* @param   node        The node to read the file from. If kNULL, loads from local disk.
* @param   fileName    The full path to load from.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_Load(kL3dSensorCal* sensorCal, kNode node, const kChar* fileName, kAlloc allocator);

/**
* Saves a kL3dSensorCal object to file.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   node        The node to save the file to. If kNULL, saves to local disk.
* @param   fileName    The full path to save to.
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_Store(kL3dSensorCal sensorCal, kNode node, const kChar* filename);

/**
* Saves the calibration object to a stream.
*
* @public               @memberof kL3dSensorCal
* @param   sensorCal    The class object.
* @param   stream       Output stream handle
* @return               Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_SaveStream(kL3dSensorCal sensorCal, kStream stream);

/**
* Loads the calibration object from a stream.
*
* @public               @memberof kL3dSensorCal
* @param   sensorCal    Pointer receiving the read calibration object
* @param   stream       Input stream handle
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_LoadStream(kL3dSensorCal *sensorCal, kStream stream, kAlloc alloc);


/**
* Gets the kL3dSensorCalParams parameters of the kL3dSensorCal object.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @return              kL3dSensorCalParams parameters.
*/
kVsFx(const kL3dSensorCalParams*) kL3dSensorCal_Parameters(kL3dSensorCal sensorCal);

/**
* Sets the Id (node id) of the kL3dSensorCal object. This should match the id of the node that the calibration is for.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   id          A Node id.
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_SetId(kL3dSensorCal sensorCal, k32u id);

/**
* Gets the Id (node id) of the kL3dSensorCal object. This should match the id of the node that the calibration is for.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @return              A Node id.
*/
kVsFx(k32u) kL3dSensorCal_Id(kL3dSensorCal sensorCal);

/**
* Sets the timestamp of the kL3dSensorCal object.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   timestamp   The timestamp, as a string.
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_SetTimestamp(kL3dSensorCal sensorCal, const kChar* timestamp);

/**
* Gets the timestamp of the kL3dSensorCal object.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @return              The timestamp, as a string.
*/
kVsFx(const kChar*) kL3dSensorCal_Timestamp(kL3dSensorCal sensorCal);

/**
* Sets the camera window used during calibration. Programmed by Factory calibration generation and used by runtime logic when
* generating runtime LUTs.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   window      Camera window.
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSensorCal_SetWindow(kL3dSensorCal sensorCal, const kL3dCameraWindow* window);

/**
* Gets the camera window used during calibration. Programmed by Factory calibration generation and used by runtime logic when
* generating runtime LUTs.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @return              Camera window.
*/
kVsFx(const kL3dCameraWindow*) kL3dSensorCal_Window(kL3dSensorCal sensorCal);

/**
* Constructs and adds a new PolynomialCalData entry to the kL3dSensorCal object. 
*
* Each line has at most 1 entry. Constructing a new entry at the same line will dispose the existing entry and add the new one.
* 
* MaxOrder is the maximum order of the polynomials used for the specific range coordinates. 
* Used to determine the buffer size for the calibration entries.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   lineIndex   The line index for this entry.
* @param   maxXOrder   The maximum order of the X polynomial.
* @param   maxYOrder   The maximum order of the Y polynomial.
* @param   maxZOrder   The maximum order of the Z polynomial.
* @return              Camera window.
*/
kVsFx(kStatus) kL3dSensorCal_ConstructPolynomialEntry(kL3dSensorCal sensorCal, k32u lineIndex, k32u maxXOrder, k32u maxYOrder, k32u maxZOrder);

/**
* Constructs and adds a new RangeCalData entry to the kL3dSensorCal object.
*
* Each line has at most 1 entry. Constructing a new entry at the same line will dispose the existing entry and add the new one.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   lineIndex   The line index for this entry.
* @param   params      The parameters to use for constructing the kL3dSensorCal object.
* @return              Camera window.
*/
kVsFx(kStatus) kL3dSensorCal_ConstructRangeEntry(kL3dSensorCal sensorCal, k32u lineIndex, kL3dRangeCalDataParams* params);

/**
* Checks if an entry exists at a line index.
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   lineIndex   The line index for this entry.
* @return              kTRUE if an entry exists, kFALSE otherwise.
*/
kVsFx(kBool) kL3dSensorCal_EntryExists(kL3dSensorCal sensorCal, k32u lineIndex);

/**
* Centroid range over which the polynomial or LUT calibration data is valid for a given line. Specified as kArray1<64f>
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   lineIndex   The line index.
* @return              kArray1<64f>
*/
kVsFx(kArray1) kL3dSensorCal_CentreBegin(kL3dSensorCal sensorCal, k32u lineIndex);

/**
* Centroid range over which the polynomial or LUT calibration data is valid for a given line. Specified as kArray1<64f>
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   lineIndex   The line index.
* @return              kArray1<64f>
*/
kVsFx(kArray1) kL3dSensorCal_CentreEnd(kL3dSensorCal sensorCal, k32u lineIndex);

/**
* Calibration entry for a given line (polynomial or LUT types, although LUT is not implemented)
*
* @public              @memberof kL3dSensorCal
* @param   sensorCal   A kL3dSensorCal object.
* @param   lineIndex   The line index.
* @return              kL3dCalData object.
*/
kVsFx(kL3dCalData) kL3dSensorCal_Data(kL3dSensorCal sensorCal, k32u lineIndex);

#include <kVision/L3d/kL3dSensorCal.x.h>

#endif  /* #ifndef LMITECH_KVISION_L3D_SENSORCAL_H_INCLUDED */
