/**
 * @file    kL3dProfileSampler.h
 * @brief   Declares the kVsClassTemplate class.
 *
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kL3D_PROFILE_SAMPLER_H
#define kL3D_PROFILE_SAMPLER_H

#include <kVision/Common/kVision.h>

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
 * @class       kL3dProfileSampler
 * @extends     kObject
 * @ingroup     kVision-L3d
 * @brief       Profile sampling: 2D/3D line sampling class
 *              See Gocator Developer Manual (GDM) [section Profile (G2)]
 *                https://docs.google.com/document/d/1o6mQ7RtBRzr1w20ZXiUhd4kcClHQxnIIh4I_qOEeDnU/edit#bookmark=id.f2fv81gh0xoz
 *              See kL3dLineSamplerBlock for an example use case
 *              See kL3dLineSamplerBlock_VOnStart() for parameter conversion from floats to k16s 
 */

typedef kObject kL3dProfileSampler;

/**
 * Constructs a kL3dProfileSampler object.
 *
 * @public              @memberof kL3dProfileSampler
 * @param   sampler     Destination for the constructed object handle.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status
 */

kVsFx(kStatus) kL3dProfileSampler_Construct(kL3dProfileSampler* sampler, kAlloc allocator);

/**
* Samples an array of 2D/3D points at specified intervals (xStep).
* In and Out dimension variants:
* (In)              => (Out)
* (x,z)             => (z)
* (x,y,z)           => (y,z)
* (x,z,intensity)   => (z,intensity)
* (x,y,z,intensity) => (y,z,intensity)
*
* @public                   @memberof kL3dProfileSampler
* @param   sampler          Sampler object (this)
* @param   inPointData      An array of 2D or 3D points (x,z or x,y,z) to resample
* @param   inIntensityData  An array of intensities corresponding to each point (can be NULL)
* @param   inPointCount     A number of input points
* @param   inDimensions     x,z or x,y,z [2 or 3]
* @param   outPointData     An array of sampled points as z or y,z
* @param   outIntensityData An array of sampled intensities (can be NULL, if inIntensityData is NULL and outIntensityData is not it will be filled with 0s)
* @param   outPointCount    A number of output points (should be equal to XCount)

* @return                  Operation status.
*/

kVsFx(kStatus) kL3dProfileSampler_Resample(kL3dProfileSampler sampler,
    const k16s* inPointData, const k8u* inIntensityData, kSize inPointCount, kSize inDimensions,
    k16s* outPointData, k8u* outIntensityData, kSize outPointCount);

/**
* Update and check resample parameters
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_Setup(kL3dProfileSampler sampler);

/**
* Set sampling starting point.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param  val          Sampling starting point
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetXStart(kL3dProfileSampler sampler, k16s val);

/**
* Get sampling starting point.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              XStart
*/

kInlineFx(k16s) kL3dProfileSampler_XStart(kL3dProfileSampler sampler);

/**
* Set sampling step (sampling intervals).
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param  val          Sampling step
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetXStep(kL3dProfileSampler sampler, k16s val);

/**
* Get sampling step (sampling intervals).
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              XStep
*/

kInlineFx(k16s) kL3dProfileSampler_XStep(kL3dProfileSampler sampler);

/**
* Set total number of samples.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         Number of samples to perform
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetXCount(kL3dProfileSampler sampler, kSize val);

/**
* Get total number of samples.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              XCount
*/

kInlineFx(kSize) kL3dProfileSampler_XCount(kL3dProfileSampler sampler);

/**
* Set point step threshold.
* If two consecutive points are further then threshold a hole in sampling is created.
* XStepThreshold is only checked if FillAllAdjacent is set to kFALSE
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         Threshold value
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetXStepThreshold(kL3dProfileSampler sampler, k16s val);

/**
* Get point step threshold.
* If two consecutive points are further then threshold a hole in sampling is created.
* XStepThreshold is only checked if FillAllAdjacent is set to kFALSE
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              XStepThreshold
*/

kInlineFx(k16s) kL3dProfileSampler_XStepThreshold(kL3dProfileSampler sampler);

/**
* Get/Set point slope threshold.
* If two consecutive points are rising sharper than specified ZSlopeThreshold (angle) a hole in sampling is created.
* ZSlopeThreshold is only checked if FillAllAdjacent is set to kTRUE
* Use a helper function (kL3dProfileSampler_CalculateZSlopeThreshold) for calculating ZSlopeThreshold based on angle and xResolution, zResolution.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         ZSlopeThreshold value (use kL3dProfileSampler_CalculateZSlopeThreshold() to calculate)
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetZSlopeThreshold(kL3dProfileSampler sampler, k16s val);

/**
* Set point slope threshold.
* If two consecutive points are rising sharper than specified ZSlopeThreshold (angle) a hole in sampling is created.
* ZSlopeThreshold is only checked if FillAllAdjacent is set to kTRUE
* Use a helper function (kL3dProfileSampler_CalculateZSlopeThreshold) for calculating ZSlopeThreshold based on angle and xResolution, zResolution.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              ZSlopeThreshold
*/

kInlineFx(k16s) kL3dProfileSampler_ZSlopeThreshold(kL3dProfileSampler sampler);

/**
* A helper function for calculating ZSlopeThreshold based on angle and xResolution, zResolution.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   angle       Angle in degrees [0, 90]
* @param   xResolution X resolution (X conversion factor from float to k16s)
* @param   zResolution Z resolution (Z conversion factor from float to k16s)
* @return              Calculated ZSlopeThreshold
*/

kInlineFx(k16s) kL3dProfileSampler_CalculateZSlopeThreshold(k64f angle, k64f xResolution, k64f zResolution);

/**
* Set valid range for point depth (Z).
* Anything outside those bounds will be ignored.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         ZMin/ZMax value
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetZMin(kL3dProfileSampler sampler, k16s val);

/**
* Get valid range for point depth (Z).
* Anything outside those bounds will be ignored.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              ZMin
*/

kInlineFx(k16s) kL3dProfileSampler_ZMin(kL3dProfileSampler sampler);

/**
* Set valid range for point depth (Z).
* Anything outside those bounds will be ignored.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         ZMax value
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetZMax(kL3dProfileSampler sampler, k16s val);

/**
* Get valid range for point depth (Z).
* Anything outside those bounds will be ignored.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              ZMax
*/

kInlineFx(k16s) kL3dProfileSampler_ZMax(kL3dProfileSampler sampler);

/**
* Set flag for generating samples between points which otherwise would be treated a gap (hole).
* If set to kTRUE XStepThreshold is ignored (samples are still generated even inside the gap)
*   Sample generation will rely on ZSlopeThreshold checks for gap generation.
* If set to kFALSE ZSlopeThreshold is ignored
*   Sample generation will rely on XStepThreshold checks
* 
* 
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         FillAllAdjacent flag value
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetFillAllAdjacent(kL3dProfileSampler sampler, kBool val);

/**
* Get flag for generating samples between points which otherwise would be treated a gap (hole).
* If set to kTRUE XStepThreshold is ignored (samples are still generated even inside the gap)
*   Sample generation will rely on ZSlopeThreshold checks for gap generation.
* If set to kFALSE ZSlopeThreshold is ignored
*   Sample generation will rely on XStepThreshold checks
*
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              FillAllAdjacent flag
*/

kInlineFx(kBool) kL3dProfileSampler_FillAllAdjacent(kL3dProfileSampler sampler);

/**
* Set flag for identifying sensor orientation (bottom/up).
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   val         BottomSensor flag value
* @return              Operation status
*/

kInlineFx(kStatus) kL3dProfileSampler_SetBottomSensor(kL3dProfileSampler sampler, kBool val);

/**
* Get flag for identifying sensor orientation (bottom/up).
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @return              BottomSensor flag.
*/

kInlineFx(kBool) kL3dProfileSampler_BottomSensor(kL3dProfileSampler sampler);

/**
* Get value X corresponding to each point (entry) of outPointData array of the kL3dProfileSampler_Resample() function.
*
* @public              @memberof kL3dProfileSampler
* @param   sampler     Sampler object (this)
* @param   index       Index of the point (element) in question
* @return              X value of the point
*/

kInlineFx(k32s) kL3dProfileSampler_XValue(kL3dProfileSampler sampler, kSize index);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/L3d/kL3dProfileSampler.x.h>

#endif  /* kL3D_PROFILE_SAMPLER_H */
