/**
* @file    kG3dContourSegment.h
* @brief   Declares the kG3dContourSegment class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kS3D_CONTOURSEGMENT_H
#define KVISION_kS3D_CONTOURSEGMENT_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

#include <kVision/G3d/kG3dContourHistogram.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/G3d/kG3dContourTemplate.h>

/**
* @class       kG3dContourSegment
* @extends     kObject
* @ingroup     kVision-G3d
* @brief       This class is used for the segmentation of contour candidate points into groups.
*              The segmentation is based on the gradient magnitude and direction of each point on 
*              a heightmap or a intenstiy map.
*/
typedef kObject kG3dContourSegment;

/**
* Constructs a kG3dContourSegment object
*
* @public              @memberof kG3dContourSegment
* @param   segment     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_Construct(kG3dContourSegment* segment, kAlloc allocator);

/**
* Performs the evaluation of segmentation without template
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   value          Data buffer for input gradient magnitude map.   kArray2 < k16s >
* @param   angle          Data buffer for input gradient direction map.   kArray2 < k16s >
* @param   mode           Type of the original map to be evaluated.  The definition see kG3dInputImagesOrder
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_Apply(kG3dContourSegment segment, kArray2 value, kArray2 angle, k32s mode);

/**
* Performs the evaluation of segmentation using template
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   value          Data buffer for input gradient magnitude map.   kArray2 < k16s >
* @param   angle          Data buffer for input gradient direction map.   kArray2 < k16s >
* @param   maskMap        Data buffer for input rough segmentation according to the template  kArray2 < k16s >
* @param   mode           Type of the original map to be evaluated.  The definition see kG3dInputImagesOrder
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_RunTemplate(kG3dContourSegment segment, kArray2 value, kArray2 angle, kArray2 maskMap, k32s mode);

/**
* Get the number of point groups which are segmented as contour candidate.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @return                 Number of point groups.
*/
kVsFx(kSize) kG3dContourSegment_ContourCount(kG3dContourSegment segment);

/**
* Set sample index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   index          Sample index of a measurement series
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_SetSampleIndex(kG3dContourSegment segment, kSize index);

/**
* Set repeat index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   index          Repeat index of a measurement series
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_SetRepeatIndex(kG3dContourSegment segment, kSize index);

/**
* Creates a data buffer and stores the detected candidate points for border contours into the buffer.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   borderPoints   Handle of the data buffer
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_BorderContour(kG3dContourSegment segment, kArray1* borderPoints);

/**
* Creates a data buffer and stores the detected candidate points for closed contours into the buffer.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   contourPoints  Handle of the data buffer
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_ClosedContour(kG3dContourSegment segment, kArray1* contourPoints);

/**
* Set a boolean value to determine, whether the candidate points for border contours should be detected.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   detect         The candidate points for border contours should be detected?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_SetClosedContour(kG3dContourSegment segment, kBool detect);

/**
* Set a boolean value to determine, whether the candidate points for closed contours should be detected.
*
* @public                 @memberof kG3dContourSegment
* @param   segment        kG3dContourSegment object.
* @param   detect         The candidate points for closed contours should be detected?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourSegment_SetDetectBorder(kG3dContourSegment segment, kBool detect);

#include <kVision/G3d/kG3dContourSegment.x.h>

#endif  /* KVISION_kS3D_CONTOURSEGMENT_H */
