/**
* @file    kG3dAffineMergeHeightMap.h
* @brief   Declares the kG3dAffineMergeHeightMap class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_AFFINEMERGEHEIGHTMAP_H
#define KVISION_kG3D_AFFINEMERGEHEIGHTMAP_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/G3d/kG3dContourTemplate.h>

/**
* @class       kG3dAffineMergeHeightMap
* @extends     kObject
* @ingroup     kVision-G3d
* @brief       Class for merging two height maps by using affine 2D transformation and iterative 
*              best fit method to minimize the difference between the two height maps.
*              Note: This class can only be used when the affine transformation matrix is known.
*              This means that the 3 degrees of freedom are already determined. 
*              The remaining 3 degrees of freedom is determined in the class.
*/
typedef kObject kG3dAffineMergeHeightMap;

/**
* Constructs a kG3dAffineMergeHeightMap object
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_Construct(kG3dAffineMergeHeightMap* merge, kAlloc allocator);

//temporary
/**
* Set a data buffer to store the difference map between the both to be merged height maps.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   map         Data buffer to store the difference map. 
*                      If the buffer is not provided. the buffer is created by the class itself.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_SetDiffeerenceMap(kG3dAffineMergeHeightMap merge, kArray2 map);

/**
* Set an affine transformation matrix.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   trans       Affine transformation matrix
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_SetTransform(kG3dAffineMergeHeightMap merge, kL3dTransform3d* trans);

/**
* Set two scale sets for the both to be merged height maps.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   scale       scaling cooficient for the main height map
* @param   scale2      scaling cooficient for the buddy height map
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_SetDifferenceScale(kG3dAffineMergeHeightMap merge, kPoint3d64f scale, kPoint3d64f scale2);

/**
* Performs the merge operation from the two height maps.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   heightMap   Data buffer of the main height map
* @param   buddyMap    Data buffer of the buddy height map
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_Run(kG3dAffineMergeHeightMap merge, kArray2 heightMap, kArray2 buddyMap);

/**
* Performs the transformation of an intensity map (kArray2 < k8u > ) after the merge operation.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   outputMap   Data buffer for the transformed output intensity map
* @param   inputMap    Data buffer of the input intensity map
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_RunIntensity(kG3dAffineMergeHeightMap merge, kArray2 outputMap, kArray2 inputMap);

/**
* Performs the transformation of an bitmap (kImage) after the merge operation.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   outputMap   Data buffer for the transformed output texture map
* @param   inputBitmap Data buffer of the input bitmap
* @param   flip        Whether the map should be flipped in the y direction
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_RunBitmap(kG3dAffineMergeHeightMap merge, kArray2 outputMap, kImage inputBitmap, kBool flip);

/**
* Performs the transformation of a height map after the merge operation.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   outputMap   Data buffer for the transformed output heightmap
* @param   inputmap    Data buffer of the input heightmap
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_RunHeightmap(kG3dAffineMergeHeightMap merge, kArray2 outputMap, kArray2 inputMap);

/**
* Performs the transformation of a mask map after the merge operation.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   outputMap   Data buffer for the transformed output mask map
* @param   inputmap    Data buffer of the input mask map
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_RunMaskMap(kG3dAffineMergeHeightMap merge, kArray2 outputMap, kArray2 inputMap);

/**
* Set the file name for storing the difference map btween the main and buddy height maps
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   fileName    Filename of the difference map btween the main and buddy height maps
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_SetDifferenceFileName(kG3dAffineMergeHeightMap merge, kChar* fileName);

/**
* Set repeat index of a measurement series.This function is only used for debug purpose to store 
* the intermediate results without overwriting.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   index       Repeat index of a measurement series
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_SetRepeatIndex(kG3dAffineMergeHeightMap merge, kSize index);

/**
* Set sample index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public              @memberof kG3dAffineMergeHeightMap
* @param   merge       Object of kG3dAffineMergeHeightMap.
* @param   index       Sample index of a measurement series
* @return              Operation status.
*/
kVsFx(kStatus) kG3dAffineMergeHeightMap_SetSampleIndex(kG3dAffineMergeHeightMap merge, kSize index);

#include <kVision/G3d/kG3dAffineMergeHeightMap.x.h>

#endif  /* KVISION_kG3D_AFFINEMERGEHEIGHTMAP_H */
