/** 
 * @file    kNodeDef.h
 * @brief   Essential declarations related to FireSync nodes.
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NODE_DEF_H
#define K_FIRESYNC_NODE_DEF_H

#include <kFireSync/kFsDef.h>
#include <kApi/kApiDef.h>
#include <kApi/Utils/kDateTime.h>
#include <kFireSync/Utils/kInfo.h>

/**
 * @struct  kSystemChange
 * @extends kValue
 * @ingroup kFireSync-Client
 * @brief   Represents the type of a change within a sensor network.
 */
typedef k32s kSystemChange; 

/** @relates kSystemChange @{ */
#define kSYSTEM_CHANGE_NULL             (0x00)      ///< No changes.
#define kSYSTEM_CHANGE_TOPOLOGY         (0x01)      ///< Potential network topology change has occurred; refresh system. 
#define kSYSTEM_CHANGE_CONFIGURATION    (0x02)      ///< Local configuration is out of sync with remote configuration; refresh node(s).
/** @} */

/** @relates kSystem @{ */
#define kSYSTEM_DATA_NET_BUFFER_SIZE_LARGE      (1024*1024)     ///< Recommended net buffer size for systems with few sensors (e.g., 1-2). 
#define kSYSTEM_DATA_NET_BUFFER_SIZE_MEDIUM     (64*1024)       ///< Recommended net buffer size for systems with multiple sensors (e.g., 3-10). 
#define kSYSTEM_DATA_NET_BUFFER_SIZE_SMALL      (16*1024)       ///< Recommended net buffer size for systems with many sensors (e.g., 10+). 
#define kSYSTEM_DATA_NET_BUFFER_SIZE_DEFAULT    (kSYSTEM_DATA_NET_BUFFER_SIZE_MEDIUM)     ///< Default net buffer size.
/** @} */


/**
 * @struct  kSyncDomain
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents FireSync synchronization domain options.
 */
typedef k32s kSyncDomain; 

/** @relates kSyncDomain @{ */
#define kSYNC_DOMAIN_NULL           (0x00)      ///< Unknown synchronization domain.
#define kSYNC_DOMAIN_TIME           (0x01)      ///< FireSync time domain. 
#define kSYNC_DOMAIN_ENCODER        (0x02)      ///< FireSync encoder domain.
#define kSYNC_DOMAIN_OPTIONS        (0x03)      ///< Bitset containing all kSyncDomain options.
/** @} */

/** 
 * Formats a kSyncDomain value to a string buffer. 
 *
 * @public              @memberof kSyncDomain
 * @param   domain      kSyncDomain value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSyncDomain_Format(kSyncDomain domain, kChar* text, kSize capacity); 

/**
 * @struct  kEventControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents event manager control modes.
 */
typedef k32s kEventControl; 

/** @relates kEventControl @{ */
#define kEVENT_CONTROL_NULL                             (0x00)          ///< Unknown event control mode.
#define kEVENT_CONTROL_CONTINUOUS                       (0x01)          ///< Continuous, periodic event sequence. 
#define kEVENT_CONTROL_SCHEDULED_BY_SOFTWARE            (0x02)          ///< Software command schedules future event sequences. 
#define kEVENT_CONTROL_TRIGGERED_BY_SOFTWARE            (0x04)          ///< Software command triggers immediate event sequences. 
#define kEVENT_CONTROL_TRIGGERED_BY_NODE_INPUT          (0x10)          ///< Local node digital input triggers immediate event sequences. 
#define kEVENT_CONTROL_TRIGGERED_BY_MASTER_INPUT        (0x20)          ///< Master digital input triggers immediate event sequences. 
#define kEVENT_CONTROL_TRIGGERED_BY_CAMERA              (0x40)          ///< Camera exposure triggers immediate event sequences. 
#define kEVENT_CONTROL_OPTIONS                          (0x77)          ///< Bitset containing all valid kEventControl options.
/** @} */

/** 
 * Formats a kEventControl value to a string buffer. 
 *
 * @public              @memberof kEventControl
 * @param   control     kEventControl value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventControl_Format(kEventControl control, kChar* text, kSize capacity); 

/**
 * @struct  kEventEncoder
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents event manager encoder mode options.
 */
typedef k32s kEventEncoder; 

/** @relates kEventEncoder @{ */
#define kEVENT_ENCODER_NULL                 (0x00)          ///< Unknown encoder mode.
#define kEVENT_ENCODER_FORWARD_ONLY         (0x01)          ///< Event manager counts forward ticks only.
#define kEVENT_ENCODER_FORWARD_TRACK        (0x02)          ///< Event manager counts forward and backward ticks, but only generates forward events.
#define kEVENT_ENCODER_FORWARD_BACKWARD     (0x04)          ///< Event manager generates forward and backward events.
#define kEVENT_ENCODER_OPTIONS              (0x07)          ///< Bitset containing all kEventEncoder options.
/** @} */

/** 
 * Formats a kEventEncoder value to a string buffer. 
 *
 * @public              @memberof kEventEncoder
 * @param   mode        kEventEncoder value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventEncoder_Format(kEventEncoder mode, kChar* text, kSize capacity); 

/**
 * @struct  kEventManagerType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents event manager types.
 */
typedef k32s kEventManagerType; 

/** @relates kEventManagerType @{ */
#define kEVENT_MANAGER_TYPE_NULL                      (0x00)          ///< Unknown event manager type.
#define kEVENT_MANAGER_TYPE_FULL                      (0x01)          ///< Full event manager type. 
#define kEVENT_MANAGER_TYPE_SIMPLE                    (0x02)          ///< Simple event manager type. 
/** @} */

/** 
 * Formats a kEventManagerType value to a string buffer. 
 *
 * @public              @memberof kEventManagerType
 * @param   type        kEventManagerType value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManagerType_Format(kEventManagerType type, kChar* text, kSize capacity);

/**
 * @struct  kEventStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents event manager run-time statistics. 
 */
typedef struct kEventStats
{
    k32u queueCount;                    ///< Current count of event requests in FIFO. 
    k64u overflowCount;                 ///< Count of event request FIFO overflow errors. 
    k64u eventCount;                    ///< Count of individual events generated. 
    k64u sequenceDropCount;             ///< Count of event sequences dropped. 
    k64u eventDropCount;                ///< Count of individual events dropped. 
    k32u queueCapacity;                 ///< Capacity of event FIFO. 
} kEventStats;

/**
 * @struct  kEventMissStyle
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents event manager miss styles.
 */
typedef k32s kEventMissStyle; 

/** @relates kEventMissStyle @{ */
#define kEVENT_MISS_STYLE_NULL                      (0x00)          ///< Unknown event miss style.
#define kEVENT_MISS_STYLE_DROP                      (0x01)          ///< Drop event and notify dependent cameras, outputs, etc.
#define kEVENT_MISS_STYLE_RAISE                     (0x02)          ///< Raise the event normally, as though it was not missed. 
#define kEVENT_MISS_STYLE_OPTIONS                   (0x03)          ///< Bitset containing all kEventMissStyle options.
/** @} */

/** 
 * Formats a kEventMissStyle value to a string buffer. 
 *
 * @public              @memberof kEventMissStyle
 * @param   missStyle   kEventMissType value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventMissStyle_Format(kEventMissStyle missStyle, kChar* text, kSize capacity); 

/**
 * @struct  kCameraModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported camera device types.
 */
typedef k32s kCameraModel; 

/** @relates kCameraModel @{ */
#define kCAMERA_MODEL_NULL          (0x00)          ///< Unknown camera model.
#define kCAMERA_MODEL_LC150         (0x01)          ///< LC150 (DLIS 2K mono). 
#define kCAMERA_MODEL_LC650         (0x02)          ///< LC650 (Vita 1300 mono). 
#define kCAMERA_MODEL_LC651         (0x03)          ///< LC651 (Vita 1300 color). 
#define kCAMERA_MODEL_LC1300        (0x04)          ///< LC1300 (Python 1300 mono). 
#define kCAMERA_MODEL_LC1301        (0x05)          ///< LC1301 (Python 1300 color). 
#define kCAMERA_MODEL_RC940         (0x06)          ///< RC940 (Lupa 300).
#define kCAMERA_MODEL_LC2000        (0x07)          ///< LC2000 (Python 2000 mono).
#define kCAMERA_MODEL_LC2001        (0x08)          ///< LC2001 (Python 2000 colour).
#define kCAMERA_MODEL_LC5000        (0x09)          ///< LC5000 (Python 5000 mono).
#define kCAMERA_MODEL_LC5001        (0x0A)          ///< LC5001 (Python 5000 colour).
#define kCAMERA_MODEL_LC2500        (0x0B)          ///< LC2500 (LMI_IMG2500 mono).
#define kCAMERA_MODEL_LC2501        (0x0C)          ///< LC2501 (LMI_IMG2500 colour).
#define kCAMERA_MODEL_LC9000        (0x0D)          ///< LC9000 (LMI_IMG9000).
/** @} */


/** 
 * Formats a kCameraModel value to a string buffer. 
 *
 * @public              @memberof kCameraModel
 * @param   model       kCameraModel value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraModel_Format(kCameraModel model, kChar* text, kSize capacity); 

/**
 * @struct  kCameraControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera control modes.
 */
typedef k32s kCameraControl; 

/** @relates kCameraControl @{ */
#define kCAMERA_CONTROL_NULL                        (0x00)      ///< Unknown camera control mode.
#define kCAMERA_CONTROL_TRIGGERED_BY_EVENT          (0x01)      ///< Camera exposure sequence triggered by event manager. 
#define kCAMERA_CONTROL_COUPLED_TO_NODE_INPUT       (0x02)      ///< Camera exposure controlled by node digital input. 
#define kCAMERA_CONTROL_COUPLED_TO_MASTER_INPUT     (0x04)      ///< Camera exposure controlled by master digital input. 
#define kCAMERA_CONTROL_COUPLED_TO_CAMERA           (0x08)      ///< Camera exposure controlled by other camera exposure. 
#define kCAMERA_CONTROL_COUPLED_TO_PROJECTOR        (0x10)      ///< Camera exposure controlled by projector exposure.
#define kCAMERA_CONTROL_TRIGGERED_BY_SOFTWARE       (0x20)      ///< Camera exposure sequence triggered by software command.
#define kCAMERA_CONTROL_OPTIONS                     (0x3F)      ///< Bitset containing all valid kCameraControl options.
/** @} */

/**
 * @struct  kCameraExtension
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera control extension numeric type identifiers.
 */
typedef k32s kCameraExtension; 

/** @relates kCameraExtension @{ */
#define kCAMERA_EXTENSION_NULL               (0x00)          ///< Unknown camera sequence type.
#define kCAMERA_EXTENSION_STEPS              (0x01)          ///< Programmed state sequence. 
#define kCAMERA_EXTENSION_DYN_EXP            (0x02)          ///< Dynamic exposure sequence. 
#define kCAMERA_EXTENSION_TRACKING           (0x04)          ///< Window tracking sequence. 
#define kCAMERA_EXTENSION_OVERRIDE           (0x08)          ///< Configuration override. 
#define kCAMERA_EXTENSION_PHASE_DECODER      (0x10)          ///< Stripe and phase acceleration.
#define kCAMERA_EXTENSION_DYN_FPN            (0x20)          ///< Dynamic FPN.
#define kCAMERA_EXTENSION_OPTIONS            (0x3F)          ///< Bitset containing all valid kCameraExtension options.
/** @} */

/** 
 * Formats a kCameraExtension value to a string buffer. 
 *
 * @public              @memberof kCameraExtension
 * @param   extension    kCameraExtension value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraExtension_Format(kCameraExtension extension, kChar* text, kSize capacity);

/** 
 * Gets the type information associated with a kCameraExtension value.
 *
 * @public              @memberof kCameraExtension
 * @param   extension   kCameraExtension value. 
 * @return              Type information.
 */
kFsFx(kType) kCameraExtension_ToType(kCameraExtension extension); 

/** 
 * Gets the extension id associated with an extension type.
 *
 * @public              @memberof kCameraExtension
 * @param   type        Type information.
 * @return              kCameraExtension value. 
 */
kFsFx(kCameraExtension) kCameraExtension_FromType(kType type); 

/**
* @struct  kCameraStepsStyle
* @extends kValue
* @ingroup kFireSync
* @brief   Represents camera steps style.
*/
typedef k32s kCameraStepsStyle;

#define kCAMERA_STEPS_STYLE_NULL                (0x00)      ///< Unknown steps style.
#define kCAMERA_STEPS_STYLE_SINGLE_EXPOSURE     (0x01)      ///< Single exposure steps style.
#define kCAMERA_STEPS_STYLE_MULTIPLE_EXPOSURE   (0x02)      ///< Multiple exposure steps style.
#define kCAMERA_STEPS_STYLE_OPTIONS             (0x03)      ///< Bitset containing all valid kCameraStepsStyle options.

/**
* Formats a kCameraStepsStyle value to a string buffer.
*
* @public              @memberof kCameraStepsStyle
* @param   style       kCameraStepsStyle value.
* @param   text        Receives formatted string.
* @param   capacity    Buffer capacity.
* @return              Operation status.
*/
kFsFx(kStatus) kCameraStepsStyle_Format(kCameraStepsStyle style, kChar* text, kSize capacity);

/**
 * @struct  kCameraOverrideType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera override type.
 */
typedef k32s kCameraOverrideType;

/** @relates kCameraOverrideType @{ */
#define kCAMERA_OVERRIDE_TYPE_NULL              (0x00)      ///< Unknown camera override type.
#define kCAMERA_OVERRIDE_TYPE_STATE             (0x01)      ///< Override current camera state (used to switch between states).
#define kCAMERA_OVERRIDE_TYPE_DYN_EXP           (0x02)      ///< Override dynamic expsosure, FireSync ns. New exposure value is provided; delay is updated automatically.
#define kCAMERA_OVERRIDE_TYPE_EXPOSURE          (0x04)      ///< Override exposure only, FireSync ns.
#define kCAMERA_OVERRIDE_TYPE_DELAY             (0x08)      ///< Override delay only, FireSync ns.
#define kCAMERA_OVERRIDE_TYPE_WINDOW_TOP        (0x10)      ///< Override camera window top location.
#define kCAMERA_OVERRIDE_TYPE_OPTIONS           (0x1F)      ///< Bitset containing all valid kCameraOverrideType options.
/** @} */

/**
 * Formats a kCameraOverrideType value to a string buffer.
 *
 * @public              @memberof kCameraExtension
 * @param   type        kCameraOverrideType value.
 * @param   text        Receives formatted string.
 * @param   capacity    Buffer capacity.
 * @return              Operation status.
 */
kFsFx(kStatus) kCameraOverrideType_Format(kCameraOverrideType type, kChar* text, kSize capacity);

/**
 * @struct  kCameraOverrideItem
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera override item.
 */
typedef struct kCameraOverrideItem
{
    kCameraExtension extension;     ///< Which extension to override.
    kCameraOverrideType type;       ///< Override type.
    k64s value;                     ///< Override value.
} kCameraOverrideItem;

/**
 * @struct  kCameraPhaseDecoderLutType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera phase decoder LUT type.
 */
typedef k32s kCameraPhaseDecoderLutType;

/** @relates kCameraPhaseDecoderLutType @{ */
#define kCAMERA_PHASE_DECODER_LUT_TYPE_NULL     (0x00)      ///< Unknown camera phase decoder LUT type.
#define kCAMERA_PHASE_DECODER_LUT_TYPE_STRIPE   (0x01)      ///< Phase decoder stripe LUT.
#define kCAMERA_PHASE_DECODER_LUT_TYPE_SIN      (0x02)      ///< Phase decoder sin LUT.
#define kCAMERA_PHASE_DECODER_LUT_TYPE_COS      (0x04)      ///< Phase decoder cos LUT.
#define kCAMERA_PHASE_DECODER_LUT_TYPE_EPS      (0x08)      ///< Phase decoder EPS shift matrix.
#define kCAMERA_PHASE_DECODER_LUT_TYPE_OPTIONS  (0x0F)      ///< Bitset containing all valid kCameraPhaseDecoderLutType options.
/** @} */

/**
 * @struct  kCameraPhaseDecoderFormat
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported camera phase decoder formats.
*/
typedef k32s kCameraPhaseDecoderFormat;

/** @relates kCameraPhaseDecoderFormat @{ */
#define kCAMERA_PHASE_DECODER_FORMAT_NULL           (0x00)  ///< Unknown phase decoder format.
#define kCAMERA_PHASE_DECODER_FORMAT_S8_C8I8P16     (0x01)  ///< 8-bit stripe (S8), 8-bit contrast (C8), 8-bit intensity (I8), 16-bit phase (P16).
#define kCAMERA_PHASE_DECODER_FORMAT_S8_P16         (0x02)  ///< 8-bit stripe (S8), 16-bit phase (P16).
#define kCAMERA_PHASE_DECODER_FORMAT_I8P24          (0x04)  ///< 8-bit intensity(I8), 24-bit phase (P24).
#define kCAMERA_PHASE_DECODER_FORMAT_DELTA_I8_P24   (0x08)  ///< Compressed 8-bit intensity(I8), 24-bit phase (P24).
#define kCAMERA_PHASE_DECODER_FORMAT_DELTA_P24      (0x10)  ///< Compressed 24-bit phase (P24).

#define kCAMERA_PHASE_DECODER_FORMAT_OPTIONS        (0x1F)  ///< Bitset containing all valid camera phase decoder formats.
/** @} */

/**
 * Formats a kCameraPhaseDecoderFormat value to a string buffer.
 *
 * @public              @memberof kCameraPhaseDecoderFormat
 * @param   format      kCameraPhaseDecoderFormat value.
 * @param   text        Receives formatted string.
 * @param   capacity    Buffer capacity.
 * @return              Operation status.
 */
kFsFx(kStatus) kCameraPhaseDecoderFormat_Format(kCameraPhaseDecoderFormat format, kChar* text, kSize capacity);

/** 
 * Reports whether the format is considered a 'legacy' format. 
 * 
 * S8_C8I8P16 and  S8_P16 are currently considered legacy formats; they may still be supported by older
 * PL configurations included in the FireSync SDK, but will not be supported in future.
 *
 * @public              @memberof kCameraPhaseDecoderFormat
 * @param   format      kCameraPhaseDecoderFormat value.
 * @return              kTRUE if the format is considered a legacy format. 
 */
kFsFx(kBool) kCameraPhaseDecoderFormat_IsLegacy(kCameraPhaseDecoderFormat format);

/** 
 * Reports whether the format is compressed. 
 * 
 * Compressed formats produce kCompressedPhase messages, which must be decompresssed by clients.
 *
 * @public              @memberof kCameraPhaseDecoderFormat
 * @param   format      kCameraPhaseDecoderFormat value.
 * @return              kTRUE if the format is compressed. 
 */
kFsFx(kBool) kCameraPhaseDecoderFormat_IsCompressed(kCameraPhaseDecoderFormat format);

/** 
 * Reports whether the format includes intensity data.
 * 
 * @public              @memberof kCameraPhaseDecoderFormat
 * @param   format      kCameraPhaseDecoderFormat value.
 * @return              kTRUE if the format includes intensity data. 
 */
kFsFx(kBool) kCameraPhaseDecoderFormat_HasIntensity(kCameraPhaseDecoderFormat format);

/**
 * @struct  kCameraPhaseDecoderType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported camera phase decoder type.
*/
typedef k32s kCameraPhaseDecoderType;

/** @relates kCameraPhaseDecoderType @{ */
#define kCAMERA_PHASE_DECODER_TYPE_NULL             (0x00)  ///< Unknown camera phase decoder type mode.
#define kCAMERA_PHASE_DECODER_TYPE_STRIPE_PHASE     (0x01)  ///< Stripe/Phase mode.
#define kCAMERA_PHASE_DECODER_TYPE_EPS              (0x02)  ///< EPS mode.
/** @} */

/**
 * @struct  kCameraExposure
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera exposure mode.
 */
typedef k32s kCameraExposure; 

/** @relates kCameraExposure @{ */
#define kCAMERA_EXPOSURE_NULL               (0x00)          ///< Unknown camera exposure mode.
#define kCAMERA_EXPOSURE_GLOBAL_SHUTTER     (0x01)          ///< Global shutter mode. 
#define kCAMERA_EXPOSURE_OPTIONS            (0x01)          ///< Bitset containing all valid kCameraExposure options.
/** @} */

/** 
 * Formats a kCameraExposure value to a string buffer. 
 *
 * @public              @memberof kCameraExposure
 * @param   mode        kCameraExposure value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraExposure_Format(kCameraExposure mode, kChar* text, kSize capacity);

/**
 * @struct  kCameraOutputMode
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera output modes.
 */
typedef k32s kCameraOutputMode;

/** @relates kCameraOutputMode @{ */
#define kCAMERA_OUTPUT_MODE_NULL                    (0x00)      ///< Unknown camera output mode.
#define kCAMERA_OUTPUT_MODE_VIDEO                   (0x01)      ///< Video camera output mode.
#define kCAMERA_OUTPUT_MODE_ALGORITHM               (0x02)      ///< Algorithm camera output mode.
#define kCAMERA_OUTPUT_MODE_VIDEO_AND_ALGORITHM     (0x04)      ///< Video and algorithm camera output mode.
#define kCAMERA_OUTPUT_MODE_OPTIONS                 (0x07)      ///< Bitset containing all valid kCameraOutputMode options.
/** @} */

/**
 * @struct  kCameraPortMode
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera port modes.
 * 
 * In Combined Mode, a single camera output port is defined per camera state, and algorithm and video data 
 * are delivered on the same port. In Separate Mode, separate camera state ports are defined for each kind 
 * of output data (e.g., Video, Algorithm). 
 * 
 * Combined Mode is deprecated and will eventually be removed. However, it will continue to be supported 
 * while platform and application code migrate to Separate Mode. New applications and features should be 
 * developed using Separate Mode. 
 */
typedef k32s kCameraPortMode;

/** @relates kCameraPortMode @{ */
#define kCAMERA_PORT_MODE_NULL                      (0x00)      ///< Unknown camera port mode.
#define kCAMERA_PORT_MODE_COMBINED                  (0x01)      ///< Output messages are delivered on a single port per state.
#define kCAMERA_PORT_MODE_SEPARATE                  (0x02)      ///< Output messages are delivered on multiple ports per state, depending on data type.
#define kCAMERA_PORT_MODE_OPTIONS                   (0x03)      ///< Bitset containing all valid kCameraPortMode options.
/** @} */

/**
 * @struct  kCameraReadout
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera readout mode.
 * 
 * Modern cameras have the ability to start a new exposure while the previous frame is 
 * being read out. This is referred to as simultaneous mode and provides higher frame 
 * rates, especially when exposure and readout times are roughly equal. Alternatively, 
 * sequential mode can be selected to prevent exposure and readout times from overlapping.
 *
 * Simultaneous mode may have undesirable side effects, depending on capabilities/limitations 
 * of the camera chip. Side effects may include exposure jitter, parasitic light sensitivity, 
 * and increased noise. Accordingly, if pixel quality is more important than speed, it may 
 * be advantageous to use sequential mode. 
 */
typedef k32s kCameraReadout; 

/** @relates kCameraReadout @{ */
#define kCAMERA_READOUT_NULL               (0x00)          ///< Unknown camera readout mode.
#define kCAMERA_READOUT_SEQUENTIAL         (0x01)          ///< Sequential exposure and readout. 
#define kCAMERA_READOUT_SIMULTANEOUS       (0x02)          ///< Simultaneous exposure and readout. 
#define kCAMERA_READOUT_OPTIONS            (0x03)          ///< Bitset containing all valid kCameraReadout options.
/** @} */

/** 
 * Formats a kCameraReadout value to a string buffer. 
 *
 * @public              @memberof kCameraReadout
 * @param   mode        kCameraReadout value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraReadout_Format(kCameraReadout mode, kChar* text, kSize capacity);

/**
 * @struct  kCameraGain
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera gain channels.
 */
typedef k32s kCameraGain; 

/** @relates kCameraGain @{ */
#define kCAMERA_GAIN_RED                    (0)             ///< Red gain channel.
#define kCAMERA_GAIN_GREEN0                 (1)             ///< Green-0 gain channel.
#define kCAMERA_GAIN_GREEN1                 (2)             ///< Green-1 gain channel.
#define kCAMERA_GAIN_BLUE                   (3)             ///< Blue gain channel.
#define kCAMERA_GAIN_ANALOG                 (4)             ///< Analog gain channel.
#define kCAMERA_GAIN_DIGITAL                (5)             ///< Digital gain channel.
#define kCAMERA_GAIN_CAPACITY               (6)             ///< Count of defined camera gain channels. 
/** @} */

/** 
 * Formats a kCameraGain value to a string buffer. 
 *
 * @public              @memberof kCameraGain
 * @param   gain        kCameraGain value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraGain_Format(kCameraGain gain, kChar* text, kSize capacity);

/**
 * @struct  kCameraSubsampling
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera subsampling modes.
 */
typedef k32s kCameraSubsampling;

/** @relates kCameraSubsampling @{ */
#define kCAMERA_SUBSAMPLING_NULL            (0x00)          ///< Unknown camera subsampling mode.
#define kCAMERA_SUBSAMPLING_SKIP            (0x01)          ///< Subsample by omitting cells. 
#define kCAMERA_SUBSAMPLING_BIN             (0x02)          ///< Subsample by combining cells. 
#define kCAMERA_SUBSAMPLING_OPTIONS         (0x03)          ///< Bitset containing all valid kCameraSubsampling options.
/** @} */

/** 
 * Formats a kCameraSubsampling value to a string buffer. 
 *
 * @public              @memberof kCameraSubsampling
 * @param   mode        kCameraSubsampling value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraSubsampling_Format(kCameraSubsampling mode, kChar* text, kSize capacity);

/**
 * @struct  kCameraAlgorithm
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera hardware-accelerated algorithm types.
 */
typedef k32s kCameraAlgorithm; 

/** @relates kCameraAlgorithm @{ */
#define kCAMERA_ALGORITHM_NULL                      (0)          ///< Unknown camera algorithm type.
#define kCAMERA_ALGORITHM_ACCGD_2K                  (1)          ///< ACCGD_2K algorithm. 
#define kCAMERA_ALGORITHM_ARCGD_2ND                 (2)          ///< ARCGD_2ND algorithm. 
#define kCAMERA_ALGORITHM_BIN_CSUM3                 (3)          ///< BIN_CSUM3 algorithm. 
#define kCAMERA_ALGORITHM_SOBEL_ACCGD               (4)          ///< SOBEL_ACCGD algorithm. 
#define kCAMERA_ALGORITHM_SOBEL_ARCGD               (5)          ///< SOBEL_ARCGD algorithm. 
#define kCAMERA_ALGORITHM_SOBELW_ARCGDW             (6)          ///< SOBELW_ARCGDW algorithm. 
#define kCAMERA_ALGORITHM_SOBEL_ARCGDW              (7)          ///< SOBEL_ARCGDW algorithm. 
#define kCAMERA_ALGORITHM_SOBELV_ACCGDW             (8)          ///< SOBELV_ACCGDW algorithm. 
#define kCAMERA_ALGORITHM_BIN_CSUM4                 (9)          ///< BIN_CSUM4 algorithm. 
/** @} */

/**
 * @struct  kCameraSpotSort
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera algorithm spot sort options.
 */
typedef k32s kCameraSpotSort; 

/** @relates kCameraSpotSort @{ */
#define kCAMERA_SPOT_SORT_NULL                  (0)     ///< Unknown spot sorting type.
#define kCAMERA_SPOT_SORT_SUM                   (1)     ///< Elect spot based on sum (highest sum has highest priority).
#define kCAMERA_SPOT_SORT_CG_LOW                (2)     ///< Elect spot based on CG (lowest CG has highest priority).
#define kCAMERA_SPOT_SORT_CG_HIGH               (3)     ///< Elect spot based on CG (highest CG has highest priority).
/** @} */

/**
 * @struct  kAcgSpotFormat
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents spot format options for PL laser-line spot-detection algorithms. 
 */
typedef k32s kAcgSpotFormat; 

/** @relates kAcgSpotFormat @{ */
#define kACG_SPOT_FORMAT_NULL             (0x0)     ///< Unknown spot format.
#define kACG_SPOT_FORMAT_ENTRY_VALUE      (0x1)     ///< Spot includes optional entry value field.
#define kACG_SPOT_FORMAT_ENTRY_OFFSET     (0x2)     ///< Spot includes optional entry offset field.
#define kACG_SPOT_FORMAT_EXIT_OFFSET      (0x4)     ///< Spot includes optional exit offset field.
#define kACG_SPOT_FORMAT_OPTIONS          (0x7)     ///< Bitset containing all valid kAcgSpotFormat options.

/** @} */

#define kCAMERA_PL_LUT_X_REDUCTION              (4)     ///< Slice Step reduction factor for X 

typedef struct kRangeLutCaps
{
    k32u centroidCount;         ///< Camera natural number of centroids
    k32u sliceCount;            ///< Camera number of slices
    k32u centroidStep;          ///< Nominal Centroid Step for Z and X
    k32u validStep;             ///< Nominal Centroid Step for V
    k32u sliceStep;             ///< Nominal Slice Step for Z
} kRangeLutCaps;

typedef struct kRangeLutParams
{
    k32u intensityMultiplier;   ///< Sum to intensity multiplier
    k32s centroidOffset;        ///< Offset for Z and X as multiple of step
    k32u centroidStep;          ///< Centroid step for Z and X
    k32u validStep;             ///< Centroid step for V
    k32u sliceStep;             ///< Slice step for Z, X and V
} kRangeLutParams;

typedef struct kRangeLutArray
{
    k32u centroidStep;          ///< Centroid Step for this array
    k32u sliceStep;             ///< Slice Step for this array
    k32u length0;               ///< Slice Count for this array
    k32u length1;               ///< Centroid Count for this array
    k32u itIncr;                ///< Data pointer increment
    void* data;                 ///< Data pointer to array content
} kRangeLutArray;

/**
 * @struct  kCameraStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera run-time statistics. 
 */
typedef struct kCameraStats
{
    k64u exposureCount;                 ///< Exposure count.
    k64u exposureDropCount;             ///< Exposure drop count.
    k64u bytesReceived;                 ///< Count of bytes received from PL. 
    k64u bufferDropCount;               ///< Count of frames dropped due to insufficient frame buffer capacity.
    k64u allocDropCount;                ///< Count of frames dropped due to lack of dynamic memory.
    k64u soeInterruptCount;             ///< Deprecated; no longer supported.
    k64u eoeInterruptCount;             ///< Deprecated; no longer supported.
    k64u controlErrorCount;             ///< Count of camera sequencing/control errors. 
    k64u imagerDesyncCount;             ///< Count of camera communication desync errors. 
    k64u syncAlignmentErrorCount;       ///< Count of camera sync channel alignment errors. 
    k64u dataAlignmentErrorCount;       ///< Count of camera data channel alignment errors. 
    k64u channelAlignmentErrorCount[4]; ///< Packed multiple counters of camera data channel alignment errors. 
    k64u channelAlignmentErrorFlags;    ///< Camera data channel alignment error flags. 
    k64u writeDmaStatus;                ///< Status of PL write DMA.
    k64u readDmaStatus;                 ///< Status of PL read DMA.
    k64u inputOverflowCount;            ///< Count of input buffer overflow events. 
    k64u streamDesyncCount;             ///< Count of stream desync errors. 
    k64u processingErrorCount;          ///< Count of stream processing errors. 
    k64u processedFrameCount;           ///< Count of frames that have been received from PL with valid stamps.
    k64u searchFrameCount;              ///< Count of search frames while on tracking mode.
    k64u frameRate;                     ///< Camera frame rate (1/1000 Hz). 
    k64u errorFlags;                    ///< Bitset representing current driver-specific, error information. 
    k64u errorHistory;                  ///< Bitset representing previous driver-specific, error information. 
    k64u errorFlagsAccum;               ///< Bitset representing accumulated driver-specific, error information. 
} kCameraStats;

/**
 * @struct  kCameraInputCounterSource
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents the input counter source.
 */
typedef k32s kCameraInputCounterSource; 

/** @relates kCameraInputCounterSource @{ */
#define kCAMERA_INPUT_COUNTER_SOURCE_NULL           (0x00)  ///< Unknown input counter source.
#define kCAMERA_INPUT_COUNTER_SOURCE_NODE_INPUT     (0x01)  ///< Node input counter source.
#define kCAMERA_INPUT_COUNTER_SOURCE_MASTER_INPUT   (0x02)  ///< Master input counter source.
#define kCAMERA_INPUT_COUNTER_SOURCE_CAMERA         (0x04)  ///< Camera input counter source.
#define kCAMERA_INPUT_COUNTER_SOURCE_PROJECTOR      (0x08)  ///< Projector input counter source.
#define kCAMERA_INPUT_COUNTER_SOURCE_SOFTWARE       (0x10)  ///< Software input counter source.
#define kCAMERA_INPUT_COUNTER_SOURCE_OPTIONS        (0x1F)  ///< Bitset containing all kCameraInputCounterSource options.
/** @} */

/** 
 * Formats a kCameraInputCounterSource value to a string buffer. 
 *
 * @public              @memberof kCameraInputCounterSource
 * @param   source      kCameraInputCounterSource value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraInputCounterSource_Format(kCameraInputCounterSource source, kChar* text, kSize capacity);

/**
 * @struct  kCameraAccelerationTestMode
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents the acceleration test mode.
 */
typedef k32s kCameraAccelerationTestMode; 

/** @relates kCameraAccelerationTestMode @{ */
#define kCAMERA_ACCELERATION_TEST_MODE_NULL                     (0x00)  ///< Unknown Acceleration test mode.
#define kCAMERA_ACCELERATION_TEST_MODE_NONE                     (0x01)  ///< No Acceleration test mode.
#define kCAMERA_ACCELERATION_TEST_MODE_PHASE_COMPRESSION        (0x02)  ///< Compression test mode.
#define kCAMERA_ACCELERATION_TEST_MODE_EPS_FILTER               (0x04)  ///< Filter test mode.
#define kCAMERA_ACCELERATION_TEST_MODE_EPS_SUBTRACT_PHASES      (0x08)  ///< Bypass subtract phases PL block.
#define kCAMERA_ACCELERATION_TEST_MODE_EPS_BASE_RAMP            (0x10)  ///< Bypass base_ramp and unwrap PL block.
#define kCAMERA_ACCELERATION_TEST_MODE_INTERMEDIATE_DATA        (0x20)  ///< Perform normal operation.
#define kCAMERA_ACCELERATION_TEST_MODE_OPTIONS                  (0x3F)  ///< Bitset containing all kCameraAccelerationTestMode options.
/** @} */

/**
 * @struct  kLightModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported light device types.
 */
typedef k32s kLightModel; 

#define kLIGHT_MODEL_NULL           (0x0000)        ///< Unknown light model.
#define kLIGHT_MODEL_GENERIC        (0x0001)        ///< Generic light device. 
#define kLIGHT_MODEL_LD12           (0x0002)        ///< LD12 (unknown variety). 
#define kLIGHT_MODEL_LD12_A         (0x0004)        ///< LD12-A. 
#define kLIGHT_MODEL_LD12_B         (0x0008)        ///< LD12-B.
#define kLIGHT_MODEL_LD12_C         (0x0010)        ///< LD12-C.
#define kLIGHT_MODEL_LD12_D         (0x0020)        ///< LD12-D.
#define kLIGHT_MODEL_LD12_E         (0x0040)        ///< LD12-E.
#define kLIGHT_MODEL_LD12_A2        (0x0080)        ///< LD12-A2.
#define kLIGHT_MODEL_LD12_F         (0x0100)        ///< LD12-F.
#define kLIGHT_MODEL_LD12_F2        (0x0200)        ///< LD12-F2.
#define kLIGHT_MODEL_LD12_G         (0x0400)        ///< LD12-G.
#define kLIGHT_MODEL_LD16           (0x0401)        ///< LD16 (unknown variety).
#define kLIGHT_MODEL_LD16_E         (0x0402)        ///< LD16-E.
#define kLIGHT_MODEL_LD16_E2        (0x0403)        ///< LD16-E2.
#define kLIGHT_MODEL_LD16_F         (0x0404)        ///< LD16-F.
#define kLIGHT_MODEL_LD16_F2        (0x0405)        ///< LD16-F2.
#define kLIGHT_MODEL_LD16_G         (0x0406)        ///< LD16-G.
#define kLIGHT_MODEL_LD16_G2        (0x0407)        ///< LD16-G2.
#define kLIGHT_MODEL_LD16_H         (0x0408)        ///< LD16-H.

/**
 * @struct  kLightDiodeModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported light diode types.
 */
typedef k32s kLightDiodeModel; 

#define kLIGHT_DIODE_MODEL_NULL                             (0)     ///< Unknown light diode model.
#define kLIGHT_DIODE_MODEL_USHIO_HL65051DG                  (1)     ///< Ushio HL65051DG.
#define kLIGHT_DIODE_MODEL_USHIO_HL6501MG                   (2)     ///< Ushio HL6501MG.
#define kLIGHT_DIODE_MODEL_EGISMOS_D6_7_405_40              (3)     ///< Egismos D6-7-405-40.
#define kLIGHT_DIODE_MODEL_WAVE_SPECTRUM_WSLD_405_150M_1    (4)     ///< Wave Spectrum WSLD-405-150m-1.
#define kLIGHT_DIODE_MODEL_DAHENG_GCSLD_405_150M_T018S_PD   (5)     ///< Daheng GCSLD-405-150m-TO18S-pd 150mW

/** 
 * Formats a kLightDiodeModel value to a stable, unique string id. 
 *
 * The strings formatted by this method are guaranteed to be stable and are therefore
 * safe to use in serialized data. 
 *
 * @public              @memberof kLightDiodeModel
 * @param   model       Diode model. 
 * @param   buffer      Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightDiodeModel_Format(kLightDiodeModel model, kChar* buffer, kSize capacity); 

/** 
 * Parses a kLightDiodeModel value from a string buffer. 
 *
 * The method can be used to parse strings that were generated with kLightDiodeModel_Format. 
 * 
 * @public              @memberof kLightDiodeModel
 * @param   model       Receives the diode model. 
 * @param   buffer      Formatted string. 
 * @return              Operation status (kERROR_FORMAT if string unrecognized).
 */
kFsFx(kStatus) kLightDiodeModel_Parse(kLightDiodeModel* model, const kChar* buffer); 

/**
 * @struct  kLightControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents light control modes.
 */
typedef k32s kLightControl; 

/** @relates kLightControl @{ */
#define kLIGHT_CONTROL_NULL                     (0x00)          ///< Unknown light control mode.
#define kLIGHT_CONTROL_TRIGGERED_BY_EVENT       (0x01)          ///< Light exposure sequence triggered by event manager. 
#define kLIGHT_CONTROL_SET_BY_SOFTWARE          (0x02)          ///< Light output state controlled by software commands. 
#define kLIGHT_CONTROL_COUPLED_TO_NODE_INPUT    (0x04)          ///< Light output state controlled by node digital input. 
#define kLIGHT_CONTROL_COUPLED_TO_MASTER_INPUT  (0x08)          ///< Light output state controlled by master digital input.
#define kLIGHT_CONTROL_COUPLED_TO_CAMERA        (0x10)          ///< Light output state controlled by camera exposure.
#define kLIGHT_CONTROL_COUPLED_TO_PROJECTOR     (0x20)          ///< Light output state controlled by projector exposure.
#define kLIGHT_CONTROL_TRIGGERED_BY_SOFTWARE    (0x40)          ///< Light output triggered by software commands.
#define kLIGHT_CONTROL_OPTIONS                  (0x7F)          ///< Bitset containing all valid kLightControl options.
/** @} */

/**
 * @struct  kLightMask
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents light mask sources.
 */
typedef k32s kLightMask; 

/** @relates kLightMask @{ */
#define kLIGHT_MASK_NULL                     (0x00)          ///< Unknown/invalid mask setting. 
#define kLIGHT_MASK_NONE                     (0x01)          ///< No light mask. 
#define kLIGHT_MASK_NODE_INPUT               (0x02)          ///< Light output masked by node digital input signal. 
#define kLIGHT_MASK_MASTER_INPUT             (0x04)          ///< Light output masked by master digital input signal. 
#define kLIGHT_MASK_OPTIONS                  (0x07)          ///< Bitset containing all valid kLightMask options.
/** @} */

/**
 * @struct  kLightExtension
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents light control extension numeric type identifiers.
 */
typedef k32s kLightExtension; 

/** @relates kLightExtension @{ */
#define kLIGHT_EXTENSION_NULL                (0x00)          ///< Unknown light sequence type.
#define kLIGHT_EXTENSION_STEPS               (0x01)          ///< Programmed state sequence. 
#define kLIGHT_EXTENSION_OVERRIDE            (0x02)          ///< Configuration override sequence. 
#define kLIGHT_EXTENSION_OPTIONS             (0x03)          ///< Bitset containing all valid kLightExtension options.
/** @} */

/** 
 * Formats a kLightExtension value to a string buffer. 
 *
 * @public              @memberof kLightExtension
 * @param   type        kLightExtension value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightExtension_Format(kLightExtension type, kChar* text, kSize capacity); 

/** 
 * Gets the type information associated with a kLightExtension value.
 *
 * @public              @memberof kLightExtension
 * @param   extension   kLightExtension value. 
 * @return              Type information.
 */
kFsFx(kType) kLightExtension_ToType(kLightExtension extension); 

/** 
 * Gets the extension id associated with an extension type.
 *
 * @public              @memberof kLightExtension
 * @param   type        Type information.
 * @return              kLightExtension value. 
 */
kFsFx(kLightExtension) kLightExtension_FromType(kType type); 

/**
* @struct  kLightStepsStyle
* @extends kValue
* @ingroup kFireSync
* @brief   Represents light steps style.
*/
typedef k32s kLightStepsStyle;

#define kLIGHT_STEPS_STYLE_NULL                 (0x00)      ///< Unknown steps style.
#define kLIGHT_STEPS_STYLE_SINGLE_EXPOSURE      (0x01)      ///< Single exposure steps style.
#define kLIGHT_STEPS_STYLE_MULTIPLE_EXPOSURE    (0x02)      ///< Multiple exposure steps style.
#define kLIGHT_STEPS_STYLE_OPTIONS              (0x03)      ///< Bitset containing all valid kLightStepsStyle options.

/**
* Formats a kLightStepsStyle value to a string buffer.
*
* @public              @memberof kLightStepsStyle
* @param   style       kLightStepsStyle value.
* @param   text        Receives formatted string.
* @param   capacity    Buffer capacity.
* @return              Operation status.
*/
kFsFx(kStatus) kLightStepsStyle_Format(kLightStepsStyle style, kChar* text, kSize capacity);

/**
* @struct  kLightOverrideType
* @extends kValue
* @ingroup kFireSync
* @brief   Represents light override type.
*/
typedef k32s kLightOverrideType;

/** @relates kLightOverrideType @{ */
#define kLIGHT_OVERRIDE_TYPE_NULL              (0x00)      ///< Unknown light override type.
#define kLIGHT_OVERRIDE_TYPE_STATE             (0x01)      ///< Override current light state (used to switch between states).
#define kLIGHT_OVERRIDE_TYPE_DYN_EXP           (0x02)      ///< Override dynamic expsosure, FireSync ns. New exposure value is provided; delay is updated automatically.
#define kLIGHT_OVERRIDE_TYPE_EXPOSURE          (0x04)      ///< Override exposure only, FireSync ns.
#define kLIGHT_OVERRIDE_TYPE_DELAY             (0x08)      ///< Override delay only, FireSync ns.
#define kLIGHT_OVERRIDE_TYPE_OPTIONS           (0x0F)      ///< Bitset containing all valid kLightOverrideType options.
/** @} */

/**
* Formats a kLightOverrideType value to a string buffer.
*
* @public              @memberof kLightExtension
* @param   type        kLightOverrideType value.
* @param   text        Receives formatted string.
* @param   capacity    Buffer capacity.
* @return              Operation status.
*/
kFsFx(kStatus) kLightOverrideType_Format(kLightOverrideType type, kChar* text, kSize capacity);

/**
* @struct  kLightOverrideItem
* @extends kValue
* @ingroup kFireSync
* @brief   Represents light override item.
*/
typedef struct kLightOverrideItem
{
    kLightOverrideType type;    ///< Override type.
    k64s value;                 ///< Override value.
} kLightOverrideItem;

/**
* @struct  kLightPrimingMode
* @extends kValue
* @ingroup kFireSync
* @brief   Represents light priming mode.
*/
typedef k32s kLightPrimingMode;

/** @relates kLightPrimingMode @{ */
#define kLIGHT_PRIMING_MODE_NULL               (0x00)      ///< Unknown light priming mode.
#define kLIGHT_PRIMING_MODE_NONE               (0x01)      ///< Light priming disabled.
#define kLIGHT_PRIMING_MODE_ONCE               (0x02)      ///< Prime lights once at acquisition startup.
#define kLIGHT_PRIMING_MODE_ALWAYS             (0x04)      ///< Prime lights at beginning of every exposure period.
#define kLIGHT_PRIMING_MODE_OPTIONS            (0x07)      ///< Bitset containing all valid kLightPrimingMode options.
/** @} */

/**
* @struct  kLightCouplingMode
* @extends kValue
* @ingroup kFireSync
* @brief   Represents light coupling mode.
*/
typedef k32s kLightCouplingMode;

/** @relates kLightCouplingMode @{ */
#define kLIGHT_COUPLING_MODE_NULL               (0x00)      ///< Unknown light coupling mode.
#define kLIGHT_COUPLING_MODE_NONE               (0x01)      ///< When paired with a camera, couples light to phyiscal imager exposure signals.
#define kLIGHT_COUPLING_MODE_LEAD_IN            (0x02)      ///< When paired with a camera, coordinates camera/light lead-ins only.
#define kLIGHT_COUPLING_MODE_FULL               (0x04)      ///< When paired with a camera, fully coordinates camera/light leads-in and lead-outs.
#define kLIGHT_COUPLING_MODE_OPTIONS            (0x07)      ///< Bitset containing all valid kLightCouplingMode options.
/** @} */

/**
 * @struct  kTempControlEx
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents configuration parameter for a light temperature control algorithm. 
 */
typedef struct kTempControlEx
{
    kBool enabled;                      ///< Enable PID
    k64f setpoint;                      ///< Temperature setpoint
    k64f dt;                            ///< Time constant (seconds)
    k64f kp;                            ///< Proportional constant
    k64f ki;                            ///< Integral constant
    k64f kd;                            ///< Derivative constant
    k64f ks;                            ///< Offset constant
    k64f min;                           ///< Min
    k64f max;                           ///< Max
    k64f integratorMax;                 ///< Max (absolute) value for integrator
} kTempControlEx;

/** 
 * Initializes a kTempControlEx structure to default values. 
 *
 * @public          @memberof kInfo
 * @param   temp    LightTemp structure. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kTempControlEx_Init(kTempControlEx* temp);

/**
 * @struct  kLightStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents light run-time statistics. 
 */
typedef struct kLightStats
{
    k64u exposureCount;             ///< Exposure count.
    k64u exposureDropCount;         ///< Exposure drop count.
    k64u interruptCount;            ///< Deprecated; no longer supported.
    kBool isLockedOut;              ///< Is light output locked (forced off)?
    kBool outputState;              ///< Current light output state (Set By Software mode only).  
} kLightStats;

/**
 * @struct  kProjectorModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported projector device types.
 */
typedef k32s kProjectorModel; 

/** @relates kProjectorModel @{ */
#define kPROJECTOR_MODEL_NULL               (0x00)       ///< Unknown projector model.
#define kPROJECTOR_MODEL_LP01               (0x01)       ///< LP01. 
#define kPROJECTOR_MODEL_DP01               (0x02)       ///< DP01. 
#define kPROJECTOR_MODEL_DP03               (0x04)       ///< DP03. 
/** @} */

/**
* @struct  kProjectorStepsStyle
* @extends kValue
* @ingroup kFireSync
* @brief   Represents projector steps style.
*/
typedef k32s kProjectorStepsStyle;

#define kPROJECTOR_STEPS_STYLE_NULL                (0x00)      ///< Unknown steps style.
#define kPROJECTOR_STEPS_STYLE_SINGLE_EXPOSURE     (0x01)      ///< Single exposure steps style.
#define kPROJECTOR_STEPS_STYLE_MULTIPLE_EXPOSURE   (0x02)      ///< Multiple exposure steps style.
#define kPROJECTOR_STEPS_STYLE_OPTIONS             (0x03)      ///< Bitset containing all valid kProjectorStepsStyle options.

/**
 * @struct  kProjectorControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents projector control modes.
 */
typedef k32s kProjectorControl; 

/** @relates kProjectorControl @{ */
#define kPROJECTOR_CONTROL_NULL                     (0x00)      ///< Unknown projector control mode.
#define kPROJECTOR_CONTROL_TRIGGERED_BY_EVENT       (0x01)      ///< Projector exposure sequence triggered by event manager. 
#define kPROJECTOR_CONTROL_OPTIONS                  (0x01)      ///< Bitset containing all valid kProjectorControl options.
/** @} */

/** 
 * Formats a kProjectorControl value to a string buffer. 
 *
 * @public              @memberof kProjectorControl
 * @param   control     kProjectorControl value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorControl_Format(kProjectorControl control, kChar* text, kSize capacity); 

/**
 * @struct  kProjectorExtension
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents projector control extension numeric type identifiers. 
 */
typedef k32s kProjectorExtension; 

/** @relates kProjectorExtension @{ */
#define kPROJECTOR_EXTENSION_NULL            (0x00)          ///< Unknown projector sequence type.
#define kPROJECTOR_EXTENSION_STEPS           (0x01)          ///< Programmed state sequence. 
#define kPROJECTOR_EXTENSION_AUTO_INTENSITY  (0x02)          ///< Programmed auto intensity. 
#define kPROJECTOR_EXTENSION_OPTIONS         (0x03)          ///< Bitset containing all valid kProjectorExtension options.
/** @} */

/** 
 * Formats a kProjectorExtension value to a string buffer. 
 *
 * @public              @memberof kProjectorExtension
 * @param   sequence    kProjectorExtension value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorExtension_Format(kProjectorExtension sequence, kChar* text, kSize capacity); 

/** 
 * Gets the type information associated with a kProjectorExtension value.
 *
 * @public              @memberof kProjectorExtension
 * @param   extension   kProjectorExtension value. 
 * @return              Type information.
 */
kFsFx(kType) kProjectorExtension_ToType(kProjectorExtension extension); 

/** 
 * Gets the extension id associated with an extension type.
 *
 * @public              @memberof kProjectorExtension
 * @param   type        Type information.
 * @return              kProjectorExtension value. 
 */
kFsFx(kProjectorExtension) kProjectorExtension_FromType(kType type); 

/**
 * @struct  kProjectorPatternMode
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents projector pattern modes.
 */
typedef k32s kProjectorPatternMode; 

/** @relates kProjectorPatternMode @{ */
#define kPROJECTOR_PATTERN_MODE_NULL        (0x00)          ///< Unknown projector pattern mode.
#define kPROJECTOR_PATTERN_MODE_1D_8BPP     (0x01)          ///< Set of vertical 1D pattern arrays with 8-bit greyscale pixels. 
#define kPROJECTOR_PATTERN_MODE_2D_1BPP     (0x02)          ///< Single 2D pattern array with 1-bit greyscale pixels.
#define kPROJECTOR_PATTERN_MODE_1DH_8BPP    (0x04)          ///< Set of horizontal 1D pattern arrays with 8-bit greyscale pixels. 
#define kPROJECTOR_PATTERN_MODE_TEST_0      (0x08)          ///< Test pattern mode 0.
#define kPROJECTOR_PATTERN_MODE_TEST_1      (0x10)          ///< Test pattern mode 1.
#define kPROJECTOR_PATTERN_MODE_TEST_2      (0x20)          ///< Test pattern mode 2.
#define kPROJECTOR_PATTERN_MODE_TEST_3      (0x40)          ///< Test pattern mode 3.
#define kPROJECTOR_PATTERN_OPTIONS          (0x7F)          ///< Bitset containing all valid kProjectorPatternMode options.
/** @} */

/** 
 * Formats a kProjectorPatternMode value to a string buffer. 
 *
 * @public              @memberof kProjectorPatternMode
 * @param   mode        kProjectorPatternMode value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorPatternMode_Format(kProjectorPatternMode mode, kChar* text, kSize capacity); 

/**
* @struct  kProjectorPatternSource
* @extends kValue
* @ingroup kFireSync
* @brief   Represents projector pattern source.
*/
typedef k32s kProjectorPatternSource;

/** @relates kProjectorPatternMode @{ */
#define kPROJECTOR_PATTERN_SOURCE_NULL          (0)
#define kPROJECTOR_PATTERN_SOURCE_STREAMING     (1)
#define kPROJECTOR_PATTERN_SOURCE_STORED        (2)
/** @} */

#define kPROJECTOR_PALETTE_CAPACITY         (2)             ///< Count of defined projector palette pixels. 

/**
 * @struct  kProjectorStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents projector run-time statistics. 
 */
typedef struct kProjectorStats
{
    k64u exposureCount;                 ///< Exposure count.
    k64u exposureDropCount;             ///< Exposure drop count.
    k64u soeInterruptCount;             ///< Count of projector SOE interrupts.
    k64u eoeInterruptCount;             ///< Count of projector EOE interrupts.
    k64u controlErrorCount;             ///< Count of projector control errors (eg. PL communication errors).
    k64u displayErrorCount;             ///< Count of projector display errors (eg. LCOS/DLP communication errors).
    k64u missedInterruptCount;          ///< Count of missed projector interrupts.
    k64u sequenceDropCount;             ///< Sequence drop count.
    k64u overheatCount;                 ///< Overheat count.
    k64u errorFlags;                    ///< Bitset representing detailed, driver-specific, error information. 
    k32u effectiveIntensity;            ///< Projector effective intensity.
    kBool oscillatorCalibrated;         ///< Oscillator calibration status.
} kProjectorStats;

/**
 * @struct  kEncoderDirection
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents encoder A/B signal interpretation.
 */
typedef k32s kEncoderDirection;

/** @relates kEncoderDirection @{ */
#define kENCODER_DIRECTION_NULL             (0x00)          ///< Encoder direction unknonwn. 
#define kENCODER_DIRECTION_A_LEADS_B        (0x01)          ///< A leads B. 
#define kENCODER_DIRECTION_B_LEADS_A        (0x02)          ///< B leads A.
#define kENCODER_DIRECTION_OPTIONS          (0x03)          ///< Bitset containing all valid kEncoderDirection options.
/** @} */

/** 
 * Formats a kEncoderDirection value to a string buffer. 
 *
 * @public              @memberof kEncoderDirection
 * @param   direction   kEncoderDirection value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoderDirection_Format(kEncoderDirection direction, kChar* text, kSize capacity);

/**
 * @struct  kEncoderSnapshotSource
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents encoder snapshot source.
 */
typedef k32s kEncoderSnapshotSource;

/** @relates kEncoderSnapshotSource @{ */
#define kENCODER_SNAPSHOT_SOURCE_NULL               (0x00)      ///< Unknown snapshot source.
#define kENCODER_SNAPSHOT_SOURCE_NODE_INPUT         (0x01)      ///< Node input snapshot source.
#define kENCODER_SNAPSHOT_SOURCE_MASTER_INPUT       (0x02)      ///< Master input snapshot source.
#define kENCODER_SNAPSHOT_SOURCE_Z_PULSE            (0x04)      ///< Z pulse snapshot source.
#define kENCODER_SNAPSHOT_SOURCE_OPTIONS            (0x07)      ///< Bitset containing all valid kEncoderSnapshotSource options.
/** @} */

/** 
 * Formats a kEncoderSnapshotSource value to a string buffer. 
 *
 * @public              @memberof kEncoderSnapshotSource
 * @param   source      kEncoderSnapshotSource value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoderSnapshotSource_Format(kEncoderSnapshotSource source, kChar* text, kSize capacity);

/**
 * @struct  kDigitalOutControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents digital output control modes.
 */
typedef k32s kDigitalOutControl; 

/** @relates kDigitalOutControl @{ */
#define kDIGITAL_OUT_CONTROL_NULL                   (0x00)          ///< Unknown digital output control mode.
#define kDIGITAL_OUT_CONTROL_TRIGGERED_BY_EVENT     (0x01)          ///< Digital output pulse triggered by event manager. 
#define kDIGITAL_OUT_CONTROL_SET_BY_SOFTWARE        (0x02)          ///< Digital output state set by software commands.
#define kDIGITAL_OUT_CONTROL_TRIGGERED_BY_SOFTWARE  (0x04)          ///< Digital output pulse triggered by software commands.
#define kDIGITAL_OUT_CONTROL_COUPLED_TO_NODE_INPUT  (0x08)          ///< Digital output controlled by node input.
#define kDIGITAL_OUT_CONTROL_COUPLED_TO_PROJECTOR   (0x10)          ///< Digital output controlled by projector.
#define kDIGITAL_OUT_CONTROL_OPTIONS                (0x1F)          ///< Bitset containing all valid kDigitalOutControl options.
/** @} */

/**
 * @struct  kDigitalOutStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents digital output run-time statistics. 
 */
typedef struct kDigitalOutStats
{
    k64u outputCount;               ///< Output count.
    k64u outputDropCount;           ///< Output drop count.
} kDigitalOutStats;

/**
 * @struct  kAnalogOutModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents types of analog output (DAC) models. 
 */
typedef k32s kAnalogOutModel; 

/** @relates kAnalogOutModel @{ */
#define kANALOG_OUT_MODEL_UNKNOWN              (0)         ///< Unknown DAC model.
#define kANALOG_OUT_MODEL_MCP4725              (1)         ///< DAC model used by M2 controller. 
#define kANALOG_OUT_MODEL_AD5623               (2)         ///< DAC model used by M50 controller.
/** @} */

/**
 * @struct  kAnalogOutControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents analog output control modes.
 */
typedef k32s kAnalogOutControl; 

/** @relates kAnalogOutControl @{ */
#define kANALOG_OUT_CONTROL_NULL                    (0x00)              ///< Unknown analog output control mode.
#define kANALOG_OUT_CONTROL_TRIGGERED_BY_EVENT      (0x01)              ///< Analog output triggered by event manager. 
#define kANALOG_OUT_CONTROL_SET_BY_SOFTWARE         (0x02)              ///< Analog output level set by software commands.
#define kANALOG_OUT_CONTROL_OPTIONS                 (0x03)              ///< Bitset containing all valid kAnalogOutControl options.
/** @} */

/** 
 * Formats a kAnalogOutControl value to a string buffer. 
 *
 * @public              @memberof kAnalogOutControl
 * @param   control     kAnalogOutControl value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOutControl_Format(kAnalogOutControl control, kChar* text, kSize capacity);

/**
 * @struct  kAnalogOutStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents analog output run-time statistics. 
 */
typedef struct kAnalogOutStats
{
    k64u outputCount;               ///< Output count.
    k64u outputDropCount;           ///< Output drop count.
    k16u queueCount;                ///< Output queue current count.
    k64u overflowCount;             ///< Output queue overflow count.
    k64u underflowCount;            ///< Output queue underflow count.
} kAnalogOutStats;

/**
 * @struct  kSerialOutControl
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents serial output control modes.
 */
typedef k32s kSerialOutControl; 

/** @relates kSerialOutControl @{ */
#define kSERIAL_OUT_CONTROL_NULL                (0x00)              ///< Unknown serial output control mode.
#define kSERIAL_OUT_CONTROL_TRIGGERED_BY_EVENT  (0x01)              ///< Serial output triggered by event manager. 
#define kSERIAL_OUT_CONTROL_IMMEDIATE           (0x02)              ///< Serial output sent immediately.
#define kSERIAL_OUT_CONTROL_OPTIONS             (0x03)              ///< Bitset containing all valid kSerialOutControl options.
/** @} */

/** 
 * Formats a kSerialOutControl value to a string buffer. 
 *
 * @public              @memberof kSerialOutControl
 * @param   control     kSerialOutControl value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOutControl_Format(kSerialOutControl control, kChar* text, kSize capacity);

/**
 * @struct  kSerialOutMode
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents serial output data modes.
 */
typedef k32s kSerialOutMode;

/** @relates kSerialOutMode @{ */
#define kSERIAL_OUT_MODE_NULL                   (0x00)           ///< Unknown serial output data mode.
#define kSERIAL_OUT_MODE_UART                   (0x01)           ///< UART data output mode.
#define kSERIAL_OUT_MODE_SELCOM                 (0x02)           ///< Selcom data output mode.
#define kSERIAL_OUT_MODE_OPTIONS                (0x03)           ///< Bitset containing all valid kSerialOutMode options.
/** @} */

/** 
 * Formats a kSerialOutMode value to a string buffer. 
 *
 * @public              @memberof kSerialOutMode
 * @param   mode        kSerialOutMode value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOutMode_Format(kSerialOutMode mode, kChar* text, kSize capacity);

/**
 * @struct  kSerialOutStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents serial output run-time statistics. 
 */
typedef struct kSerialOutStats
{
    k64u outputCount;               ///< Count of output words transmitted. 
    k64u outputDropCount;           ///< Count of output words dropped. 
    k64u queueCount;                ///< Count of output words enqueued. 
    k64u overflowCount;             ///< Count of queue overflow events detected.
    k64u underflowCount;            ///< Count of queue underflow events detected.
} kSerialOutStats;

/**
 * @struct  kIoTestModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported I/O test jig device types.
 */
typedef k32s kIoTestModel; 

/** @relates kIoTestModel @{ */
#define kIO_TEST_MODEL_NULL               (0x00)       ///< Unknown IoTest model.
#define kIO_TEST_MODEL_AD16               (0x01)       ///< AD16 test jig. 
/** @} */

/** 
 * Formats a kIoTestModel value to a string buffer. 
 *
 * @public              @memberof kIoTestModel
 * @param   model       kIoTestModel value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTestModel_Format(kIoTestModel model, kChar* text, kSize capacity); 

/**
 * @struct  kIoTestSerialMode
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents I/O test jig serial output modes.
 */
typedef k32s kIoTestSerialMode;

/** @relates kIoTestSerialMode @{ */
#define kIO_TEST_SERIAL_MODE_NULL         (0x00)            ///< Unknown mode. 
#define kIO_TEST_SERIAL_MODE_UART         (0x01)            ///< UART test mode.
#define kIO_TEST_SERIAL_MODE_SELCOM       (0x02)            ///< Selcom test mode. 
#define kIO_TEST_SERIAL_MODE_OPTIONS      (0x03)            ///< Bitset containing all valid kIoTestSerialMode options.
/** @} */

/** 
 * Formats a kIoTestSerialMode value to a string buffer. 
 *
 * @public              @memberof kIoTestModel
 * @param   mode        kIoTestSerialMode value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTestSerialMode_Format(kIoTestSerialMode mode, kChar* text, kSize capacity); 

/**
 * @struct  kIoTestStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents I/O test jig run-time statistics. 
 */
typedef struct kIoTestStats
{
    k64f power5V;                   ///< Voltage reading of 5V, in Volt.
    k64f inputPower;                ///< Voltage reading of input power, in Volt. Could be 24 or 48V.
    k64f safety5V;                  ///< Voltage reading of SAFETY_5V, in Volt.
    k64f safetyPlus;                ///< Voltage reading of SAFETY+, in Volt.
    kBool analogIncalibrated[2];    ///< Is analog input calibrated?
    k64f analogIn[2];               ///< Current reading of analog output, in mA.
    k32u digitalIn;                 ///< Digital input stats.
    k32u uartFifoStatus;            ///< UART FIFO status (full, empty and overflowed).
    k32u selcomFifoStatus;          ///< Selcom FIFO status (full, empty and overflowed).
    k32u lastSelcomData;            ///< Last received Selcom data.
    k32u lastUartData;              ///< Last received UART data.
    k64u uartDataCount;             ///< Count of UART data available.
    k64u selcomDataCount;           ///< Count of Selcom data available.
} kIoTestStats;

#define kIO_TEST_DIGITAL_OUTPUT_COUNT       (2)  ///< Cout of digital output channel in the test jig.

/**
 * @struct  kOrientationModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported orientation device types.
 */
typedef k32s kOrientationModel;

/** @relates kOrientationModel @{ */
#define kORIENTATION_MODEL_NULL         (0x00)  ///< Unknown Orientation model.
#define kORIENTATION_MODEL_BNO055       (0x01)  ///< Bosch BNO055 IMU.
#define kORIENTATION_MODEL_OPTIONS      (0x01) 
/** @} */

#define kORIENTATION_MOTION_THRESHOLD_GRANULARITY   (7810)      ///< Orientation device motion detection threshold granularity (uG)
#define kORIENTATION_MOTION_DELAY_GRANULARITY       (1000000)   ///< Orientation device motion detection delay granularity (uS)

/**
 * Formats a kOrientationModel value to a string buffer.
 *
 * @public              @memberof kOrientationModel
 * @param   model       kOrientationModel value.
 * @param   text        Receives formatted string.
 * @param   capacity    Buffer capacity.
 * @return              Operation status.
 */
kFsFx(kStatus) kOrientationModel_Format(kOrientationModel model, kChar* text, kSize capacity);

/**
 * @struct  kOrientationAxis
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported orientation axes.
 */
typedef k32s kOrientationAxis;

/** @relates kOrientationAxis @{ */
#define kORIENTATION_AXIS_NULL          (0x00)
#define kORIENTATION_AXIS_X             (0x01)
#define kORIENTATION_AXIS_Y             (0x02)
#define kORIENTATION_AXIS_Z             (0x04)
#define kORIENTATION_AXIS_OPTIONS       (0x07)
/** @} */

/**
 * Formats a kOrientationAxis value to a string buffer.
 *
 * @public              @memberof kOrientationAxis
 * @param   axis        kOrientationAxis value.
 * @param   text        Receives formatted string.
 * @param   capacity    Buffer capacity.
 * @return              Operation status.
 */
kFsFx(kStatus) kOrientationAxis_Format(kOrientationAxis axis, kChar* text, kSize capacity);

/**
 * @struct  kAngle3d64f
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents orientation rotation about a specific axis.
 */
typedef struct kAngle3d64f
{
    k64f x;                     ///< rotation about x (pitch)
    k64f y;                     ///< rotation about y (roll)
    k64f z;                     ///< rotation about z (yaw)
} kAngle3d64f;

/**
 * @struct  kOrientationCalStatus
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents orientation calibration status.
 */
typedef struct kOrientationCalStatus
{
    k32u system;
    k32u accelerometer;
    k32u magnetometer;
    k32u gyroscope;
} kOrientationCalStatus;

/**
 * @struct  kOrientationStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents orientation device run-time statistics.
 */
typedef struct kOrientationStats
{
    kBool isStationary;
    kAngle3d64f angles;
    kOrientationCalStatus calStatus;
    kSize orientationTemp;
    kSize controlError;
    k32u systemStatus;
    k32u systemError;
} kOrientationStats;

/**
 * @struct  kOrientationOffsets
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents orientation device's offsets in different axes.
 */
typedef struct kOrientationOffsets
{
    k16s xOffset;
    k16s yOffset;
    k16s zOffset;
} kOrientationOffsets;

/**
 * @struct  kOrientationCalProfile
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents orientation device's profile.
 */
typedef struct kOrientationCalProfile
{
    kOrientationOffsets accelerometer;
    kOrientationOffsets magnetometer;
    kOrientationOffsets gyroscope;
    k16s accRadius;
    k16s magRadius;
} kOrientationCalProfile;

/**
 * @struct  kTestJigModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents supported test jig device types.
 */
typedef k32s kTestJigModel;

/** @relates kTestJigModel @{ */
#define kTEST_JIG_MODEL_NULL                (0x00)       ///< Unknown TestJig model.
#define kTEST_JIG_MODEL_M2                  (0x01)       ///< M2 test jig. 
#define kTEST_JIG_MODEL_ZYNQ7K              (0x02)       ///< M1/M3 test jig. 
/** @} */

/**
 * @struct  kTestJigCommand
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents test jig command types.
 */
typedef k32s kTestJigCommand;

/** @relates kTestJigCommand @{ */
#define kTEST_JIG_COMMAND_NULL                  (0x00)      ///< Unknown command.
#define kTEST_JIG_COMMAND_RESET                 (0x01)      ///< Reset test jig. 
#define kTEST_JIG_COMMAND_CAMERA_POWER_ON       (0x02)      ///< Turn on camera power. 
#define kTEST_JIG_COMMAND_CAMERA_POWER_OFF      (0x03)      ///< Turn off camera power. 
#define kTEST_JIG_COMMAND_SET_PL_REGISTER       (0x04)      ///< Set PL scratch register. 
#define kTEST_JIG_COMMAND_SET_PL_IO             (0x05)      ///< PL I/O output. 
#define kTEST_JIG_COMMAND_SET_PL_TEST_PIN       (0x06)      ///< PL test pin output. 
#define kTEST_JIG_COMMAND_SET_PS_TEST_PIN       (0x07)      ///< PS test pin output. 
#define kTEST_JIG_COMMAND_SET_CAMERA_DATA       (0x08)      ///< Set Camera data. 
#define kTEST_JIG_COMMAND_PL_DDR2_TEST          (0x09)      ///< PL DDR2 test. 
#define kTEST_JIG_COMMAND_VIDEO_PORT_TEST       (0x0A)      ///< Video port test. 
#define kTEST_JIG_COMMAND_GENERATE_INTERRUPT    (0x0B)      ///< Generate a PL interrupt. 
/** @} */

#define kTEST_JIG_CAMERA_COUNT                  (4)         ///< Numbers of supported cameras in the test jig.

/**
 * @struct  kTestJigStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents controller test jig run-time statistics.
 */
typedef struct kTestJigStats
{
    kStatus status;                             ///< Test result.
    k32u plRegister;                            ///< PL scratch register
    k32u plInput;                               ///< PL input status.
    k32u plTest;                                ///< PL test pins status.
    k32u psTest;                                ///< PS test pins status.
    k64u cameraInput0;                          ///< Camera input 0 status.
    k64u cameraInput1;                          ///< Camera input 1 status.
    k32u interruptCount;                        ///< PL interrupt count.
    kBool plIntegrity;                          ///< PL integrity check.
    k64u plWriteThroughput;                     ///< PL write throughput.
    k64u plReadThroughput;                      ///< PL read throughtput.
    k32s maxCpuTemp;                            ///< XADC maximum CPU temperature.
    k32s minCpuTemp;                            ///< XADC minimum CPU temperature.
    k32s maxVccInt;                             ///< XADC maximum VCCINT voltage.
    k32s minVccInt;                             ///< XADC minimum VCCINT voltage.
    k32s maxVccAux;                             ///< XADC maximum VCCAUX voltage.
    k32s minVccAux;                             ///< XADC minimum VCCAUX voltage.
    k32s maxVbram;                              ///< XADC maximum VBRAM voltage.
    k32s minVbram;                              ///< XADC minimum VBRAM voltage.
    k32s maxVccPint;                            ///< XADC maximum VCCPINT voltage.
    k32s minVccPint;                            ///< XADC minimum VCCPINT voltage.
    k32s maxVccPaux;                            ///< XADC maximum VCCPAUX voltage.
    k32s minVccPaux;                            ///< XADC minimum VCCPAUX voltage.
    k32s maxVccPdro;                            ///< XADC maximum VCCPDRO voltage.
    k32s minVccPdro;                            ///< XADC minimum VCCPDRO voltage.
} kTestJigStats;

/**
 * @struct  kPipeEngine
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents the type of data processing engine used in a kPipe instance. 
 */
typedef k32s kPipeEngine;

/** @relates kPipeEngine @{ */
#define kPIPE_ENGINE_NULL      (0x00)           ///< Unknown pipe engine. 
#define kPIPE_ENGINE_SERIAL    (0x01)           ///< Pipe engine that processes data in single background thread. 
#define kPIPE_ENGINE_PARALLEL  (0x02)           ///< Pipe engine that can process data in multiple background threads. 
#define kPIPE_ENGINE_OPTIONS   (0x03)           ///< Bitset containing all kPipeEngine options.
/** @} */

/** @relates kBlock @{ */

#define kBLOCK_SETTINGS_ROOT_NAME       "Settings"   ///< Required XML root name for block settings.

#define kBLOCK_ID_USER_BASE             (0)          ///< Minimum valid identifier for a user-defined block. 
#define kBLOCK_ID_USER_MAX              (0x7FFF)     ///< Maximum valid identifier for a user-defined block. 
#define kBLOCK_ID_SYSTEM_BASE           (0x8000)     ///< Minimum valid identifier for a system-defined block. 
#define kBLOCK_ID_CAMERA_BASE           (0x8000)     ///< Start of identifier range for camera blocks.  
#define kBLOCK_ID_NET_OUT               (0x9000)     ///< Unique identifier for the Net Out block. 
#define kBLOCK_ID_NET_IN                (0x9001)     ///< Unique identifier for the Net In block. 
#define kBLOCK_ID_LIGHT_BASE            (0xA000)     ///< Start of identifier range for light blocks.  
#define kBLOCK_ID_TIMER                 (0xB000)     ///< Start of identifier range for timer blocks.

kInlineFx(k16u) kBlockId_Camera(kSize id)   { return (k16u) (kBLOCK_ID_CAMERA_BASE + id); }     ///< Returns the block identifer for a given camera id. 
kInlineFx(k16u) kBlockId_Light(kSize id)    { return (k16u) (kBLOCK_ID_LIGHT_BASE + id);  }     ///< Returns the block identifer for a given light id. 
kInlineFx(k16u) kBlockId_NetOut()           { return (k16u) kBLOCK_ID_NET_OUT;  }               ///< Returns the block identifer for the Net Out block. 
kInlineFx(k16u) kBlockId_NetIn()            { return (k16u) kBLOCK_ID_NET_IN;  }                ///< Returns the block identifer for the Net In block. 
kInlineFx(k16u) kBlockId_Timer()            { return (k16u) kBLOCK_ID_TIMER;  }                 ///< Returns the block identifer for the timer block.

#define kPORT_ID_CAMERA_STATE_BASE      (0x0000)     ///< Start of identifier range for camera state ports.  
#define kPORT_ID_CAMERA_VIDEO_BASE      (0x1000)     ///< Start of identifier range for camera video ports.  
#define kPORT_ID_CAMERA_ALGORITHM_BASE  (0x2000)     ///< Start of identifier range for camera algorithm ports.  
#define kPORT_ID_CAMERA_STRIPE_BASE     (0x3000)     ///< Start of identifier range for camera stripe ports.  
#define kPORT_ID_CAMERA_PHASE_BASE      (0x4000)     ///< Start of identifier range for camera phase ports.  
#define kPORT_ID_CAMERA_INPUT_BASE      (0x0100)     ///< Start of identifier range for camera input ports.  

/** Returns the camera state port identifer for a given state id. */
kInlineFx(k16u) kPortId_CameraState(kSize id) { return kPORT_ID_CAMERA_STATE_BASE + (k16u)id; }

/** Returns the camera video port identifer for a given state id. */
kInlineFx(k16u) kPortId_CameraVideo(kSize id) { return kPORT_ID_CAMERA_VIDEO_BASE + (k16u)id; }

/** Returns the camera algorithm port identifer for a given state id.*/
kInlineFx(k16u) kPortId_CameraAlgorithm(kSize id) { return kPORT_ID_CAMERA_ALGORITHM_BASE + (k16u)id; }

/** Returns the camera stripe port identifer for a given state id. */
kInlineFx(k16u) kPortId_CameraStripe() { return kPORT_ID_CAMERA_STRIPE_BASE; }

/** Returns the camera phase port identifer for a given state id. */
kInlineFx(k16u) kPortId_CameraPhase() { return kPORT_ID_CAMERA_PHASE_BASE; }

/** Returns the camera override input port identifer. */
kInlineFx(k16u) kPortId_CameraOverride() { return kPORT_ID_CAMERA_INPUT_BASE; }

/** Returns the camera calibration input port identifer. */
kInlineFx(k16u) kPortId_CameraCalibration() { return kPORT_ID_CAMERA_INPUT_BASE + 1; }

/** Returns the camera dynamic FPN input port identifer. */
kInlineFx(k16u) kPortId_CameraDynFpn() { return kPORT_ID_CAMERA_INPUT_BASE + 2; }

/** @} */

/**
 * @struct  kPortDirection
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents the directionality of a block port.
 */
typedef k32s kPortDirection;

/** @relates kPortDirection @{ */
#define kPORT_DIRECTION_NULL        (0x00)          ///< Unknown port direction.
#define kPORT_DIRECTION_INPUT       (0x01)          ///< Specifies that a port is an input port.
#define kPORT_DIRECTION_OUTPUT      (0x02)          ///< Specifies that a port is an output port.
#define kPORT_DIRECTION_OPTIONS     (0x03)          ///< Bitset containing all kPortDirection options.
/** @} */

/**
 * @struct  kRoutePoint
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents a source or destination endpoint address for a pipe data route.
 */
typedef struct kRoutePoint
{
    k16u block;             ///< Block identifer. 
    k16u port;              ///< Port identifier. 
} kRoutePoint; 

/**
 * @struct  kRouteEntry
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents a pair of source/destination endpoints describing a pipe data route.
 */
typedef struct kRouteEntry
{
    kRoutePoint source;         ///< Source endpoint.
    kRoutePoint dest;           ///< Destination endpoint.
} kRouteEntry; 

/**
 * @struct  kDataInSource
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents a remote Net Out port that will act as the source for a local Net In port. 
 */
typedef struct kDataInSource
{
    k32u sourceNode;            //Remote node device id. 
    k16u sourcePort;            //Remote node Net Out block port id. 
} kDataInSource; 

/**
 * @struct  kActionTrigger
 * @extends kValue
 * @ingroup kFireSync-Action
 * @brief   Represents an action trigger type.
 */
typedef k32s kActionTrigger;  

/** @relates kActionTrigger @{ */
#define kACTION_TRIGGER_NULL            (0x00)         ///< Invalid/unknown trigger. 
#define kACTION_TRIGGER_INVOKE          (0x01)         ///< Action is executed manually via kNode_InvokeAction
#define kACTION_TRIGGER_BOOT            (0x02)         ///< Action is executed automatically at node boot time.
#define kACTION_TRIGGER_START           (0x04)         ///< Action is executed automatically when node is started.
#define kACTION_TRIGGER_STOP            (0x08)         ///< Action is executed automatically when node is stopped.
#define kACTION_TRIGGER_PAUSE           (0x10)         ///< Action is executed automatically when node is paused.
#define kACTION_TRIGGER_RESUME          (0x20)         ///< Action is executed automatically when node is resumed.
#define kACTION_TRIGGER_OPTIONS         (0x3F)         ///< Bitset containing all kActionTrigger options.
/** @} */

/** 
 * Formats a kActionTrigger value to a string buffer. 
 *
 * @public              @memberof kActionTrigger
 * @param   type        kActionTrigger value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kActionTrigger_Format(kActionTrigger type, kChar* text, kSize capacity); 

/** @relates kAction @{ */
#define kACTION_SETTINGS_ROOT_NAME  "Settings"   ///< Required XML root name for action settings.
/** @} */

/**
 * @struct  kSocType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents system-on-chip families.
 */
typedef k32s kSocType; 

/** @relates kSocType @{ */
#define kSOC_TYPE_UNKNOWN                   (0)         ///< Unknown controller.
#define kSOC_TYPE_VIRTUAL                   (1)         ///< Virtual SoC. 
#define kSOC_TYPE_TI_DM647                  (2)         ///< Texas Instruments TMS320DM647.
#define kSOC_TYPE_XILINX_ZYNQ_7020          (3)         ///< Xilinx Zynq-7020.
#define kSOC_TYPE_XILINX_ZYNQ_7030          (4)         ///< Xilinx Zynq-7030. 
#define kSOC_TYPE_NVIDIA_PARKER             (5)         ///< Nvidia Parker series.
#define kSOC_TYPE_NVIDIA_XAVIER             (6)         ///< Nvidia Xavier series.
/** @} */

/**
 * @struct  kControllerType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents types of node controllers. 
 */
typedef k32s kControllerType; 

/** @relates kControllerType @{ */
#define kCONTROLLER_TYPE_UNKNOWN        (0)         ///< Unknown controller.
#define kCONTROLLER_TYPE_VIRTUAL        (1)         ///< Host-based virtual device. 
#define kCONTROLLER_TYPE_M2             (2)         ///< M2 controller (DM647).
#define kCONTROLLER_TYPE_M3             (3)         ///< M3 controller (Zynq7k).
#define kCONTROLLER_TYPE_M1             (4)         ///< M1 controller (Zynq7k).
#define kCONTROLLER_TYPE_M50            (5)         ///< M50 controller (DM647).
#define kCONTROLLER_TYPE_M3_UC          (6)         ///< M3-uC controller (Zynq7k).
#define kCONTROLLER_TYPE_TX2            (7)         ///< TX2 controller.
#define kCONTROLLER_TYPE_XAVIER_NX      (8)         ///< Xavier NX controller.
/** @} */

/** 
 * Formats a kControllerType value to a string buffer. 
 *
 * @public              @memberof kControllerType
 * @param   type        kControllerType value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kControllerType_Format(kControllerType type, kChar* text, kSize capacity); 

/**
 * @struct  kNodeType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents types of nodes. 
 */
typedef k32s kNodeType; 

/** @relates kNodeType @{ */
#define kNODE_TYPE_NULL             (0)             ///< Unknown node. 
#define kNODE_TYPE_SENSOR           (1)             ///< Sensor node. 
#define kNODE_TYPE_MASTER           (2)             ///< Master node. 
#define kNODE_TYPE_STATION          (3)             ///< Station node. 
#define kNODE_TYPE_LOCAL            (4)             ///< Local co-processing node. 
/** @} */

/**
 * @struct  kNodeId
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents a FireSync node identifier. 
 * 
 * FireSync node identifers are 32-bit unsigned values. k32u is used instead 
 * of kNodeId throughout most of the FireSync API and FireSync applications.
 * The kNodeId type is defined only to organize functions and macros that can be 
 * used to compose these 32-bit node identifiers.  
 * 
 * For the most part, node identifers are synonymous with manufacturing serial 
 * numbers. However, with the introduction of host-local nodes to support distributed
 * data processsing, the uppermost bit of the node id is now reserved to distinguish 
 * between local nodes and primary/network nodes. This convention provides a 
 * namespace so that local node ids can be defined without risk of conflicting 
 * with primary node ids. This use of the uppermost bit to identify local nodes is 
 * strictly optional; it is valid to create local nodes identifiers without setting 
 * this bit. 
 */
typedef k32u kNodeId; 

/** 
 * Composes a node identifier from its constituent parts. 
 *
 * @public                  @memberof kNodeId
 * @param   isLocal         Is the node a host-local node? 
 * @param   serialNumber    Unique numeric identifier.
 * @return                  Node identifier.
 */
kFsFx(kNodeId) kNodeId_Create(kBool isLocal, k32u serialNumber); 

/** 
 * Extracts the serial number from a node identifier. 
 *
 * @public          @memberof kNodeId
 * @param   id      Node identifier.
 * @return          Serial number.
 */
kFsFx(k32u) kNodeId_SerialNumber(kNodeId id); 

/** 
 * Reports whether a node identifier represents a local coprocessing node.
 *
 * @public          @memberof kNodeId
 * @param   id      Node identifier.
 * @return          kTRUE if the node is a local coprocessing node. 
 */
kFsFx(kBool) kNodeId_IsLocal(kNodeId id); 

/** 
 * Formats a kNodeId value to a string buffer. 
 *
 * @public              @memberof kNodeId
 * @param   id          kNodeId value. 
 * @param   buffer      Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeId_Format(kNodeId id, kChar* buffer, kSize capacity); 

/** 
 * Parses a kNodeId value from a string buffer. 
 *
 * @public              @memberof kNodeId
 * @param   id          Receives the kNodeId value. 
 * @param   buffer      Formatted string. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeId_Parse(kNodeId* id, const kChar* buffer); 

/**
 * @struct  kDaughterboardModel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents model of daughterboard. 
 */
typedef k32s kDaughterboardModel;

/** @relates kDaughterboardModel @{ */
#define kDAUGHTERBOARD_MODEL_NULL       (0x00)      ///< Unknown model.
#define kDAUGHTERBOARD_MODEL_NONE       (0x01)      ///< Daughterboard is not connected.
#define kDAUGHTERBOARD_MODEL_LP01       (0x02)      ///< LP01
#define kDAUGHTERBOARD_MODEL_LP01_12V   (0x04)      ///< LP01 12V
#define kDAUGHTERBOARD_MODEL_DP01_5A    (0x08)      ///< DP01-5A
#define kDAUGHTERBOARD_MODEL_DP01_1A    (0x10)      ///< DP01-1A
#define kDAUGHTERBOARD_MODEL_DP03       (0x20)      ///< DP03
#define kDAUGHTERBOARD_MODEL_OPTION     (0x3F)      ///< Bitset containing all kDaughterboardModel options.
/** @} */

/** 
 * Formats a kDaughterboardModel value to a string buffer. 
 *
 * @public              @memberof kDaughterboardModel
 * @param   model       kDaughterboardModel value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kDaughterboardModel_Format(kDaughterboardModel model, kChar* text, kSize capacity); 

/**
 * @struct  kNodeFirmwareType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents types of node firmware. 
 */
typedef k32s kNodeFirmwareType; 

/** @relates kNodeFirmwareType @{ */
#define kNODE_FIRMWARE_TYPE_BOOT_LOADER                 (0)         ///< Boot loader. 
#define kNODE_FIRMWARE_TYPE_MAIN_SOFTWARE_PROGRAM       (1)         ///< Main software program. 
#define kNODE_FIRMWARE_TYPE_RESCUE_SOFTWARE_PROGRAM     (2)         ///< Rescue software program.
#define kNODE_FIRMWARE_TYPE_MAIN_PL_PROGRAM             (3)         ///< Main programmable logic program.
#define kNODE_FIRMWARE_TYPE_SECONDARY_PL_PROGRAM        (4)         ///< Secondary programmable logic program.
#define kNODE_FIRMWARE_TYPE_MAIN_OPERATING_SYSTEM       (5)         ///< Main operating system. 
#define kNODE_FIRMWARE_TYPE_RESCUE_OPERATING_SYSTEM     (6)         ///< Rescue operating system. 
/** @} */

/** 
 * Formats a kNodeFirmwareType value to a string buffer. 
 *
 * @public              @memberof kNodeFirmwareType
 * @param   type        kNodeFirmwareType value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeFirmwareType_Format(kNodeFirmwareType type, kChar* text, kSize capacity);

/**
 * @struct  kBootMode
 * @ingroup kFireSync
 * @brief   Represents the mode by which a node's program was loaded.
 */
typedef k32s kBootMode;

/** @relates kBootMode @{ */
#define kBOOT_MODE_NULL             (0x0)      ///< Program loading method unknown. 
#define kBOOT_MODE_NORMAL           (0x1)      ///< Program was loaded normally by bootloader.
#define kBOOT_MODE_RESCUE           (0x2)      ///< Program was loaded by bootloader under rescue conditions.
#define kBOOT_MODE_NETWORK          (0x4)      ///< Program was loaded from network.
#define kBOOT_MODE_DEBUGGER         (0x8)      ///< Program was loaded directly via hardware debug emulator.
#define kBOOT_MODE_FALLBACK        (0x10)      ///< Program is the boot loader, operating in fallback mode.
/** @} */

/** 
 * Formats a kBootMode value to a string buffer. 
 *
 * @public              @memberof kBootMode
 * @param   mode        kBootMode value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBootMode_Format(kBootMode mode, kChar* text, kSize capacity);

#define kNETWORK_BOOT_FTP_USER      "firesync"      ///< FTP server user name expected for network (debug) boot.
#define kNETWORK_BOOT_FTP_PASS      "firesync"      ///< FTP server password expected for network (debug) boot.

/**
 * @struct  kRestartReason
 * @ingroup kFireSync
 * @brief   Represents the reason that a device was rebooted. 
 * 
 * This enum is used in conjunction with watchdog timers to represent the reason 
 * that a device is rebooted. 
 */
typedef k32s kRestartReason; 

/** @relates kRestartReason @{ */
#define kRESTART_REASON_NONE                  (0)       ///< Device not rebooted by watchdog. 

#define kRESTART_REASON_UNRESPONSIVE          (1)       ///< Software became unresponsive; reset by hardware watchdog. 
#define kRESTART_REASON_MEMORY_EXHAUSTED      (2)       ///< Software detected memory exhausted.
#define kRESTART_REASON_DEADLOCK              (3)       ///< Software detected thread deadlock in part of the program. 
#define kRESTART_REASON_CRASH                 (4)       ///< Software detected a crash and initiated a reboot. 

#define kRESTART_REASON_APP_CODE_0            (8)       ///< Application-specific failure code 0. 
#define kRESTART_REASON_APP_CODE_1            (9)       ///< Application-specific failure code 1. 
#define kRESTART_REASON_APP_CODE_2            (10)      ///< Application-specific failure code 2.
#define kRESTART_REASON_APP_CODE_3            (11)      ///< Application-specific failure code 3. 

/** @} */

/**
 * @struct  kSyncSource
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents FireSync synchronization sources.
 */
typedef k32s kSyncSource;

/** @relates kSyncSource @{ */
#define kSYNC_SOURCE_NULL                   (0x00)          ///< Unknown synchronization source.
#define kSYNC_SOURCE_MASTER                 (0x01)          ///< Accept synchronization from master. 
#define kSYNC_SOURCE_SENSOR                 (0x02)          ///< Use local node synchronization. 
#define kSYNC_SOURCE_OPTIONS                (0x03)          ///< Bitset containing all kSyncSource options.
/** @} */

/** 
 * Formats a kSyncSource value to a string buffer. 
 *
 * @public              @memberof kSyncSource
 * @param   source      kSyncSource value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSyncSource_Format(kSyncSource source, kChar* text, kSize capacity); 

/**
 * @struct  kNodeState
 * @ingroup kFireSync
 * @brief   Represents the current state of a node object.
 */
typedef k32s kNodeState;

/** @relates kNodeState @{ */
#define kNODE_STATE_ONLINE              (0)        ///< Node disconnected, but detected via discovery.
#define kNODE_STATE_OFFLINE             (1)        ///< Node disconnected and no longer detected via discovery (refresh system to remove sensor). 
#define kNODE_STATE_RESETTING           (2)        ///< Node disconnected and currently resetting (wait for completion).
#define kNODE_STATE_INCOMPATIBLE        (3)        ///< Node connected, but protocol incompatible with client (upgrade required). 
#define kNODE_STATE_INCONSISTENT        (4)        ///< Node connected, but remote state was changed (refresh sensor). 
#define kNODE_STATE_UNRESPONSIVE        (5)        ///< Node connected, but no longer detected via health or discovery (disconnect).
#define kNODE_STATE_CANCELLED           (6)        ///< Node connected, but communication aborted via kNode_Cancel function (disconnect or refresh sensor).
#define kNODE_STATE_READY               (7)        ///< Node connected and ready to accept configuration commands.
#define kNODE_STATE_PAUSED              (8)        ///< Node connected and currently paused.
#define kNODE_STATE_RUNNING             (9)        ///< Node connected and currently running.
#define kNODE_STATE_REPLAY              (10)       ///< Node connected and ready to accept replay commands.
/** @} */

/** 
 * Reports whether the given state implies the potential for a successful control channel connection. 
 *
 * @public              @memberof kNodeState
 * @param   state       kNodeState value. 
 * @return              kTRUE if node is connectable; kFALSE otherwise. 
 */
kFsFx(kBool) kNodeState_IsConnectable(kNodeState state); 

/** 
 * Reports whether the given state implies responsiveness via control channel. 
 *
 * @public              @memberof kNodeState
 * @param   state       kNodeState value. 
 * @return              kTRUE if node is responsive; kFALSE otherwise. 
 */
kFsFx(kBool) kNodeState_IsResponsive(kNodeState state); 

/** 
 * Reports whether the given state implies that configuration is readable via control channel. 
 *
 * @public              @memberof kNodeState
 * @param   state       kNodeState value. 
 * @return              kTRUE if configuration is readable; kFALSE otherwise. 
 */
kFsFx(kBool) kNodeState_IsReadable(kNodeState state); 

/** 
 * Reports whether the given state implies that a device will accept configuration via control channel. 
 *
 * @public              @memberof kNodeState
 * @param   state       kNodeState value. 
 * @return              kTRUE if node is configurable; kFALSE otherwise. 
 */
kFsFx(kBool) kNodeState_IsConfigurable(kNodeState state); 

/** 
 * Reports whether the given state implies that a device is operating normally. 
 *
 * @public              @memberof kNodeState
 * @param   state       kNodeState value. 
 * @return              kTRUE if node is operating normally; kFALSE otherwise. 
 */
kFsFx(kBool) kNodeState_IsNormal(kNodeState state); 

/** 
 * Formats a kNodeState value to a string buffer. 
 *
 * @public              @memberof kNodeState
 * @param   state       kNodeState value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeState_Format(kNodeState state, kChar* text, kSize capacity); 

/**
 * @struct  kNodeInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents detailed information about node hardware/firmware. 
 */
typedef struct kNodeInfo
{
    kNodeType type;                         ///< Device category.
    kControllerType controllerType;         ///< Controller hardware type. 
    kVersion controllerVersion;             ///< Controller hardware version. 
    kSocType socType;                       ///< Controller SoC type.
    k32u socSpeedGrade;                     ///< Controller SoC speed grade.
    k32u partNumber;                        ///< Controller part number.
    k32u cpuCount;                          ///< Count of CPUs in device (0 if unknown). 
    k64u cpuFrequency;                      ///< CPU clock frequency (Hz; 0 if unknown).
    k64u memorySize;                        ///< Amount of volatile memory (bytes; 0 if unknown). 
    k64u storageSize;                       ///< Amount of non-volatile memory (bytes; 0 if unknown).
    kBool controllerFactoryResult;          ///< Controller factory test result.
    kVersion bootloaderVersion;             ///< Bootloader program version.
    kBootMode bootMode;                     ///< Method by which software was loaded. 
    kVersion platformVersion;               ///< FireSync platform firmware version.
    kVersion protocolVersion;               ///< FireSync protocol version. 
    k32u mainPlConfig;                      ///< Main programmable logic configuration identifier. 
    kVersion mainPlVersion;                 ///< Main programmable logic program version. 
    k32u secondaryPlConfig;                 ///< Secondary programmable logic configuration identifier.
    kVersion secondaryPlVersion;            ///< Secondary programmable logic program version. 
    kText32 appName;                        ///< Application name. 
    kVersion appVersion;                    ///< Application version. 
    kVersion osVersion;                     ///< Operating system version. 
    kDaughterboardModel daughterboardModel; ///< Daughterboard model.
    kVersion daughterboardVersion;          ///< Daughterboard version.
    kBool isDebug;                          ///< Was the firmware built with debug settings? 
    kRestartReason restartReason;           ///< Reports restart details if rebooted by watchdog.
    k32u crashCount;                        ///< Reports number of crashes encountered by the device.
} kNodeInfo; 

/**
 * @struct  kNodeStats
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents node run-time statistics. 
 */
typedef struct kNodeStats
{
    k8u cpuUsed;                    ///< CPU utilization (percent). 
    k64u memoryUsed;                ///< Device memory currently in use (bytes).
    k64u memoryCapacity;            ///< Device memory capacity (bytes).
    k64u memoryLeaked;              ///< Device memory leaked (bytes); not supported on all platforms. 
    k64u storageUsed;               ///< Device storage ("user" volume) currently in used (bytes).
    k64u storageCapacity;           ///< Device storage ("user" volume) capacity (bytes).
    k64u netDataSent;               ///< Total network data bytes sent.
    k64u netDataReceived;           ///< Total network data bytes received.
    k64u uptime;                    ///< Time since boot (microseconds). 
    k64u time;                      ///< Current time (sync us). 
    k64s encoder;                   ///< Current encoder value. 
    kDateTime dateTime;             ///< Current date/time (microseconds since 00:00:00 Jan 1, 1 CE, UTC); zero if unknown.
    k16u digitalInputs;             ///< Current digital input states. 
    k32s internalTemp;              ///< Internal device temperature (millidegrees C).
    k32u recoveryCount;             ///< Count of error recovery operations.
    kSyncSource syncSource;         ///< Current synchronization source. 
    k64u syncErrorCount;            ///< Count of synchronization errors. 
} kNodeStats;

/**
 * @struct  kNodeVariableInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents information about a remote variable. 
 */
typedef struct kNodeVariableInfo
{
    kType type;             ///< Variable type.   
    kSize size;             ///< Variable size (bytes). 
} kNodeVariableInfo; 

#define kNODE_DIR_SYSTEM            "system"            ///< Standard directory name for node system files.
#define kNODE_DIR_TEMP              "temp"              ///< Standard directory name for node temp files.
#define kNODE_DIR_USER              "user"              ///< Standard directory name for node user files.
#define kNODE_DIR_HEALTH_LOG        "health"            ///< Standard directory name for node health log.

#define kNODE_PATH_SYSTEM           "/system"           ///< Standard path name for node system files.
#define kNODE_PATH_TEMP             "/temp"             ///< Standard path name for node temp files.
#define kNODE_PATH_USER             "/user"             ///< Standard path name for node user files.
#define kNODE_PATH_HEALTH_LOG       "/health"           ///< Standard path name for node health log.
#define kNODE_PATH_ROOT             "/"                 ///< Standard path name for node root directory.

/**
 * @struct  kNodeClearOption
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents a FireSync node clear option. 
 */
typedef k32s kNodeClearOption; 

/** @relates kNodeClearOption @{ */
#define kNODE_CLEAR_OPTION_NULL              (0x00000000)        ///< Empty clear option.
#define kNODE_CLEAR_OPTION_MODE_HW           (0x00000001)        ///< Transient hardware-related settings (e.g., camera exposure). 
#define kNODE_CLEAR_OPTION_MODE_PIPE         (0x00000002)        ///< Transient pipe settings.
#define kNODE_CLEAR_OPTION_MODE_ACTION       (0x00000004)        ///< Transient action settings.
#define kNODE_CLEAR_OPTION_MODE_NET          (0x00000008)        ///< Transient net settings.
#define kNODE_CLEAR_OPTION_MODE_ALL          (0x000000FF)        ///< All transient settings.
#define kNODE_CLEAR_OPTION_DEV_HW            (0x00000100)        ///< Persistent hardware settings (e.g., camera frame buffer size). 
#define kNODE_CLEAR_OPTION_DEV_BOOT          (0x00000200)        ///< Boot configuration. 
#define kNODE_CLEAR_OPTION_DEV_ID            (0x00000400)        ///< Device id. 
#define kNODE_CLEAR_OPTION_DEV_NET           (0x00000800)        ///< Persistent network settings (e.g., link speed, IP address).
#define kNODE_CLEAR_OPTION_DEV_ALL           (0x0000FF00)        ///< All persistent device settings.
#define kNODE_CLEAR_OPTION_VARIABLES         (0x00010000)        ///< Node variables. 
#define kNODE_CLEAR_OPTION_STORAGE_DATA      (0x00020000)        ///< Storage user/temp files.
#define kNODE_CLEAR_OPTION_STORAGE_LOGS      (0x00040000)        ///< Special log files, such as crash logs and health logs.
#define kNODE_CLEAR_OPTION_ALL               (0x00FFFFFF)        ///< All settings (excluding hazardous, special-case settings, e.g., laser power limits.)
/** @} */

/**
 * @struct  kNodeMemorySpace
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents an addressable memory space within a node.
 */
typedef k32u kNodeMemorySpace; 

#define kNODE_MEMORY_SPACE_SOC              (1)                  ///< System-on-chip memory space. 

/**
 * @struct  kRegisterPermission
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents Pl module permissions.
 */
typedef k32s kRegisterPermission; 

/** @relates kRegisterPermission @{ */
#define kREGISTER_PERMISSION_NULL          (0x00) ///< Null.
#define kREGISTER_PERMISSION_READ          (0x01) ///< Can be read.
#define kREGISTER_PERMISSION_WRITE         (0x02) ///< Can be written.
#define kREGISTER_PERMISSION_PEEK          (0x04) ///< Can be non-destructively read.
/** @} */

/**
 * @struct  kRegisterInfo32u
 * @extends kValue
 * @ingroup kFireSync 
 * @brief   Represents 32-bit address/data register pair info. 
 */
typedef struct kRegisterInfo32u
{
    k32u address;                       ///< Address value.
    k32u data;                          ///< Data value.
    kRegisterPermission permissions;    ///< Permissions.
} kRegisterInfo32u; 

/**
 * @struct  kRegisterOverride32u
 * @extends kValue
 * @ingroup kFireSync 
 * @brief   Represents 32-bit address/data register pair override data. 
 */
typedef struct kRegisterOverride32u
{
    kSize moduleIndex;                  ///< Module index.
    k32u address;                       ///< Address value.
    k32u data;                          ///< Data value.
} kRegisterOverride32u; 

#define kPL_MODULE_CAPACITY                     (64)    ///< PL module count.
#define kPL_MODULE_REGISTER_CAPACITY            (64)    ///< Number of registers per PL module.
#define kPL_MODULE_CAMERA_REGISTER_CAPACITY     (128)   ///< Number of registers per camera module.

#define kPL2_MODULE_CAPACITY                    (64)    ///< PL2 module count.
#define kPL2_MODULE_REGISTER_CAPACITY           (64)    ///< Number of registers per PL2 module.

#define kCAMERA_MODULE_CAPACITY                 (2)     ///< Camera module count.
#define kCAMERA_REGISTER_CAPACITY               (512)   ///< Maximum number of registers per camera module.

/**
 * @struct  kRegisterModule
 * @extends kValue
 * @ingroup kFireSync-Data
 * @brief   Represents an enumeration of register modules.
 */
typedef k32s kRegisterModule;

/** @relates kRegisterModule @{ */
#define kREGISTER_MODULE_PL                     (0)     ///< Main PL. 
#define kREGISTER_MODULE_PL2                    (1)     ///< Secondary PL. 
#define kREGISTER_MODULE_CAMERA                 (2)     ///< Camera.
#define kREGISTER_MODULE_CAPACITY               (3)     ///< Count of defined register modules. 
/** @} */

/**
 * @struct  kGpioBankType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents GPIO bank types.
 */
typedef k32s kGpioBankType; 

/** @relates kGpioBankType @{ */
#define kGPIO_BANK_TYPE_UNKNOWN                       (0x00)    ///< Unknown.
#define kGPIO_BANK_TYPE_PL                            (0x01)    ///< Digital input GPIO bank.
/** @} */

/** 
 * Formats a kGpioBankType value to a string buffer. 
 *
 * @public              @memberof kGpioBankType
 * @param   bankId      kGpioBankType value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBankType_Format(kGpioBankType bankId, kChar* text, kSize capacity);

/**
 * @struct  kGpioPin
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents GPIO pin types.
 */
typedef k32s kGpioPin; 

/** @relates kGpioPin @{ */
#define kGPIO_PIN_MISC_IO                       (-1)        ///< Misc I/O pin.
#define kGPIO_PIN_GENERIC_INPUT                 (0x01)      ///< Generic GPIO input pin.
#define kGPIO_PIN_GENERIC_OUTPUT                (0x02)      ///< Generic GPIO output pin.
#define kGPIO_PIN_PUSH_BUTTON                   (0x03)      ///< Push Button input (with software debouncing).
/** @} */

/** 
 * Formats a kGpioPin value to a string buffer. 
 *
 * @public              @memberof kGpioPin
 * @param   pinType     kGpioPin value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioPin_Format(kGpioPin pinType, kChar* text, kSize capacity);

/**
 * @struct  kLed
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents indicator LED types.
 */
typedef k32s kLed; 

/** @relates kLed @{ */
#define kLED_CPU            (0x00)      ///< CPU LED.
#define kLED_PL             (0x01)      ///< PL LED.
#define kLED_RANGE          (0x02)      ///< Range LED.
/** @} */

/**
* @struct  kLedMode
* @extends kValue
* @ingroup kFireSync
* @brief   Represents indicator LED operational modes.
*/
typedef k32s kLedMode;

/** @relates kLedMode @{ */
#define kLED_MODE_NULL          (0x00)      ///< Undefined.
#define kLED_MODE_OFF           (0x01)      ///< Off.
#define kLED_MODE_ON            (0x02)      ///< On.
#define kLED_MODE_BLINK_125_MS  (0x04)      ///< Blink with 125ms period (62.5ms on/off states).
#define kLED_MODE_BLINK_250_MS  (0x08)      ///< Blink with 250ms period (125ms on/off states).
#define kLED_MODE_BLINK_500_MS  (0x10)      ///< Blink with 500ms period (250ms on/off states).
#define kLED_MODE_BLINK_1000_MS (0x20)      ///< Blink with 1000ms period (500ms on/off states).
#define kLED_MODE_BLINK_2000_MS (0x40)      ///< Blink with 2000ms period (1000ms on/off states).
#define kLED_MODE_BLINK_4000_MS (0x80)      ///< Blink with 4000ms period (2000ms on/off states).
#define kLED_MODE_BLINK_OPTIONS (0xFF)      ///< Blink with 4000ms period (2000ms on/off states).
/** @} */

/**
 * @struct  kMemoryTrafficOp
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents kMemoryTraffic operations.
 */
typedef k32s kMemoryTrafficOp;

/** @relates kMemoryTrafficOp @{ */
#define kMEMORY_TRAFFIC_OP_COPY     (0x01)      ///< Copy.
#define kMEMORY_TRAFFIC_OP_READ     (0x02)      ///< Read.
#define kMEMORY_TRAFFIC_OP_WRITE    (0x04)      ///< Write.
#define kMEMORY_TRAFFIC_OP_OPTIONS  (0x07)      ///< Valid kMemoryTraffic operations.
/** @} */

/**
 * @struct  kMsgSource
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents information about the source of a pipe message. 
 */
typedef struct kMsgSource
{
    k32u nodeId;            ///< Source node identifer. 
    k16u blockId;           ///< Source block identifer. 
    k16u portId;            ///< Source port identifer. 
} kMsgSource; 

/** 
 * Composes a kMsgSource value based on its constituent parts. 
 *
 * @public              @memberof kMsgSource
 * @param   nodeId      Source node identifer. 
 * @param   blockId     Source block identifer. 
 * @param   portId      Source port identifer. 
 * @return              Operation status. 
 */
kFsFx(kMsgSource) kMsgSource_Create(k32u nodeId, k16u blockId, k16u portId); 

/** 
 * Initializes a kMsgSource value based on its constituent parts. 
 *
 * @public              @memberof kMsgSource
 * @param   source      Pointer to kMsgSource value.
 * @param   nodeId      Source node identifer. 
 * @param   blockId     Source block identifer. 
 * @param   portId      Source port identifer. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kMsgSource_Init(kMsgSource* source, k32u nodeId, k16u blockId, k16u portId)
{
    source->nodeId = nodeId; 
    source->blockId = blockId; 
    source->portId = portId; 

    return kOK; 
}

#define kSTAMP_SIGNATURE        (0x12345678)        ///< Constant at beginning of each stamp value.

#define kSTAMP_STATUS_MAX_INPUT_COUNT  (3)                 ///< Maximum input count change that can be reported.

/**
 * @struct  kStampStatus
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents status data in the kStamp structure. 
 */
typedef k16u kStampStatus; 

/** 
 * Retrieves a node input value from kStampStatus. 
 *
 * @relates             kStampStatus
 * @param   status      kStampStatus value.
 * @param   index       Node input index. 
 * @return              Extracted value. 
 */
kInlineFx(kBool) kStampStatus_NodeInputState(kStampStatus status, kSize index);

/** 
 * Retrieves a master input value from kStampStatus. 
 *
 * @relates             kStampStatus
 * @param   status      kStampStatus value.
 * @param   index       Master input index. 
 * @return              Extracted value. 
 */
kInlineFx(kBool) kStampStatus_MasterInputState(kStampStatus status, kSize index);

/** 
 * Retrieves a projector state value from kStampStatus. 
 *
 * @relates             kStampStatus
 * @param   status      kStampStatus value.
 * @param   index       Projector index. 
 * @return              Extracted value. 
 */
kInlineFx(kBool) kStampStatus_ProjectorState(kStampStatus status, kSize index);

/** 
 * Retrieves a camera state value from kStampStatus. 
 *
 * @relates             kStampStatus
 * @param   status      kStampStatus value.
 * @param   index       Camera index. 
 * @return              Extracted value. 
 */
kInlineFx(kBool) kStampStatus_CameraState(kStampStatus status, kSize index); 

/** 
 * Retrieves the gate value from kStampStatus. 
 *
 * @relates             kStampStatus
 * @param   status      kStampStatus value.
 * @return              Extracted value. 
 */
kInlineFx(kBool) kStampStatus_GateState(kStampStatus status);

/** 
 * Retrieves the input counter value from kStampStatus. 
 *
 * @relates             kStampStatus
 * @param   status      kStampStatus value.
 * @return              Extracted value. 
 */
kInlineFx(k32u) kStampStatus_InputCounter(kStampStatus status);

/**
 * @struct  kStampSubframe
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents subframe information in the kStamp structure. 
 * 
 * Subframes are used when algorithm results associated with a single camera frame are 
 * split up into multiple messages. In this case, each subframe will have the same frame 
 * index, but will carry additional information identifying the subframe index and whether
 * the subframe represents the final subframe in the frame.
 */
typedef k16u kStampSubframe; 

/** 
 * Creates a subframe value from its constituent parts.
 *
 * @relates             kStampSubframe
 * @param   index       Subframe index.
 * @param   isFinal     Last subframe flag.  
 * @return              Subframe value. 
 */
kInlineFx(kStampSubframe) kStampSubframe_Create(k16u index, kBool isFinal);

/** 
 * Reports the subframe index. 
 *
 * @relates             kStampSubframe
 * @param   subframe    kStampSubframe value.
 * @return              Subframe index. 
 */
kInlineFx(k16u) kStampSubframe_Index(kStampSubframe subframe);

/** 
 * Reports whether this subframe is the last in a frame. 
 *
 * @relates             kStampSubframe
 * @param   subframe    kStampSubframe value.
 * @return              kTRUE if this subframe is the last subframe in frame.  
 */
kInlineFx(kBool) kStampSubframe_IsFinal(kStampSubframe subframe);

/**
 * @struct  kStamp
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents acquisition metadata associated with a data message. 
 */
typedef struct kStamp
{
    k32u signature;             ///< Equal to kSTAMP_SIGNATURE for camera messages; otherwise unused.
    kStampStatus status;        ///< Digital inputs, gate state and other information at time of message capture.
    k8u state;                  ///< Camera state id associated with message. 
    k8u tag;                    ///< Optional tag that can be used for debugging or other special purposes.
    k64u frame;                 ///< Frame index of message data (counts up from zero).
    k64u time;                  ///< Timestamp corresponding to message data (FS us). 
    k64s encoder;               ///< Encoder value corresponding to message data (encoder ticks). 
    k64s encoderSnapshot;       ///< Encoder value at time of most recent encoder snapshot signal (encoder ticks).
    k32u exposure;              ///< Source image exposure (FS ns).
    k32u delay;                 ///< Delay before exposure (FS ns).
    k16u width;                 ///< Source image width, before subsampling.
    k16u height;                ///< Source image height, before subsampling.
    k16u y;                     ///< Source image y position. 
    k16u x;                     ///< Source image x position.
    k8u xSubsampling;           ///< X subsampling amount, expressed as power of 2. 
    k8u ySubsampling;           ///< Y subsampling amount, expressed as power of 2. 
    k16s temperature;           ///< Temperature corresponding to message data (centidegrees Celcius).
    kStampSubframe subframe;    ///< Subframe status information.
    k16u reserved;              ///< Reserved for future use.
} kStamp; 

/**
 * @struct   kTempProbeId
 * @extends  kValue
 * @ingroup  kFireSync
 * @brief    Numeric identifier for kStamp temperature source. 
 */
typedef k32s kTempProbeId;

/** @relates kTempProbeId @{ */
#define kTEMP_PROBE_ID_NULL               (0x00)     ///< Undefined.
#define kTEMP_PROBE_ID_INTERNAL           (0x01)     ///< Internal temperature probe.
#define kTEMP_PROBE_ID_EXTERNAL           (0x02)     ///< External temperature probe.
#define kTEMP_PROBE_ID_PELTIER            (0x04)     ///< Peltier control temperature probe.
#define kTEMP_PROBE_ID_PROJECTOR_LIGHT    (0x08)     ///< Projector light temperature probe.
#define kTEMP_PROBE_ID_CAMERA             (0x10)     ///< Camera temperature probe.
#define kTEMP_PROBE_ID_OPTIONS            (0x1F)     ///< Bitset containing all kTempProbeId options.
/** @} */

/** 
 * Formats a kTempProbeId value to a string buffer. 
 *
 * @public              @memberof kTempProbeId
 * @param   probeId     kTempProbeId value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempProbeId_Format(kTempProbeId probeId, kChar* text, kSize capacity);

/**
 * @struct   kHealthOption
 * @extends  kValue
 * @ingroup  kFireSync-Health
 * @brief    Represents a bitset of health probe options.
 * @see      kHealthProbe
 */
typedef k32s kHealthOption; 

/** @relates kHealthOption @{ */

#define kHEALTH_OPTION_NULL                   (0x0)    ///< Empty set.
#define kHEALTH_OPTION_STREAM                 (0x1)    ///< Send probe via health updates.
#define kHEALTH_OPTION_PERSIST                (0x2)    ///< Log probe to persistent storage (if present).
#define kHEALTH_OPTION_RESTORE                (0x4)    ///< Load persistent value as initial probe value (override default value).
#define kHEALTH_OPTION_BASELINE               (0x8)    ///< Load persistent value as initial probe baseline.
#define kHEALTH_OPTION_CONDITIONALLY_STREAM   (0x10)   ///< Stream the probe, but only if the node also supports persistence.
/** @} */

/**
 * @struct   kHealthId
 * @extends  kValue
 * @ingroup  kFireSync-Health
 * @brief    Numeric identifier for a health statistic. 
 * @see      kHealthProbe, kHealthMsg
 */
typedef k32s kHealthId; 

/** @relates kHealthId @{ */

#define kHEALTH_ID_NULL                                     (0)             ///< Anonymous health stat.
#define kHEALTH_ID_PLATFORM_PUBLIC_BASE                     (1000)          ///< Start of range for public platform health stats.
#define kHEALTH_ID_PLATFORM_PRIVATE_BASE                    (10000)         ///< Start of range for private (undocumented/unofficial) platform health stats.
#define kHEALTH_ID_APPLICATION_BASE                         (20000)         ///< Start of range for application health stats.

#define kHEALTH_ID_ENCODER_VALUE                            (1003)          ///< Current encoder value.
#define kHEALTH_ID_ENCODER_FREQUENCY                        (1005)          ///< Current encoder frequency.
#define kHEALTH_ID_LIGHT_SAFETY                             (1010)          ///< Light safety signal status.
#define kHEALTH_ID_EXTERNAL_POWER                           (1011)          ///< External power and 48V status.
#define kHEALTH_ID_SECONDARY_PL                             (1012)          ///< Loading status of the secondary PL.
#define kHEALTH_ID_SYNC_ERROR_COUNT                         (1100)          ///< Master synchronization error count. 
#define kHEALTH_ID_APPLICATION_VERSION                      (2000)          ///< Application firmware version.
#define kHEALTH_ID_OS_VERSION                               (2001)          ///< Operating System version.
#define kHEALTH_ID_INTERNAL_TEMPERATURE                     (2002)          ///< Controller temperature (millidegrees C). 
#define kHEALTH_ID_MEMORY_USED                              (2003)          ///< Device memory currently in use (bytes). 
#define kHEALTH_ID_MEMORY_CAPACITY                          (2004)          ///< Device memory capacity (bytes). 
#define kHEALTH_ID_STORAGE_VOLUME_USED                      (2005)          ///< Storage volume currently in used (bytes). 
#define kHEALTH_ID_STORAGE_VOLUME_CAPACITY                  (2006)          ///< Storage volume ("user" volume) capacity (bytes). 
#define kHEALTH_ID_CPU_USED                                 (2007)          ///< Total average CPU utilization (percent). 
#define kHEALTH_ID_NET_DATA_OUT_RATE                        (2008)          ///< Current network data output rate (bytes/s). 
#define kHEALTH_ID_NET_INTERFACE_CAPACITY                   (2009)          ///< Maximum network throughput (bytes/s). 
#define kHEALTH_ID_STATE                                    (2010)          ///< Node state. 
#define kHEALTH_ID_PIPE_INPUT_DROPS                         (2015)          ///< Number of messages dropped before data processing.
#define kHEALTH_ID_NET_DATA_OUT_DROPS                       (2016)          ///< Number of messages generated but not sent. 
#define kHEALTH_ID_UPTIME                                   (2017)          ///< Time elapsed since boot-up or reset (seconds). 
#define kHEALTH_ID_DIGITAL_INPUTS                           (2024)          ///< Current status of digital inputs. 
#define kHEALTH_ID_LASER_DRIVER_TEMPERATURE                 (2026)          ///< Laser driver control temperature (millidegrees C).
#define kHEALTH_ID_EXTERNAL_TEMPERATURE                     (2027)          ///< Ambient temperature (millidegrees C). 
#define kHEALTH_ID_CONTROL_TEMPERATURE                      (2028)          ///< Peltier control temperature (millidegrees C). 
#define kHEALTH_ID_CONTROL_OUTPUT                           (2029)          ///< Peltier control output (debug). 
#define kHEALTH_ID_NET_DATA_OUT_BACKLOG_SIZE                (2030)          ///< Current bytes enqueued for transmission.  
#define kHEALTH_ID_NET_DATA_OUT_SENT                        (2031)          ///< Total data bytes tranmitted (bytes). 
#define kHEALTH_ID_NET_DATA_IN_RATE                         (2032)          ///< Current network data input rate (bytes/s). 
#define kHEALTH_ID_NET_DATA_IN_RECEIVED                     (2033)          ///< Total data bytes received. 
#define kHEALTH_ID_NET_INTERFACE_LINK_STATUS                (2034)          ///< Current Ethernet link status (kEthernetLink value). 
#define kHEALTH_ID_BLOCK_DROPS                              (2040)          ///< Number of messages dropped by blocks. 
#define kHEALTH_ID_PIPE_INPUT_BACKLOG_SIZE                  (2041)          ///< Current bytes enqueued for processing.  
#define kHEALTH_ID_RECOVERY_COUNT                           (2042)          ///< Count of acquisition error recovery operations. 
#define kHEALTH_ID_SYNC_SOURCE                              (2043)          ///< Node synchronization source (master-1, sensor-2). 
#define kHEALTH_ID_STORAGE_MEDIUM_CONTROL_ERRORS            (2044)          ///< Storage medium control errors (e.g. timeout). 
#define kHEALTH_ID_STORAGE_MEDIUM_DATA_ERRORS               (2045)          ///< Storage medium data errors (e.g. read data error). 
#define kHEALTH_ID_STORAGE_MEDIUM_OPERATIONAL               (2046)          ///< Is the storage medium currently operational?
#define kHEALTH_ID_STORAGE_VOLUME_POWER_SAFETY              (2047)          ///< Can the volume be corrupted if power is interrupted during modification?
#define kHEALTH_ID_STORAGE_VOLUME_DATA_ERRORS               (2048)          ///< Count of detected errors related to file data. 
#define kHEALTH_ID_STORAGE_VOLUME_METADATA_ERRORS           (2049)          ///< Count of detected errors related to file system metadata.
#define kHEALTH_ID_PIPE_INPUT_BACKLOG_COUNT                 (2050)          ///< Current messages enqueued for processing.  
#define kHEALTH_ID_NET_DATA_OUT_BACKLOG_COUNT               (2051)          ///< Current messages enqueued for transmission.  
#define kHEALTH_ID_CPU_CORE_USED                            (2052)          ///< Individual CPU core utilization (percent). 
#define kHEALTH_ID_MEMORY_LEAKED                            (2053)          ///< Count of memory leaks detected at app restart time (bytes; not available on all platforms). 
#define kHEALTH_ID_CPU_TEMPERATURE                          (2054)          ///< CPU temperature.
#define kHEALTH_ID_MAX_CPU_TEMPERATURE                      (2055)          ///< Maximum CPU temperature since log start.
#define kHEALTH_ID_VOLTAGE_STATUS                           (2056)          ///< Voltage status.
#define kHEALTH_ID_VOLTAGE_HISTORY                          (2057)          ///< Voltage history.
#define kHEALTH_ID_RESTART_REASON                           (2058)          ///< Watchdog restart reason. 
#define kHEALTH_ID_FAN_RPM                                  (2059)          ///< AD board fan RPM.
#define kHEALTH_ID_FAN_PWM                                  (2060)          ///< CPU fan PWM.
#define kHEALTH_ID_FAN_ERROR_FLAGS                          (2061)          ///< AD board fan error flags.
#define kHEALTH_ID_PIPE_EXECUTION_STATUS                    (2062)          ///< Pipe execution status.
#define kHEALTH_ID_CRASH_COUNT                              (2063)          ///< Count of crashes encountered by the device.
#define kHEALTH_ID_LASER_DRIVER_CURRENT                     (2064)          ///< Laser driver current (mA).
#define kHEALTH_ID_LASER_DRIVER_ERROR                       (2065)          ///< Laser driver error code (1 - has error; 0 - no error).
#define kHEALTH_ID_FAN_HAS_ERROR                            (2066)          ///< AD board fan has error flag.
#define kHEALTH_ID_POWER_IN                                 (2067)          ///< Measured input power.
#define kHEALTH_ID_POWER_IN_MIN                             (2068)          ///< Minimum input power.
#define kHEALTH_ID_POWER_IN_MAX                             (2069)          ///< Maximum input power.
#define kHEALTH_ID_POWER_USED                               (2070)          ///< Power used.
#define kHEALTH_ID_VOLTAGE_IN                               (2071)          ///< Measured input voltage.
#define kHEALTH_ID_VOLTAGE_IN_MIN                           (2072)          ///< Minimum input voltage.
#define kHEALTH_ID_VOLTAGE_IN_MAX                           (2073)          ///< Maximum input voltage.
#define kHEALTH_ID_LASER_DRIVER_SET_POWER_ERROR             (2074)          ///< Has a laser driver power-set verification failure occurred since boot?  
#define kHEALTH_ID_LASER_DRIVER_SET_FAILED_COUNT            (2075)          ///< Total count of laser driver power-set verification failures (persistent).  
#define kHEALTH_ID_NET_INTERFACE_RATE_TX                    (2076)          ///< Network output transmission rate (bytes/s)
#define kHEALTH_ID_NET_INTERFACE_RATE_RX                    (2077)          ///< Network input transmission rate (bytes/s)
#define kHEALTH_ID_NET_INTERFACE_BYTES_TX                   (2078)          ///< Total number of bytes sent on network
#define kHEALTH_ID_NET_INTERFACE_BYTES_RX                   (2079)          ///< Total number of bytes received on network
#define kHEALTH_ID_REGULATOR_OUT_VOLTS                      (2080)          ///< Output voltage of regulator (device side)
#define kHEALTH_ID_REGULATOR_OUT_SHUNT_VOLTS                (2081)          ///< Voltage across shunt resistor of power monitor (used for current calculation)
#define kHEALTH_ID_CURRENT_USED                             (2082)          ///< Current out of voltage regulator and used by the device.
#define kHEALTH_ID_LASER_DRIVER_ACTIVE_POWER                (2083)          ///< Laser driver active power value;
#define kHEALTH_ID_UPTIME_TOTAL                             (2084)          ///< Total elapsed uptime since log start (seconds). 
#define kHEALTH_ID_PLATFORM_VERSION                         (2085)          ///< Platform firmware version.
#define kHEALTH_ID_BOOT_COUNT                               (2086)          ///< Total node boot cycles since log start. 
#define kHEALTH_ID_START_TOTAL                              (2087)          ///< Total node acquisition start events since log start. 
#define kHEALTH_ID_RECOVERY_TOTAL                           (2088)          ///< Total acquisition error recovery operations since log start. 
#define kHEALTH_ID_RUN_TIME_TOTAL                           (2089)          ///< Total amount of time spent in acquisition (running) mode since log start (seconds).
#define kHEALTH_ID_HEALTH_LOG_SNAPSHOT                      (2090)          ///< Current health log snapshot index.
#define kHEALTH_ID_STORAGE_MEDIUM_READ_TOTAL                (2091)          ///< Total storage medium read operations since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_WRITE_TOTAL               (2092)          ///< Total storage medium write operations since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_ERASE_TOTAL               (2093)          ///< Total storage medium erase operations since log start. 
#define kHEALTH_ID_PIPE_INPUT_DROP_TOTAL                    (2094)          ///< Total pipe input messages dropped since log start.
#define kHEALTH_ID_PIPE_ERROR_TOTAL                         (2095)          ///< Total pipe execution errors since log start.
#define kHEALTH_ID_LASER_DRIVER_ERROR_TOTAL                 (2096)          ///< Total laser driver errors since log start.
#define kHEALTH_ID_CPU_AVERAGE_USED                         (2097)          ///< Average CPU utilization (percent) over a long period (e.g., 10 seconds).
#define kHEALTH_ID_FAN_ERROR_TOTAL                          (2098)          ///< Total fan errors since start log start. 
#define kHEALTH_ID_SYNC_ERROR_TOTAL                         (2099)          ///< Total master synchronization errors since log start.

/* Note: top-level node health ids continue in 30xx range, below*/

#define kHEALTH_ID_EVENT_QUEUE_COUNT                        (2100)          ///< Current count of event requests in FIFO. 
#define kHEALTH_ID_EVENT_OVERFLOW_COUNT                     (2101)          ///< Count of event request FIFO overflow errors. 
#define kHEALTH_ID_EVENT_COUNT                              (2102)          ///< Count of individual events generated. 
#define kHEALTH_ID_EVENT_SEQUENCE_DROP_COUNT                (2103)          ///< Count of event sequences dropped. 
#define kHEALTH_ID_EVENT_DROP_COUNT                         (2104)          ///< Count of individual events dropped. 
#define kHEALTH_ID_EVENT_OVERFLOW_TOTAL                     (2105)          ///< Total event FIFO overflows since log start. 
#define kHEALTH_ID_EVENT_DROP_TOTAL                         (2106)          ///< Total event (or event sequence) drops since log start.

#define kHEALTH_ID_CAMERA_EXPOSURE_COUNT                    (2200)          ///< Camera exposure count. 
#define kHEALTH_ID_CAMERA_EXPOSURE_DROPS                    (2201)          ///< Camera exposure drop count.
#define kHEALTH_ID_CAMERA_BUFFER_DROPS                      (2203)          ///< Count of camera frames dropped due to insufficient frame buffer capacity.
#define kHEALTH_ID_CAMERA_ALLOC_DROPS                       (2204)          ///< Count of camera frames dropped due to lack of dynamic memory.
#define kHEALTH_ID_CAMERA_INTERRUPT_COUNT                   (2205)          ///< Count of camera interrupts.
#define kHEALTH_ID_CAMERA_CONTROL_ERROR_COUNT               (2206)          ///< Count of camera sequencing/control errors. 
#define kHEALTH_ID_CAMERA_IMAGER_DESYNC_COUNT               (2207)          ///< Count of camera communication desync errors. 
#define kHEALTH_ID_CAMERA_STREAM_DESYNC_COUNT               (2208)          ///< Count of camera stream desync errors. 
#define kHEALTH_ID_CAMERA_PROCESSING_ERROR_COUNT            (2209)          ///< Count of camera stream processing errors. 
#define kHEALTH_ID_CAMERA_ERROR_FLAGS                       (2210)          ///< Bitset representing driver-specific camera error information from current acquisition cycle.
#define kHEALTH_ID_CAMERA_RATE                              (2211)          ///< Current frame rate (Hz). 
#define kHEALTH_ID_CAMERA_PROCESSED_FRAME_COUNT             (2212)          ///< Count of frames that have been received with valid stamps.
#define kHEALTH_ID_CAMERA_INPUT_OVERFLOW_COUNT              (2213)          ///< Count of input buffer (video port) overflows. 
#define kHEALTH_ID_CAMERA_BYTES_RECEIVED                    (2214)          ///< Count of data bytes received from sender. 
#define kHEALTH_ID_CAMERA_SYNC_ALIGNMENT_ERROR_COUNT        (2215)          ///< Count of imager sync channel alignment errors. 
#define kHEALTH_ID_CAMERA_PROCESSING_BACKLOG                (2216)          ///< Current bytes received but not yet processed into pipe messages.  
#define kHEALTH_ID_CAMERA_SEARCH_FRAME_COUNT                (2217)          ///< Count of search frames when tracking is enabled.
#define kHEALTH_ID_CAMERA_DATA_ALIGNMENT_ERROR_COUNT        (2218)          ///< Count of overal imager data channel alignment errors. 
#define kHEALTH_ID_CAMERA_ERROR_FLAGS_PREVIOUS              (2220)          ///< Bitset representing driver-specific camera error information from previous acquisition cycle.
#define kHEALTH_ID_CAMERA_TEMPERATURE                       (2221)          ///< Camera board temperature (millidegrees C).
#define kHEALTH_ID_CAMERA_HAS_ERROR                         (2222)          ///< Camera top level error indicator.
#define kHEALTH_ID_CAMERA_PL_WRITE_DMA_STATUS               (2223)          ///< PL write DMA status
#define kHEALTH_ID_CAMERA_PL_READ_DMA_STATUS                (2224)          ///< PL read DMA status
#define kHEALTH_ID_CAMERA_CHANNEL_ALIGNMENT_ERROR_COUNT_0   (2225)          ///< Count of each imager data channel alignment errors. 
#define kHEALTH_ID_CAMERA_CHANNEL_ALIGNMENT_ERROR_COUNT_1   (2226)          ///< Count of each imager data channel alignment errors. 
#define kHEALTH_ID_CAMERA_CHANNEL_ALIGNMENT_ERROR_COUNT_2   (2227)          ///< Count of each imager data channel alignment errors. 
#define kHEALTH_ID_CAMERA_CHANNEL_ALIGNMENT_ERROR_COUNT_3   (2228)          ///< Count of each imager data channel alignment errors. 
#define kHEALTH_ID_CAMERA_ALLOC_DROP_TOTAL                  (2229)          ///< Total camera alloc drops since log start.
#define kHEALTH_ID_CAMERA_BUFFER_DROP_TOTAL                 (2230)          ///< Total camera buffer drops since log start.
#define kHEALTH_ID_CAMERA_CHANNEL_ALIGN_ERROR_TOTAL         (2231)          ///< Total camera channel alignment errors since log start.
#define kHEALTH_ID_CAMERA_CONTROL_ERROR_TOTAL               (2232)          ///< Total camera control errors since log start.
#define kHEALTH_ID_CAMERA_DATA_ALIGN_ERROR_TOTAL            (2233)          ///< Total camera data alignment errors since log start.
#define kHEALTH_ID_CAMERA_ERROR_FLAGS_ACCUM                 (2234)          ///< Bitset representing driver-specific camera error information accumulated since boot.
#define kHEALTH_ID_CAMERA_EXPOSURE_COUNT_TOTAL              (2235)          ///< Total camera exposures since log start.
#define kHEALTH_ID_CAMERA_EXPOSURE_DROP_TOTAL               (2236)          ///< Total camera exposure drops since log start.
#define kHEALTH_ID_CAMERA_IMAGER_DESYNC_TOTAL               (2237)          ///< Total imager desyncs since log start.
#define kHEALTH_ID_CAMERA_INPUT_OVERFLOW_TOTAL              (2238)          ///< Total camera PL input overflows since log start.
#define kHEALTH_ID_CAMERA_PL_READ_DMA_ERROR_ACCUM           (2239)          ///< Accumlated camera PL DMA read error flags since boot (bitset).
#define kHEALTH_ID_CAMERA_PL_WRITE_DMA_ERROR_ACCUM          (2240)          ///< Accumlated camera PL DMA write error flags since boot (bitset).
#define kHEALTH_ID_CAMERA_PROCESSING_ERROR_TOTAL            (2241)          ///< Total camera processing errors since log start.
#define kHEALTH_ID_CAMERA_STREAM_DESYNC_TOTAL               (2242)          ///< Total camera stream desyncs since log start.
#define kHEALTH_ID_CAMERA_SYNC_ALIGN_ERROR_TOTAL            (2243)          ///< Total camera sync alignment errors since log start.
#define kHEALTH_ID_CAMERA_EXPOSURE_TIME_TOTAL               (2244)          ///< Total camera exposure time since log start (seconds).

#define kHEALTH_ID_LIGHT_EXPOSURE_COUNT                     (2300)          ///< Light exposure count.
#define kHEALTH_ID_LIGHT_EXPOSURE_DROP_COUNT                (2301)          ///< Light exposure drop count.
#define kHEALTH_ID_LIGHT_INTERRUPT_COUNT                    (2302)          ///< Count of light interrupts.
#define kHEALTH_ID_LIGHT_HAS_ERROR                          (2303)          ///< Light top level error indicator.
#define kHEALTH_ID_LIGHT_EXPOSURE_COUNT_TOTAL               (2304)          ///< Total light exposures since log start.
#define kHEALTH_ID_LIGHT_EXPOSURE_DROP_TOTAL                (2305)          ///< Total light exposure drops since log start.
#define kHEALTH_ID_LIGHT_EXPOSURE_TIME_TOTAL                (2306)          ///< Total light exposure time since log start (seconds).

#define kHEALTH_ID_PROJECTOR_EXPOSURE_COUNT                 (2400)          ///< Projector exposure count.
#define kHEALTH_ID_PROJECTOR_EXPOSURE_DROP_COUNT            (2401)          ///< Projector exposure drop count.
#define kHEALTH_ID_PROJECTOR_SOE_INTERRUPT_COUNT            (2402)          ///< Count of projector SOE interrupts.
#define kHEALTH_ID_PROJECTOR_EOE_INTERRUPT_COUNT            (2403)          ///< Count of projector EOE interrupts.
#define kHEALTH_ID_PROJECTOR_LIGHT_TEMPERATURE              (2404)          ///< Projector illumination temperature (millidegrees C). 
#define kHEALTH_ID_PROJECTOR_PANEL_TEMPERATURE              (2405)          ///< Projector filter array temperature (millidegrees C).
#define kHEALTH_ID_PROJECTOR_CONTROL_ERROR_COUNT            (2406)          ///< Count of projector control errors (eg. PL communication errors).
#define kHEALTH_ID_PROJECTOR_DISPLAY_ERROR_COUNT            (2407)          ///< Count of projector display errors (eg. LCOS/DLP communication errors).
#define kHEALTH_ID_PROJECTOR_OSCILLATOR_CALIBRATED          (2408)          ///< Projector oscillator calibration status.
#define kHEALTH_ID_PROJECTOR_HAS_ERROR                      (2409)          ///< Projector top level error indicator.
#define kHEALTH_ID_PROJECTOR_ERROR_FLAGS                    (2410)          ///< Projector error flags.
#define kHEALTH_ID_PROJECTOR_SEQUENCE_DROP_COUNT            (2411)          ///< Projector sequence drop count.
#define kHEALTH_ID_PROJECTOR_CONTROL_ERROR_TOTAL            (2412)          ///< Total projector control errors since log start. 
#define kHEALTH_ID_PROJECTOR_DISPLAY_ERROR_TOTAL            (2413)          ///< Total projector display errors since log start. 
#define kHEALTH_ID_PROJECTOR_EXPOSURE_COUNT_TOTAL           (2414)          ///< Total projector exposures since log start.
#define kHEALTH_ID_PROJECTOR_EXPOSURE_DROP_TOTAL            (2415)          ///< Total projector exposure drops since log start.
#define kHEALTH_ID_PROJECTOR_EXPOSURE_TIME_TOTAL            (2416)          ///< Total projector exposure time since log start (seconds).
#define kHEALTH_ID_PROJECTOR_SEQUENCE_DROP_TOTAL            (2417)          ///< Total projector sequence drops since log start.
#define kHEALTH_ID_PROJECTOR_MISSED_INTERRUPT_COUNT         (2418)          ///< Count of missed projector interrupts.
#define kHEALTH_ID_PROJECTOR_MISSED_INTERRUPT_TOTAL         (2419)          ///< Total projector missed interrupts.
#define kHEALTH_ID_PROJECTOR_OVERHEAT_COUNT                 (2420)          ///< Projector overheats.
#define kHEALTH_ID_PROJECTOR_OVERHEAT_COUNT_TOTAL           (2421)          ///< Total projector overheats since log start.
#define kHEALTH_ID_PROJECTOR_EFFECTIVE_INTENSITY            (2422)          ///< Projector effective intensity.

#define kHEALTH_ID_ANALOG_OUTPUT_COUNT                      (2500)          ///< Analog output count.
#define kHEALTH_ID_ANALOG_OUTPUT_DROP_COUNT                 (2501)          ///< Analog output drop count.
#define kHEALTH_ID_ANALOG_QUEUE_COUNT                       (2502)          ///< Analog output queue current count.
#define kHEALTH_ID_ANALOG_OVERFLOW_COUNT                    (2503)          ///< Analog output queue overflow count.
#define kHEALTH_ID_ANALOG_UNDERFLOW_COUNT                   (2504)          ///< Analog output queue underflow count.
#define kHEALTH_ID_ANALOG_HAS_ERROR                         (2505)          ///< Analog output top level error indicator.
#define kHEALTH_ID_ANALOG_OUTPUT_DROP_TOTAL                 (2506)          ///< Total analog output drops since log start.
#define kHEALTH_ID_ANALOG_OVERFLOW_TOTAL                    (2507)          ///< Total analog overflows since log start.
#define kHEALTH_ID_ANALOG_UNDERFLOW_TOTAL                   (2508)          ///< Total analog underflows since log start.

#define kHEALTH_ID_DIGITAL_OUTPUT_COUNT                     (2600)          ///< Digital output count.
#define kHEALTH_ID_DIGITAL_OUTPUT_DROP_COUNT                (2601)          ///< Digital output drop count.
#define kHEALTH_ID_DIGITAL_OUTPUT_HAS_ERROR                 (2602)          ///< Digital output top level error indicator.
#define kHEALTH_ID_DIGITAL_OUTPUT_DROP_TOTAL                (2603)          ///< Total digital output drops since log start.

#define kHEALTH_ID_SERIAL_OUTPUT_COUNT                      (2700)          ///< Count of output words transmitted. 
#define kHEALTH_ID_SERIAL_OUTPUT_DROP_COUNT                 (2701)          ///< Count of output words dropped. 
#define kHEALTH_ID_SERIAL_QUEUE_COUNT                       (2702)          ///< Count of output words enqueued. 
#define kHEALTH_ID_SERIAL_UNDERFLOW_COUNT                   (2703)          ///< Count of queue underflow events detected.
#define kHEALTH_ID_SERIAL_OVERFLOW_COUNT                    (2704)          ///< Count of queue overflow events detected.
#define kHEALTH_ID_SERIAL_HAS_ERROR                         (2705)          ///< Serial top level error indicator.
#define kHEALTH_ID_SERIAL_OUTPUT_DROP_TOTAL                 (2706)          ///< Total serial output words dropped since log start. 
#define kHEALTH_ID_SERIAL_OVERFLOW_TOTAL                    (2707)          ///< Total serial overflows since log start.
#define kHEALTH_ID_SERIAL_UNDERFLOW_TOTAL                   (2708)          ///< Total serial underflows since log start.

#define kHEALTH_ID_IO_TEST_POWER_5V                         (2800)          ///< Voltage reading of 5V (millivolt). 
#define kHEALTH_ID_IO_TEST_INPUT_POWER                      (2801)          ///< Voltage reading of input power (millivolt).
#define kHEALTH_ID_IO_TEST_SAFETY_5V                        (2802)          ///< Voltage reading of SAFETY_5V (millivolt). 
#define kHEALTH_ID_IO_TEST_SAFETY_PLUS                      (2803)          ///< Voltage reading of SAFETY+ (millivolt). 
#define kHEALTH_ID_IO_TEST_ANALOG_INPUT_0_CALIBRATED        (2804)          ///< Is analog input 0 calibrated. 
#define kHEALTH_ID_IO_TEST_ANALOG_IN_0                      (2805)          ///< Current reading of analog output 0 (milliamp). 
#define kHEALTH_ID_IO_TEST_ANALOG_INPUT_1_CALIBRATED        (2806)          ///< Is analog input 1 calibrated. 
#define kHEALTH_ID_IO_TEST_ANALOG_IN_1                      (2807)          ///< Current reading of analog output 1 (milliamp). 
#define kHEALTH_ID_IO_TEST_DIGITAL_IN                       (2808)          ///< Digital input status. 
#define kHEALTH_ID_IO_TEST_UART_FIFO_STATUS                 (2809)          ///< UART FIFO status (full, empty and overflowed)? 
#define kHEALTH_ID_IO_TEST_UART_DATA_COUNT                  (2810)          ///< Count of UART data available. 
#define kHEALTH_ID_IO_TEST_LAST_UART_DATA                   (2811)          ///< Last read UART data. 
#define kHEALTH_ID_IO_TEST_SELCOM_FIFO_STATUS               (2812)          ///< Selcom FIFO status (full, empty and overflowed)? 
#define kHEALTH_ID_IO_TEST_SELCOM_DATA_COUNT                (2813)          ///< Count of Selcom data available. 
#define kHEALTH_ID_IO_TEST_LAST_SELCOM_DATA                 (2814)          ///< Last read Selcom data. 

#define kHEALTH_ID_ORIENTATION_TEMP                         (2900)          ///< Orientation temperature (degree Celsius).
#define kHEALTH_ID_ORIENTATION_CONTROL_ERROR_COUNT          (2901)          ///< Count of I2C transaction failures.
#define kHEALTH_ID_ORIENTATION_SYS_CALIB_STAT               (2902)          ///< Current system calibration status.
#define kHEALTH_ID_ORIENTATION_GYR_CALIB_STAT               (2903)          ///< Current calibration status of Gyroscope.
#define kHEALTH_ID_ORIENTATION_ACC_CALIB_STAT               (2904)          ///< Current calibration status of Accelerometer,.
#define kHEALTH_ID_ORIENTATION_MAG_CALIB_STAT               (2905)          ///< Current calibration status of Magnetometer,,.
#define kHEALTH_ID_ORIENTATION_STATIONARY                   (2906)          ///< Is orientation device stationary?
#define kHEALTH_ID_ORIENTATION_X                            (2907)          ///< Orientation device's x (pitch) angle (millidegree).
#define kHEALTH_ID_ORIENTATION_Y                            (2908)          ///< Orientation device's y (roll) angle (millidegree).
#define kHEALTH_ID_ORIENTATION_Z                            (2909)          ///< Orientation device's z (yaw) angle (millidegree).
#define kHEALTH_ID_ORIENTATION_SYSTEM_STATUS                (2920)          ///< Orientation device's system status.
#define kHEALTH_ID_ORIENTATION_SYSTEM_ERROR                 (2911)          ///< Orientation device's system error.

#define kHEALTH_ID_NET_DATA_OUT_DROP_TOTAL                  (3000)          ///< Total network data output messages dropped since log start. 
#define kHEALTH_ID_STORAGE_VOLUME_WRITE_TOTAL               (3001)          ///< Total storage volume bytes written since log start. 
#define kHEALTH_ID_PL_CONFIG                                (3002)          ///< PL configuration ID (primary: instance 0, secondary: instance 1).
#define kHEALTH_ID_PL_VERSION                               (3003)          ///< PL configuration version (primary: instance 0, secondary: instance 1). 
#define kHEALTH_ID_DATE_TIME                                (3004)          ///< Current date and time, in seconds, relative to Unix epoch (00:00:00 Jan 1, 1970 CE, UTC); zero if unknown.
#define kHEALTH_ID_MAX_INTERNAL_TEMPERATURE                 (3005)          ///< Maximum controller temperature since log start (millidegrees C). 
#define kHEALTH_ID_WATCHDOG_RESETS                          (3006)          ///< Watchdog reset count.
#define kHEALTH_ID_CUDA_STATUS                              (3007)          ///< Reports the overall status of the Cuda processing environment, as detected at node startup time (kCudaEnvironmentStatus).

#define kHEALTH_ID_STORAGE_MEDIUM_READ_FAILED_TOTAL         (3110)          ///< Total storage medium failed read operations since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_READ_FIXABLE_TOTAL        (3111)          ///< Total storage medium read operations with fixable corrections since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_READ_IGNORABLE_TOTAL      (3112)          ///< Total storage medium read operations with ignorable corrections since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_WRITE_FAILED_TOTAL        (3113)          ///< Total storage medium failed write operations since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_ERASE_FAILED_TOTAL        (3114)          ///< Total storage medium failed erase operations since log start. 
#define kHEALTH_ID_STORAGE_MEDIUM_BAD_BLOCKS                (3115)          ///< Storage medium number of bad blocks. 

/** @} */

/**
 * @struct   kHealthStat
 * @extends  kValue
 * @ingroup  kFireSync-Health
 * @brief    Represents the state of a health probe.
 * @see      kHealthProbe, kHealthMsg, kHealthId
 */
typedef struct kHealthStat
{
    kText64 name;           ///< Descriptive label (e.g. "Event/0/QueueCount").
    kHealthId id;           ///< Health stat identifier (or kHEALTH_ID_NULL). 
    k32u instance;          ///< Distinguishes multiple health stats with same id.
    k64s value;             ///< Health stat value (units are stat-specific). 
} kHealthStat; 

/**
 * @struct   kProfileStat
 * @extends  kValue
 * @ingroup  kFireSync-Health
 * @brief    Represents the state of a code profiling probe.
 * @see      kProfileProbe
 */
typedef struct kProfileStat
{
    kText64 name;           ///< Descriptive label (e.g. "Camera/0/ExposureCount").
    k64u count;             ///< Invocation count. 
    k64u sum;               ///< Sum of execution times (us).
    k32u value;             ///< Most recent execution time (us). 
    k32u min;               ///< Minimum execution time (us). 
    k32u max;               ///< Maximum execution time (us).
} kProfileStat; 

/**
 * @struct   kLogItem
 * @extends  kValue
 * @ingroup  kFireSync
 * @brief    Represents a log message.
 */
typedef struct kLogItem
{
    k64u id;                ///< Unique identifier.
    kLogOption options;     ///< Log entry options.
    k64u upTime;            ///< Device uptime at which log messages was generated.
    kDateTime dateTime;     ///< Calendar date-time at which log message was generated.
    kText64 sender;         ///< Sender label.
    kText128 text;          ///< Log text.
} kLogItem; 

/**
 * @struct  kInputEventSource
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents input event source.
 */
typedef k32s kInputEventSource; 

/** @relates kInputEventSource @{ */
#define kINPUT_EVENT_SOURCE_PL_GPIO         (1)      ///< Pl GPIO. 
/** @} */

/** 
 * Formats a kInputEventSource value to a string buffer. 
 *
 * @public              @memberof kInputEventSource
 * @param   source      kInputEventSource value. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kInputEventSource_Format(kInputEventSource source, kChar* text, kSize capacity);

/**
 * @struct   kInputEvent
 * @extends  kValue
 * @ingroup  kFireSync
 * @brief    Represents an input event.
 */
typedef struct kInputEvent
{
    kInputEventSource source;   ///< Event source.
    k64s instance;              ///< Event source index -- used when multiple sources of the same type exist in the system.
    k64s state;                 ///< Input state causing the event.
    k64u time;                  ///< Time stamp.
} kInputEvent; 

#define kNODE_BOOT_TARBALL_MANIFEST_FILE_NAME           "Manifest.xml"          ///< Bootable tarball manifest component name.
#define kNODE_BOOT_TARBALL_FRAMEWORK_FILE_NAME          "Framework.bin"         ///< Bootable tarball framework component name.
#define kNODE_BOOT_TARBALL_APPLICATION_FILE_NAME        "Application.bin"       ///< Bootable tarball application component name.

/** 
 * Converts from FireSync time units to real time units.  
 *
 * @param   FS_TIME     FireSync time units. 
 * @return              Real time units. 
 */
#define kFsTimeToTime_(FS_TIME)         kxFsTimeToTime_(FS_TIME)

/** 
 * Converts from real time units to FireSync time units.  
 *
 * @param   TIME     Real time units. 
 * @return           FireSync time units. 
 */
#define kTimeToFsTime_(TIME)            kxTimeToFsTime_(TIME) 

/** 
 * Callback signature used with the kSystem_SetNodeFilter function.
 *
 * If the callback does not return kOK, the node will be ignored during discovery.
 * 
 * @param   receiver  Context pointer for callback. 
 * @param   sender    kSystem object that discovered the node.
 * @param   nodeId    Node id.
 * @return            kOK to accept the node; otherwise the node will be ignored.
 */
typedef kStatus (kCall* kNodeFilterFx)(kPointer receiver, kObject sender, k32u nodeId); 

/**
 * @struct   kStorageEntry
 * @extends  kValue
 * @ingroup  kFireSync
 * @brief    Represents the type of storage entry.
 */
typedef k32s kStorageEntry;

/** @relates kStorageEntry @{ */
#define kSTORAGE_ENTRY_FILE         (0)     ///< Regular file.
#define kSTORAGE_ENTRY_DIRECTORY    (1)     ///< Directory.
/** @} */

#include <kFireSync/kNodeDef.x.h>

#endif
