/** 
 * @file    kTar.x.h
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.
 */
#ifndef K_FIRESYNC_TAR_FILE_X_H
#define K_FIRESYNC_TAR_FILE_X_H

#include <kApi/Io/kStream.h>

 /* https://www.gnu.org/software/tar/manual/html_node/Blocking.html#SEC158  */
#define xkTAR_BLOCK_SIZE                    (512)
#define xkTAR_RECORD_SIZE                   (20 * xkTAR_BLOCK_SIZE)

#define xkTAR_NAME_OFFSET                   (0)
#define xkTAR_NAME_SIZE                     (100)
#define xkTAR_SIZE_OFFSET                   (124)
#define xkTAR_SIZE_SIZE                     (12)
#define xkTAR_CHKSUM_OFFSET                 (148)
#define xkTAR_CHKSUM_DIGITS                 (6)
#define xkTAR_CHKSUM_SIZE                   (8)
#define xkTAR_TYPE_OFFSET                   (156)
#define xkTAR_SIGNATURE_OFFSET              (257)
#define xkTAR_VERSION_OFFSET                (263)
#define xkTAR_VERSION_SIZE                  (2)
#define xkTAR_PREFIX_OFFSET                 (345)
#define xkTAR_PREFIX_SIZE                   (155)

#define xkTAR_ITEM_UNKNOWN                  (0)          ///< Unknown format.
#define xkTAR_ITEM_REGTYPE                  ('0')        ///< Regular file.
#define xkTAR_ITEM_DIRTYPE                  ('5')        ///< Directory.
#define xkTAR_ITEM_PAXTYPE                  ('x')        ///< Extended entry.

/* https://www.gnu.org/software/tar/manual/html_node/Formats.html */
#define xkTAR_MAX_NAME_LENGTH_USTAR         (100)        ///< Maximum file name length for ustar format when using name field only.
#define xkTAR_MAX_FILE_SIZE_USTAR           (8388608)    ///< Maximum file size for ustar format.

#define xkTAR_FILE_BUFFER_SIZE              (16384)

typedef struct kTarItem
{
    kChar name[kPATH_MAX];
    k64u start;
    k64u length;
    k64u pos;
    k8u fileType;
} kTarItem;

typedef struct kTarClass
{
    kStreamClass base;
    
    kStream stream;                     //< Stream object.
    kTarMode mode;                      //< Operation mode.

    kTarItem currentItem;               //< Current item, to track about current item status.

    kByte* buffer;                      //< Temporary buffer (2*xkTAR_BLOCK_SIZE = 1k)

} kTarClass;

kDeclareClassEx(kFs, kTar, kStream)
kDeclareEnumEx(kFs, kTarMode, kValue)

/*
* Private methods.
*/

kFsFx(kStatus) xkTar_Init(kTar tar, kStream destination, kTarMode mode, kType type, kAlloc alloc);
kFsFx(kStatus) xkTar_VRelease(kTar tar);

kFsFx(kStatus) xkTar_VReadSomeImpl(kTar tar, void* buffer, kSize minCount, kSize maxCount, kSize* bytesRead);
kFsFx(kStatus) xkTar_VWriteImpl(kTar tar, const void* buffer, kSize size);
kFsFx(kStatus) xkTar_VFlush(kTar tar);

kFsFx(kStatus) xkTar_FinishCurrentItem(kTar tar);
kFsFx(kStatus) xkTar_BeginNextUstarItem(kTar tar, const kChar* itemName, k64u itemSize, k8u fileType);
kFsFx(kStatus) xkTar_PrepareNextPaxItem(kTar tar, const kChar* itemName, k64u itemSize, k8u fileType);

kFsFx(kStatus) xkTar_FormatOctal(kSize val, kByte* dest, kSize size);
kFsFx(kStatus) xkTar_FormatPaxFileName(kString dest, const kChar* fileName);
kFsFx(kStatus) xkTar_FormatPaxFileContent(kString dest, const kChar* fileName, k64u size);
kFsFx(kStatus) xkTar_FinalFormatPaxEntry(kString entry);
kFsFx(kStatus) xkTar_FormatUstarHeader(kByte* buffer, kSize bufferSize, const kChar* itemName, kSize itemSize, k8u fileType);

kFsFx(kStatus) xkTar_ReadPaxHeader(kStream stream, kTarItem* item);
kFsFx(kStatus) xkTar_InterpretPaxHeader(kStream stream, const kTarItem* item, kTarItem* paxInfo);
kFsFx(kStatus) xkTar_SplitPaxHeader(const kChar* buffer, kSize size, kMap headerMap);

kFsFx(kStatus) xkTar_SeekNext(kStream stream, const kTarItem* item, kTarItem* nextItem);

kFsFx(kStatus) xkTar_VerifyChecksum(const kByte* buffer);

kFsFx(kStatus) xkTar_ListEntriesDeep(const kChar* baseName, const kChar* sourceDirectory, kMap entries);

kFsFx(kStatus) xkTar_WriteToStream(kTar tar, const void* buffer, kSize size);

kFsFx(kStatus) xkTar_ReadTarName(const kByte* buffer, kChar* name, kSize capacity);
kFsFx(kStatus) xkTar_FormatTarName(const kChar* path, kChar* tarEntryName, kSize capacity, kBool isDirectory);

kFsFx(kStatus) xkTar_Archive(const kChar* sourcePath, kStream destStream);
kFsFx(kStatus) xkTar_Extract(kStream source, const kChar* destParent);

kFsFx(kStatus) xkTar_WriteFile(kTar tar, const kChar* itemPath, kStream source, k64u size);

kFsFx(kStatus) xkTar_WriteDirectory(kTar tar, const kChar* itemPath, const kChar* directoryPath);

#endif
