#ifndef K_LD_CAL_H
#define K_LD_CAL_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/kNodeDef.h>

/**
* @class       kLdCal
* @extends     kObject
* @ingroup     kFireSync-Utils
* @brief       Class cal for the kLdCal class.
*/
//typedef kObject kLdCal;            --forward-declared in kFsDef.x.h  

/**
* Generates the default calibration file name based on the index of the laser driver.
*
* @public               @memberof kLdCal
* @param   index        Index of the laser driver corresponding to this record
* @param   fileName     Text buffer receiving the file name
* @param   capacity     Capacity of the output text buffer
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_FileName(kSize index, kChar* fileName, kSize capacity);

/**
* Generates the backup calibration file name based on the index of the laser driver.
*
* If the default calibration file is missing, the backup calibration file will be checked.
* The original file may have been removed to address a problem in the field (see FSS-1834).
*
* @public               @memberof kLdCal
* @param   index        Index of the laser driver corresponding to this record
* @param   fileName     Text buffer receiving the file name
* @param   capacity     Capacity of the output text buffer
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_BackupFileName(kSize index, kChar* fileName, kSize capacity);

/**
* Constructs a kLdCal cal.
*
* @public               @memberof kLdCal
* @param   cal          Destination for the constructed cal handle.
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_Construct(kLdCal* cal, kAlloc alloc);

/**
* Saves the calibration record to local storage under the specified path.
*
* @public               @memberof kLdCal
* @param   cal          The class object.
* @param   path         The file location to store the resulting output.
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_Save(kLdCal cal, const kChar *path);

/**
* Loads the calibration record from local storage.
*
* @public               @memberof kLdCal
* @param   cal          The class object.
* @param   path         The file location to load the calibration record.
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_Load(kLdCal *cal, const kChar *path, kAlloc alloc);

/**
* Saves the calibration record to a stream.
*
* @public               @memberof kLdCal
* @param   cal          The class object.
* @param   stream       Output stream handle
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_SaveStream(kLdCal cal, kStream stream);

/**
* Loads the calibration record from a stream.
*
* @public               @memberof kLdCal
* @param   cal          Pointer receiving the read calibration record
* @param   stream       Input stream handle
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kFsFx(kStatus) kLdCal_LoadStream(kLdCal *cal, kStream stream, kAlloc alloc);

/**
* Sets the sensor serial number.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   sensorId      The sensorId
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetSensorId(kLdCal cal, kSize sensorId);

/**
* Gets the serial number.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                The sensor ID
*/
kFsFx(kSize) kLdCal_SensorId(kLdCal cal);

/**
* Sets the calibration timestamp.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   timestamp     Timestamp
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetTimestamp(kLdCal cal, const kChar* timestamp);

/**
* Gets the calibration timestamp.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Timestamp
*/
kFsFx(const kChar*) kLdCal_Timestamp(kLdCal cal);

/**
* Sets the calibration station Id (e.g. 'T008').
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   stationId     Station id
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetStationId(kLdCal cal, const kChar* stationId);

/**
* Gets the calibration station Id.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Station id
*/
kFsFx(const kChar*) kLdCal_StationId(kLdCal cal);

/**
* Sets the light driver index. 
* 
* This is the channel used to access the driver via functions such as kNodeDev_AdjustLightDriverPower
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   index         Driver index
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetDriverIndex(kLdCal cal, kSize index);

/**
* Gets the light driver index.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Driver index
*/
kFsFx(kSize) kLdCal_DriverIndex(kLdCal cal);

/**
* Sets the driver model. 
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   model         Driver model
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetDriverModel(kLdCal cal, kLightModel model);

/**
* Queries driver model. 
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Driver model
*/
kFsFx(kLightModel) kLdCal_DriverModel(kLdCal cal);

/**
* Sets the diode model. 
*
* @public                @memberof kLdCal
* @param   cal           kLdCal object.
* @param   model         Diode model.
* @return                Operation status.
*/
kFsFx(kStatus) kLdCal_SetDiodeModel(kLdCal cal, kLightDiodeModel model);

/**
* Queries diode model. 
*
* @public                @memberof kLdCal
* @param   cal           kLdCal object.
* @return                Diode model.
*/
kFsFx(kLightDiodeModel) kLdCal_DiodeModel(kLdCal cal);

/**
* Sets the driver version.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   version       Driver model
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetDriverVersion(kLdCal cal, kVersion version);

/**
* Gets the driver version.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Driver version
*/
kFsFx(kVersion) kLdCal_DriverVersion(kLdCal cal);

/**
* Sets the driver temperature (degrees Celsius).
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   temperature   Driver temperature (degrees Celsius)
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetDriverTemperature(kLdCal cal, k64f temperature);

/**
* Gets the driver temperature.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Driver temperature
*/
kFsFx(k64f) kLdCal_DriverTemperature(kLdCal cal);

/**
* Sets the driver serial number.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   driverId      Driver serial number
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetDriverId(kLdCal cal, kSize driverId);

/**
* Gets the driver serial number.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Driver serial number
*/
kFsFx(kSize) kLdCal_DriverId(kLdCal cal);

/**
* Sets the formatted laser driver id. 
* 
* The field represents the serial number formatted for display,
* unlike that set by kLdCal_SetDriverId, which matches the value stored in the driver exactly
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   label         Formatted driver serial number
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetDriverIdLabel(kLdCal cal, const kChar* label);

/**
* Gets the formatted driver serial number
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Driver serial number
*/
kFsFx(const kChar*) kLdCal_DriverIdLabel(kLdCal cal);

/**
* Sets the power set point input (in device units) programmed to the laser driver.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   input         Power set point input programmed to the laser driver. kSIZE_NULL for none.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetPowerInputDevice(kLdCal cal, kSize input);

/**
* Gets the power set point input (in device units) programmed to the laser driver. 
* 
* kSIZE_NULL is returned if this input is not not set/used.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Power set point input (in device units) programmed to the laser driver.
*/
kFsFx(kSize) kLdCal_PowerInputDevice(kLdCal cal);

/**
* Sets the current limit input (in device units) programmed to the laser driver.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   input         Current limit input (in device units) programmed to the laser driver. kSIZE_NULL for none.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetLimitInputDevice(kLdCal cal, kSize input);

/**
* Gets the current limit input (in device units) programmed to the laser driver. 
* 
* kSIZE_NULL is returned if this input is not not set/used.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Current limit input (in device units) programmed to the laser driver.
*/
kFsFx(kSize) kLdCal_LimitInputDevice(kLdCal cal);

/**
* Sets the power set point input (in device units) to be used at device startup.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   input         Power set point input (in device units) to be used at device startup. kSIZE_NULL for none.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetPowerInputDefault(kLdCal cal, kSize input);

/**
* Gets the power set point input (in device units) to be used at device startup. 
* 
* kSIZE_NULL is returned if this input is not not set/used.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Power set point input (in device units) to be used at device startup.
*/
kFsFx(kSize) kLdCal_PowerInputDefault(kLdCal cal);

/**
* Sets the current limit input (in device units) to be used at device startup.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   input         Current limit input (in device units) to be used at device startup. kSIZE_NULL for none.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetLimitInputDefault(kLdCal cal, kSize input);

/**
* Gets the current limit input (in device units) to be used at device startup. 
* 
* kSIZE_NULL is returned if this input is not not set/used.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Current limit input (in device units) to be used at device startup.
*/
kFsFx(kSize) kLdCal_LimitInputDefault(kLdCal cal);

/**
* Defines the range of the input values for the set point to optical power map specified via kLdCal_SetPowerOutput. 
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   begin         Device input mapping to the first entry in set point to optical power map.
* @param   step          Increment, in device units, between adjacent entries of the set point to optical power map.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetPowerInputInfo(kLdCal cal, kSize begin, kSize step);

/**
* Queries the range of the input values for the set point to the optical power map specified via kLdCal_SetPowerOutput. 
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   begin         Pointer receiving device input mapping to the first entry in set point to optical power map.
* @param   step          Pointer receiving the increment, in device units, between adjacent entries in set point to optical power map.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_PowerInputInfo(kLdCal cal, kSize* begin, kSize* step);

/**
* Defines the range of the input values for the current limit to optical power map specified via kLdCal_SetLimitOutput.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   begin         Device input mapping to the first entry in current limit to optical power map.
* @param   step          Increment, in device units, between adjacent entries of the current limit to optical power map.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetLimitInputInfo(kLdCal cal, kSize begin, kSize step);

/**
* Queries the range of the input values for the current limit to the optical power map specified via kLdCal_SetLimitOutput.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   begin         Pointer receiving device input mapping to the first entry in current limit to optical power map.
* @param   step          Pointer receiving the increment, in device units, between adjacent entries in current limit to optical power map.
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_LimitInputInfo(kLdCal cal, kSize* begin, kSize* step);

/**
* Sets the calibration mapping for the driver power set point. 
* 
* The input is expected in the form of a 1D array of output optical power values (mW) mapping to a range 
* of device inputs determined by 'begin' and 'step' parameters to kLdCal_SetPowerInputInfo.
* 
* During assignment, the calibration class makes a copy of the input array. The caller is responsible for destroying the input.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   powerMap      Set point calibration mapping. kArray1<k64f>
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetPowerOutput(kLdCal cal, kArray1 powerMap);

/**
* Gets the calibration mapping for driver set point
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Set point calibration mapping
*/
kFsFx(kArray1) kLdCal_PowerOutput(kLdCal cal);

/**
* Sets the calibration mapping for the driver current limit. 
* 
* The input is expected in the form of a 1D array of output optical power values (mW) mapping to a range of inputs 
* determined by 'begin' and 'step' parameters to kLdCal_SetLimiInputtInfo
*
* As not all drivers support programmable current limit, this field may be left unset (kLdCal_LimitOutput will return kNULL)
* During assignment, the calibration class makes a copy of the input array. The caller is responsible for destroying the input.
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @param   limitMap      Current limit calibration mapping. kArray1<k64f>
* @return                Operation status
*/
kFsFx(kStatus) kLdCal_SetLimitOutput(kLdCal cal, kArray1 limitMap);

/**
* Gets the calibration mapping for driver set point. 
* 
* As not all drivers support programmable current limit, the limit calibration may be left unset. In this case this accessor 
* will return kNULL;
*
* @public                @memberof kLdCal
* @param   cal           The class object.
* @return                Current limit calibration mapping
*/
kFsFx(kArray1) kLdCal_LimitOutput(kLdCal cal);

#include <kFireSync/Utils/kLdCal.x.h>

#endif
