/** 
 * @file    kArp.h
 * @brief   Utilities for ARP probing/zeroconf. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.
  */
#ifndef K_FIRESYNC_ARP_H
#define K_FIRESYNC_ARP_H

#include <kApi/kApiDef.h>
#include <kFireSync/kFsDef.h>

/**
 * @class       kArp
 * @extends     kObject
 * @ingroup     kFireSync-Net
 * @brief       Implements ARP protocol (IP probing and announcing). 
 */

//typedef kObject kArp;      --forward-declared in kFsDef.x.h
  
/** 
 * Constructs a kArp object.
 * 
 * @public                      @memberof kArp
 * @param   arp                 Destination for the constructed object handle. 
 * @param   adapterName         Adapter used for sending/receiving raw packets.
 * @param   allocator           Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kArp_Construct(kArp* arp, const kChar* adapterName, kAlloc allocator);

/** 
 * Constructs a kArp object.
 *
 * Socket is not freed when the arp object is destroyed. Socket must be valid through
 * the whole lifetime of the arp object.
 * 
 * @public                      @memberof kArp
 * @param   arp                 Destination for the constructed object handle. 
 * @param   sockt               Socket used for sending/receiving raw packets.
 * @param   allocator           Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kArp_ConstructEx(kArp* arp, kPacketSocket sockt, kAlloc allocator);

/** 
 * Probes whether the given IP address is in use. 
 *
 * This function sends ARP probes to detect if an IP address is used in the local network. 
 * The function returns kOK when the IP is detected and kERROR_NOT_FOUND in case it's not found.
 *
 * @public                      @memberof kArp
 * @param   arp                 kArp object.
 * @param   macAddress          Source MAC address used for probing.
 * @param   ipAddress           IP address to be probed.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kArp_Probe(kArp arp, kMacAddress macAddress, kIpAddress ipAddress);

/** 
 * Announces an IP address. 
 *
 * This function announces that the sending host is using this IP now. Any currently running announcement
 * is stopped before. This method is asynchronous. It is safe to destroy the object while announcement is in progress.
 * 
 *
 * @public                      @memberof kArp
 * @param   arp                 kArp object.
 * @param   macAddress          Source MAC address used for probing.
 * @param   ipAddress           IP address to be announced.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kArp_BeginAnnounce(kArp arp, kMacAddress macAddress, kIpAddress ipAddress);

/** 
 * Reports if announcement is still running. 
 *
 * Announce messages must be send multiple times with some specific time between messages. 
 * This function reports if a former announcement started with kArp_BeginAnnounce
 * is still running.
 *
 * @public                      @memberof kArp
 * @param   arp                 kArp object.
 * @return                      kTRUE is announcement still running, otherwise false. 
 */
kFsFx(kStatus) kArp_IsAnnouncing(kArp arp);

#include <kFireSync/Net/kArp.x.h>

#endif
