/** 
 * @file    kUnpackedCompressedPhase.h
 * @brief   Declares the kUnpackedCompressedPhase type. 
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_UNPACKED_COMPRESSED_PHASE_H
#define K_FIRESYNC_UNPACKED_COMPRESSED_PHASE_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/Data/kUnpackedCompressedPhase.x.h>

/**
 * @class   kUnpackedCompressedPhase
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Represents unpacked lists of compressed phase or intensity stream entries. 
 *
 * This data type is provided for convenience in the implementation of phase/intensity 
 * compression logic. It is not intended for general use outside of this context. 
 */
//typedef kObject kUnpackedCompressedPhase;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kUnpackedCompressedPhase object.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Destination for the constructed object handle. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUnpackedCompressedPhase_Construct(kUnpackedCompressedPhase* data, kAlloc allocator); 

/** 
 * Initializes this object based on metadata from a compressed phase object.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @param   isIntensity Will this object represent intensity data (vs phase)?
 * @param   compressed  Compressed phase object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kUnpackedCompressedPhase_ConfigureFrom(kUnpackedCompressedPhase data, kBool isIntensity, kCompressedPhase compressed); 

/** 
 * Sets whether this object represents intensity (vs phase) data.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   isIntensity     kTRUE if the object represents intensity data.
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetIsIntensity(kUnpackedCompressedPhase data, kBool isIntensity)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->isIntensity = isIntensity;

    return kOK;
}

/** 
 * Reports whether this object represents intensity (vs phase) data.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              kTRUE if the object represents intensity data. 
 */
kInlineFx(kBool) kUnpackedCompressedPhase_IsIntensity(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->isIntensity;
}

/** 
 * Sets prediction value associated with this frame.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   prediction      Prediction value.
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetPrediction(kUnpackedCompressedPhase data, k32u prediction)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->prediction = prediction;

    return kOK;
}

/** 
 * Prediction value associated with this frame.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Prediction value.
 */
kInlineFx(k32s) kUnpackedCompressedPhase_Prediction(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->prediction;
}

/** 
 * Sets uncompressed subframe length for this subframe.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   subframeLength  Subframe length, in pixels.
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetSubframeLength(kUnpackedCompressedPhase data, kSize subframeLength)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->subframeLength = subframeLength;

    return kOK;
}

/** 
 * Uncompressed subframe length for this subframe.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Subframe length, in pixels.
 */
kInlineFx(kSize) kUnpackedCompressedPhase_SubframeLength(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->subframeLength;
}

/** 
 * Sets the specified uncompressed array dimension length associated with this frame.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   dimension       Array dimension (0 or 1).
 * @param   length          Array dimension length, in pixels.
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetLength(kUnpackedCompressedPhase data, kSize dimension, kSize length)
{
    kObj(kUnpackedCompressedPhase, data); 

     obj->length[dimension] = length;

    return kOK;
}

/** 
 * Reports the specified uncompressed array dimension length associated with this frame.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @param   dimension   Array dimension (0 or 1).
 * @return              Array dimension length, in pixels.
 */
kInlineFx(kSize) kUnpackedCompressedPhase_Length(kUnpackedCompressedPhase data, kSize dimension)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->length[dimension];
}

/** 
 * Sets the delta bit depth associated with this frame.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   bitDepth        Delta bit depth (0 if not present).
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetDeltaBitDepth(kUnpackedCompressedPhase data, k32u bitDepth)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->deltaBitDepth = bitDepth;

    return kOK;
}

/** 
 * Reports the delta bit depth associated with this frame.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Delta bit depth (0 if not present).
 */
kInlineFx(k32u) kUnpackedCompressedPhase_DeltaBitDepth(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->deltaBitDepth;
}

/** 
 * Sets the fail-first bit depth associated with this frame.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   bitDepth        Fail-first bit depth (0 if not present).
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetFail1BitDepth(kUnpackedCompressedPhase data, k32u bitDepth)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->fail1BitDepth = bitDepth;

    return kOK;
}

/** 
 * Reports the fail-first bit depth associated with this frame.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Fail-first bit depth (0 if not present).
 */
kInlineFx(k32u) kUnpackedCompressedPhase_Fail1BitDepth(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->fail1BitDepth;
}

/** 
 * Sets the fail-second bit depth associated with this frame.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   bitDepth        Fail-second bit depth (0 if not present).
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetFail2BitDepth(kUnpackedCompressedPhase data, k32u bitDepth)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->fail2BitDepth = bitDepth;

    return kOK;
}

/** 
 * Reports the fail-second bit depth associated with this frame.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Fail-second bit depth (0 if not present).
 */
kInlineFx(k32u) kUnpackedCompressedPhase_Fail2BitDepth(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->fail2BitDepth;
}

/** 
 * Sets the null bit depth associated with this frame.
 *
 * @public                  @memberof kUnpackedCompressedPhase
 * @param   data            Unpacked compressed phase object.
 * @param   bitDepth        Null bit depth (0 if not present).
 * @return                  Operation status.             
 */
kInlineFx(kStatus) kUnpackedCompressedPhase_SetNullBitDepth(kUnpackedCompressedPhase data, k32u bitDepth)
{
    kObj(kUnpackedCompressedPhase, data); 

    obj->nullBitDepth = bitDepth;

    return kOK;
}

/** 
 * Reports the null bit depth associated with this frame.
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Null bit depth (0 if not present).
 */
kInlineFx(k32u) kUnpackedCompressedPhase_NullBitDepth(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->nullBitDepth;
}

/** 
 * Provides reference to delta list. 
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Delta list (kArrayList<k32s>).
 */
kInlineFx(kArrayList) kUnpackedCompressedPhase_DeltaList(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->deltaList;
}

/** 
 * Provides reference to fail-first list. 
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Fail-first list (kArrayList<kPair32s>).
 */
kInlineFx(kArrayList) kUnpackedCompressedPhase_Fail1List(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->fail1List;
}

/** 
 * Provides reference to fail-second list. 
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Fail-second list (kArrayList<kPair32s>).
 */
kInlineFx(kArrayList) kUnpackedCompressedPhase_Fail2List(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->fail2List;
}

/** 
 * Provides reference to null list. 
 *
 * @public              @memberof kUnpackedCompressedPhase
 * @param   data        Unpacked compressed phase object.
 * @return              Null list (kArrayList<kPair32s>).
 */
kInlineFx(kArrayList) kUnpackedCompressedPhase_NullList(kUnpackedCompressedPhase data)
{
    kObj(kUnpackedCompressedPhase, data); 

    return obj->nullList;
}

#endif
