/** 
 * @file    kAlgMsg.h
 * @brief   Declares the kAlgMsg type. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ALG_MSG_H
#define K_FIRESYNC_ALG_MSG_H

#include <kFireSync/Data/kMsgSet.h>

/** @relates kAlgMsg @{ */
#define kALG_MSG_MAX_PARTS         (4)         ///< Maximum number of content parts supported by kAlgMsg. 
/** @} */

#include <kFireSync/Data/kAlgMsg.x.h>

/**
 * @class   kAlgMsg
 * @extends kMsgSet
 * @ingroup kFireSync-Data
 * @brief   Abstract base class for algorithm data messages. 
 * 
 * kAlgMsg is intended to serve as the base class for messages that contain camera 
 * algorithm data. Use of kAlgMsg is not strictly limited to this purpose. If desired, 
 * kAlgMsg can also be extended by applications to represent the output from pipe blocks. 
 * 
 * kAlgMsg supports the delivery of multipart frames (that is, frames consisting of more 
 * than one type of data content) and supports the delivery of multiple frames in a single 
 * message (aka "batching"). For compatibility with FireSync message infrastructure, the 
 * first content part should always be of type kStamp; the remaining content parts can be 
 * defined as needed by particular algorithms. The kAlgMsg implementation supports up to
 * kALG_MSG_MAX_PARTS content parts; this value can be increased by adjusting the constant, 
 * if needed. 
 */
//typedef kMsgSet kAlgMsg;            --forward-declared in kFsDef.x.h

/** 
 * Reports the number of frames in the message. 
 *
 * @public          @memberof kAlgMsg
 * @param   msg     Message object.  
 * @return          Frame count.
 */
kInlineFx(kSize) kAlgMsg_FrameCount(kAlgMsg msg)
{
    kObj(kAlgMsg, msg); 

    return obj->frameCount; 
}

/** 
 * Reports the number of content parts for each frame.
 *
 * @public          @memberof kAlgMsg
 * @param   msg     Message object.  
 * @return          Content parts per frame.
 */
kInlineFx(kSize) kAlgMsg_PartCount(kAlgMsg msg)
{
    kObj(kAlgMsg, msg); 

    return obj->partCount; 
}

/** 
 * Reports the data type of the specified content part.
 *
 * @public              @memberof kAlgMsg
 * @param   msg         Message object.  
 * @param   partIndex   Content part index.   
 * @return              Part type.
 */
kInlineFx(kType) kAlgMsg_PartType(kAlgMsg msg, kSize partIndex)
{
    kObj(kAlgMsg, msg); 

    kAssert(partIndex < obj->partCount); 

    return xkAlgMsg_DescriptorAt(msg, partIndex)->itemType; 
}

/** 
 * Reports the dimensionality of the specified content part.
 *
 * @public              @memberof kAlgMsg
 * @param   msg         Message object.  
 * @param   partIndex   Content part index.   
 * @return              Count of part dimensions
 */
kInlineFx(kSize) kAlgMsg_PartDimensionCount(kAlgMsg msg, kSize partIndex)
{
    kObj(kAlgMsg, msg); 

    kAssert(partIndex < obj->partCount); 

    return xkAlgMsg_DescriptorAt(msg, partIndex)->dimensionCount; 
}

/** 
 * Reports the length of the specified part dimension.
 *
 * @public                  @memberof kAlgMsg
 * @param   msg             Message object.  
 * @param   partIndex       Content part index.   
 * @param   dimensionIndex  Part dimension index. 
 * @return                  Length of dimension.
 */
kInlineFx(kSize) kAlgMsg_PartLength(kAlgMsg msg, kSize partIndex, kSize dimensionIndex)
{
    kObj(kAlgMsg, msg); 

    kAssert(partIndex < obj->partCount); 
    kAssert(dimensionIndex < xkAlgMsg_DescriptorAt(msg, partIndex)->dimensionCount); 

    return xkAlgMsg_DescriptorAt(msg, partIndex)->length[dimensionIndex]; 
}

/** 
 * Gets a pointer to the specified frame content part.
 *
 * @public                  @memberof kAlgMsg
 * @param   msg             Message object.  
 * @param   frameIndex      Index of frame within this message.
 * @param   partIndex       Content part index.   
 * @return                  Pointer to part data.
 */
kInlineFx(void*) kAlgMsg_Part(kAlgMsg msg, kSize frameIndex, kSize partIndex)
{
    kObj(kAlgMsg, msg); 

    kAssert(frameIndex < obj->frameCount); 
    kAssert(partIndex < obj->partCount); 

    return obj->parts[frameIndex*kAlgMsg_PartCount(msg) + partIndex];
}

/** 
 * Gets a strongly-typed pointer to the specified frame content part.
 *
 * A debug assertion will be raised if the size of the specified type is not equal to the 
 * size of the part type.
 * 
 * @relates                     kAlgMsg
 * @param   kAlgMsg_msg         Message object.  
 * @param   kSize_frameIndex    Index of frame within this message.
 * @param   kSize_partIndex     Content part index.   
 * @param   T                   Item type identifier (e.g., k32s).
 * @return                      Strongly-typed pointer to part data.
 */
#define kAlgMsg_PartT(kAlgMsg_msg, kSize_frameIndex, kSize_partIndex, T) \
    kCast(T*, xkAlgMsg_PartT(kAlgMsg_msg, kSize_frameIndex, kSize_partIndex, sizeof(T)))

#endif
