/** 
 * @file    kCameraInfo.h
 * @brief   Declares the kCameraInfo class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_INFO_H
#define K_FIRESYNC_CAMERA_INFO_H

#include <kFireSync/kNodeDef.h>

/**
 * @internal
 * @class   kCameraInfo
 * @extends kObject
 * @ingroup kFireSync-Client-Camera
 * @brief   Abstract base class for camera model classes. 
 */
//typedef kObject kCameraInfo;         --forward-declared in kFsDef.x.h

/** 
 * Constructs a kCameraInfo object.
 *
 * @public                  @memberof kCameraInfo
 * @param   info            Destination for the constructed object handle. 
 * @param   model           Camera model identifier.
 * @param   capabilities    Camera algorithm capabilities. 
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kCameraInfo_Construct(kCameraInfo* info, kCameraModel model, kCameraCapability capabilities, kBool isTrailingRequired, kSize frameSizeGranularity, kVersion plVersion, kAlloc allocator);

kFsFx(kStatus) kCameraInfo_SetModel(kCameraInfo info, kCameraModel model);
kFsFx(kCameraModel) kCameraInfo_Model(kCameraInfo info);

kFsFx(k32u) kCameraInfo_MinWidth(kCameraInfo info);
kFsFx(k32u) kCameraInfo_MaxWidth(kCameraInfo info);
kFsFx(k32u) kCameraInfo_WidthGranularity(kCameraInfo info);

kFsFx(k32u) kCameraInfo_MinHeight(kCameraInfo info);
kFsFx(k32u) kCameraInfo_MaxHeight(kCameraInfo info);
kFsFx(k32u) kCameraInfo_HeightGranularity(kCameraInfo info);

kFsFx(k32u) kCameraInfo_LeftGranularity(kCameraInfo info);
kFsFx(k32u) kCameraInfo_TopGranularity(kCameraInfo info);

kFsFx(kCameraSubsampling) kCameraInfo_XSubModes(kCameraInfo info);
kFsFx(k32u) kCameraInfo_XSubOptions(kCameraInfo info);
kFsFx(kCameraSubsampling) kCameraInfo_YSubModes(kCameraInfo info);
kFsFx(k32u) kCameraInfo_YSubOptions(kCameraInfo info);
kFsFx(kBool) kCameraInfo_SubModeConstrainedEqual(kCameraInfo info);

kFsFx(kStatus) kCameraInfo_ApportionSubsampling(kCameraInfo info, k32u xSub, k32u ySub, k32u* imagerSubXShift, k32u* plSubXShift, k32u* imagerSubYShift, k32u* plSubYShift); 

kFsFx(kBool) kCameraInfo_RequiresPixelMap(kCameraInfo info); 

kFsFx(kBool) kCameraInfo_CanInvertX(kCameraInfo info);
kFsFx(kBool) kCameraInfo_CanInvertY(kCameraInfo info);

kFsFx(k32s) kCameraInfo_MinBlackLevel(kCameraInfo info);
kFsFx(k32s) kCameraInfo_MaxBlackLevel(kCameraInfo info);

kFsFx(k32u) kCameraInfo_MinLvdsPower(kCameraInfo info);
kFsFx(k32u) kCameraInfo_MaxLvdsPower(kCameraInfo info);
kFsFx(k32u) kCameraInfo_ReleaseLvdsPower(kCameraInfo info);
kFsFx(k32u) kCameraInfo_FactoryLvdsPower(kCameraInfo info);

kFsFx(kBool) kCameraInfo_HasGainChannel(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kCameraInfo_MinGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kCameraInfo_MaxGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kCameraInfo_GainStep(kCameraInfo info, kCameraGain channel, k64f gain);
kFsFx(k64f) kCameraInfo_GainFactor(kCameraInfo info, kCameraGain channel, k32u index);
kFsFx(k64f) kCameraInfo_DefaultGainFactor(kCameraInfo info, kCameraGain channel);

kFsFx(kSize) kCameraInfo_KneeCountMin(kCameraInfo info);
kFsFx(kSize) kCameraInfo_KneeCountMax(kCameraInfo info);
kFsFx(kSize) kCameraInfo_KneeCountGranularity(kCameraInfo info);

kFsFx(kCameraExposure) kCameraInfo_ExposureModes(kCameraInfo info);

kFsFx(k64u) kCameraInfo_ReadoutTime(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_MinInterExposureGap(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_MinInterReadoutGap(kCameraInfo info, const kCameraInfoParams* params);

//minimum/maximum *nominal* exposure; convoluted -- should restructure these methods to report physical exposure
kFsFx(k64u) kCameraInfo_MinExposure(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_MaxExposure(kCameraInfo info, const kCameraInfoParams* params);

kFsFx(k64u) kCameraInfo_PreExposure(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_PostExposure(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_PlProcessingTime(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_PlPostProcessingTime(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_MinIsrGap(kCameraInfo info, const kCameraInfoParams* params);

kFsFx(k64f) kCameraInfo_RowReadoutEfficiency(kCameraInfo info, const kCameraInfoParams* params);

kFsFx(kBool) kCameraInfo_IsWidthGrouped(kCameraInfo info);
kFsFx(kBool) kCameraInfo_IsHeightGrouped(kCameraInfo info);
kFsFx(kBool) kCameraInfo_IsTopGrouped(kCameraInfo info);
kFsFx(kBool) kCameraInfo_IsLeftGrouped(kCameraInfo info);

kFsFx(kCameraExtension) kCameraInfo_ExtensionTypes(kCameraInfo info);

kFsFx(kBool) kCameraInfo_HasTempSensor(kCameraInfo info);
kFsFx(kBool) kCameraInfo_IsColor(kCameraInfo info);
kFsFx(kCfa) kCameraInfo_Cfa(kCameraInfo info);
kFsFx(kPixelFormat) kCameraInfo_PixelFormat(kCameraInfo info);

kFsFx(kCameraBufferSpace) kCameraInfo_VideoBufferSapce(kCameraInfo info);

kFsFx(kBool) kCameraInfo_RequiresCalibration(kCameraInfo info); 
kFsFx(kBool) kCameraInfo_HasTestPattern(kCameraInfo info); 

kFsFx(const kRangeLutCaps*) kCameraInfo_RangeLutCaps(kCameraInfo info);

kFsFx(kStatus) kCameraInfo_ImageFrame(kCameraInfo info, k32u width, k32u height, kCameraImageInfo* imageInfo, kSize* dataSize);
kFsFx(kStatus) kCameraInfo_ListFrame(kCameraInfo info, kAlgInfo algInfo, kBool sortingEnabled, kSize sliceCount, kCameraFrameAlgType* type, kCameraListInfo* listInfo, kSize* dataSize);
kFsFx(kStatus) kCameraInfo_ProfileFrame(kCameraInfo info, kAlgInfo algInfo, kSize sliceCount, kBool hasSlices, kCameraFrameAlgType* type, kCameraProfileInfo* profileInfo, kSize* dataSize, kBool xResamplingEnabled);
kFsFx(kStatus) kCameraInfo_BinCSumFrame(kCameraInfo info, k32u width, k32u height, kCameraFrameAlgType* type, kCameraBinCSumInfo* binCSumInfo, kSize* dataSize);
kFsFx(kStatus) kCameraInfo_PhaseFrame(kCameraInfo info, k32u width, k32u height, kCameraPhaseDecoderFormat format, kCameraFrameAlgType* type, kCameraPhaseInfo* phaseInfo, kSize* stripeDataSize, kSize* phaseDataSize);
kFsFx(kStatus) kCameraInfo_CalculateFrameSize(kCameraInfo info, kSize dataSize, kSize* totalSize, kSize* trailingSize);

kFsFx(kCameraCapability) kCameraInfo_Capabilities(kCameraInfo info);

kFsFx(kStatus) kCameraInfo_PrimePeriod(kCameraInfo info, k64u* delay, k64u* exposure, k64u* gap);

#include <kFireSync/Client/Camera/kCameraInfo.x.h>

#endif
