/** 
 * @file    kCameraPhaseDecoder.h
 * @brief   Declares the kCameraPhaseDecoder type. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_PHASE_DECODER_H
#define K_FIRESYNC_CAMERA_PHASE_DECODER_H

#include <kFireSync/Client/Camera/Extensions/kCameraExt.h>

/**
 * @class   kCameraPhaseDecoder
 * @extends kCameraExt
 * @ingroup kFireSync-Client-Camera-Extensions
 * @brief   Represents parameters for a camera stripe+phase acceleration extension. 
 * 
 */
//typedef kCameraExt kCameraPhaseDecoder;            --forward-declared in kFsDef.x.h

/**
 * Sets the format.
 * 
 * Note that with legacy formats such as kCAMERA_PHASE_DECODER_FORMAT_S8_C8I8P16 and kCAMERA_PHASE_DECODER_FORMAT_S8_P16
 * (from PL CFG 623), stripe and phase data are sent as separate messages. For each camera, one stripe and one phase message 
 * will be delivered, where each message is structured as a kMsg containing a kArray2 of the relevant pixel type. 
 * 
 * For modern formats, there is only one message per camera (the 'phase' data now includes the stripe information as well). 
 * For uncompressed data (i.e., kCAMERA_PHASE_DECODER_FORMAT_I8P24), each camera produces one kMsg object containing 
 * a kArray2<kPhasePixel2>. Each 32-bit kPhasePixel2 element contains an 8-bit intensity value and a 24-bit phase value.  
 * 
 * For compressed data, (i.e., kCAMERA_PHASE_DECODER_FORMAT_DELTA_I8_P24, kCAMERA_PHASE_DECODER_FORMAT_DELTA_P24), each camera
 * produces one kMsg object containing a kCompressedPhase object. The kCAMERA_PHASE_DECODER_FORMAT_DELTA_I8_P24 format yields a 
 * kCompressedPhase object that includes both intensity and phase data. The kCAMERA_PHASE_DECODER_FORMAT_DELTA_P24 format yields a 
 * kCompressedPhase object that includes only phase data.  
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @param   format      Format.
 * @return              Operation status.
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetFormat(kCameraPhaseDecoder extension, kCameraPhaseDecoderFormat format);

/**
 * Gets the format.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Format.
 */
kFsFx(kCameraPhaseDecoderFormat) kCameraPhaseDecoder_Format(kCameraPhaseDecoder extension);

/**
 * Reports constraint and validity information for the Format setting.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kCameraPhaseDecoder_FormatInfo(kCameraPhaseDecoder extension, kInfoBits* info);

/** 
 * Sets the number of stripe frames.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   count       Stripe count. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetStripeCount(kCameraPhaseDecoder extension, kSize count, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the number of stripe frames. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Step count. 
 */
kFsFx(kSize) kCameraPhaseDecoder_StripeCount(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the StripeCount setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_StripeCountInfo(kCameraPhaseDecoder extension, kInfoSize* info); 

/** 
 * Sets the stripe threshold. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   threshold   Stripe threshold.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetStripeThreshold(kCameraPhaseDecoder extension, k32u threshold, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the stripe threshold. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Stripe threshold. 
 */
kFsFx(k32u) kCameraPhaseDecoder_StripeThreshold(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the StripeThreshold setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_StripeThresholdInfo(kCameraPhaseDecoder extension, kInfo32u* info); 

/** 
 * Sets the number of phase frames.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   count       Phase count. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetPhaseCount(kCameraPhaseDecoder extension, kSize count, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the number of phase frames. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Step count. 
 */
kFsFx(kSize) kCameraPhaseDecoder_PhaseCount(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the PhaseCount setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_PhaseCountInfo(kCameraPhaseDecoder extension, kInfoSize* info); 

/** 
 * Sets the phase period. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   period      Phase period. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetPhasePeriod(kCameraPhaseDecoder extension, k32u period, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the phase period. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Phase period.
 */
kFsFx(k32u) kCameraPhaseDecoder_PhasePeriod(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the PhasePeriod setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_PhasePeriodInfo(kCameraPhaseDecoder extension, kInfo32u* info); 

/** 
 * Sets the contrast threshold. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   threshold   Contrast threshold.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetPhaseThreshold(kCameraPhaseDecoder extension, k32u threshold, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the contrast threshold. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Contrast threshold. 
 */
kFsFx(k32u) kCameraPhaseDecoder_PhaseThreshold(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the PhaseThreshold setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_PhaseThresholdInfo(kCameraPhaseDecoder extension, kInfo32u* info); 

/** 
 * Sets the phase prediction. 
 * 
 * Phase compression requires a phase prediction value. This value is essentially the expected change in phase value 
 * from one pixel to the next (expected average of the first derivative of phase data in the x axis). A good 
 * phase prediction value decreases the average delta bit depth required to encode phase pixel changes. 
 * 
 * Selecting a good prediction value requires understanding the expected change in phase data. For more information on 
 * selecting a phase prediction value, please contact sensor software.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   prediction  Phase prediction.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetPhasePrediction(kCameraPhaseDecoder extension, k32u prediction, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the phase prediction. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Phase prediction.
 */
kFsFx(k32u) kCameraPhaseDecoder_PhasePrediction(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the PhasePrediction setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_PhasePredictionInfo(kCameraPhaseDecoder extension, kInfo32u* info); 

/** 
 * Sets the phase delta bit depth. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   bitDepth    Bit depth. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetPhaseDeltaBitDepth(kCameraPhaseDecoder extension, k32u bitDepth, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Reports the phase delta bit depth. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @return              Phase delta bit depth. 
 */
kFsFx(k32u) kCameraPhaseDecoder_PhaseDeltaBitDepth(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the PhaseDeltaBitDepth setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_PhaseDeltaBitDepthInfo(kCameraPhaseDecoder extension, kInfo32u* info); 

/** 
 * Sets the desired number of compressed subframes. 
 *
 * Compressed frames consist of one or more subframes, where each subframe is represented by 
 * a kCompressedPhase object. This method specifies the desired number of subframes.  
 * 
 * The desired subframe count is typically set to a value that corresponds with the number of 
 * processor cores on the client that will perform decompresion.
 * 
 * The actual number of subframes generated may be less than the maximum, depending 
 * on window size and PL subframing capability. Note, it is not considered a validation error if 
 * the desired subframe count exceeds capabilities. In this case, the actual number of subframes will 
 * be automatically adjusted downwards by the node.
 * 
 * By default, the desired subframe count is set to one.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   count       Desired subframe count.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetSubframeCount(kCameraPhaseDecoder extension, k32u count, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Reports the desired subframe count.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Desired number of subframes. 
 */
kFsFx(k32u) kCameraPhaseDecoder_SubframeCount(kCameraPhaseDecoder extension); 

/** 
 * Reports constraint and validity information for the SubframeCount setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SubframeCountInfo(kCameraPhaseDecoder extension, kInfo32u* info); 

/** 
 * Writes phase decoder LUT. 
 
 * This function can be used to write the phase decoder LUT. It's format is kArray1<k32s> for all types except
 * kCAMERA_PHASE_DECODER_LUT_TYPE_EPS. In case of kCAMERA_PHASE_DECODER_LUT_TYPE_EPS it's kArray2<k32s> 
 * (see kS3dEmbeddedPhaseDecoder_ConstructShiftMatrix32s() for details).
 * 
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   type        LUT type.  
 * @param   lut         LUT object (kArray1<k32s> or kArray2<k32s>).  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_WriteLut(kCameraPhaseDecoder extension, kCameraPhaseDecoderLutType type, kObject lut); 

/**
 * Gets the type.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @return              Type.
 */
kFsFx(kCameraPhaseDecoderType) kCameraPhaseDecoder_DecoderType(kCameraPhaseDecoder extension);

/** 
 * Sets the EPS frequency count. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   count       Frequency count. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetEpsFrequencyCount(kCameraPhaseDecoder extension, kSize count);

/** 
 * Reports the EPS frequency count.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @return              Frequency count. 
 */
kFsFx(kSize) kCameraPhaseDecoder_EpsFrequencyCount(kCameraPhaseDecoder extension);

/** 
 * Reports constraint and validity information for the EPS FrequencyCount setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_EpsFrequencyCountInfo(kCameraPhaseDecoder extension, kInfoSize* info);

/** 
 * Sets the EPS F1 phase count. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   count       F1 phase count. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetEpsF1PhaseCount(kCameraPhaseDecoder extension, k32u count, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the EPS F1 phase count. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @return              F1 phase count. 
 */
kFsFx(k32u) kCameraPhaseDecoder_EpsF1PhaseCount(kCameraPhaseDecoder extension);

/** 
 * Reports constraint and validity information for the F1PhaseCount setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_EpsF1PhaseCountInfo(kCameraPhaseDecoder extension, kInfo32u* info);

/** 
 * Sets the EPS embedded period at a specific index. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   index       Period index.
 * @param   period      Period. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetEpsEmbeddedPeriod(kCameraPhaseDecoder extension, kSize index, k32u period);

/** 
 * Reports the EPS embedded period at a specific index. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @param   index       Period index.
 * @return              Embedded period. 
 */
kFsFx(k32u) kCameraPhaseDecoder_EpsEmbeddedPeriod(kCameraPhaseDecoder extension, kSize index);

/** 
 * Reports constraint and validity information for the EmbeddedPeriod setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   index       Period index.
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_EpsEmbeddedPeriodInfo(kCameraPhaseDecoder extension, kSize index, kInfo32u* info);

/** 
 * Sets the EPS embedded phase shift. 
 * 
 * PL applies the shift value to phase period to calculate internal extended phase period (= phase period << shift). 
 * Same shift value will be used to shift down the output phase map on export.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   shift       Phase shift value. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_SetEpsPhaseShift(kCameraPhaseDecoder extension, k32u shift, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the EPS embedded phase shift value. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @return              Phase shift value. 
 */
kFsFx(k32u) kCameraPhaseDecoder_EpsPhaseShift(kCameraPhaseDecoder extension);

/** 
 * Reports constraint and validity information for the phase shift setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_EpsPhaseShiftInfo(kCameraPhaseDecoder extension, kInfo32u * info);

/** 
 * Sets the EPS transposed info.
 * 
 * Set to kTRUE if horizontal phase patterns are used instead of vertical phase patterns.
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   enable      kTRUE to enable horizontal patterns; kFALSE otherwise.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_EnableEpsTranspose(kCameraPhaseDecoder extension, kBool enable);

/** 
 * Reports the EPS embedded transposed setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object.
 * @return              Phase shift value. 
 */
kFsFx(kBool) kCameraPhaseDecoder_EpsTransposeEnabled(kCameraPhaseDecoder extension);

/** 
 * Reports constraint and validity information for the EPS transposed setting. 
 *
 * @public              @memberof kCameraPhaseDecoder
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraPhaseDecoder_EpsTransposeEnabledInfo(kCameraPhaseDecoder extension, kInfoBool* info);

#include <kFireSync/Client/Camera/Extensions/kCameraPhaseDecoder.x.h>

#endif
