/** 
 * @file    kAxAction.h
 * @brief   Declares the kAxAction class and related types. 
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_AX_ACTION_H
#define K_FIRESYNC_AX_ACTION_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/kNodeDef.h>

/**
 * @class   kAxAction
 * @extends kObject
 * @ingroup kFireSync-Action
 * @brief   Abstract base class for action implementations. 
 * 
 * Actions implement the server side of user-defined remote procedure calls on FireSync nodes. Actions have 
 * XML settings, and can accept and produce kDat-6 serializable objects when executed.
 */
//typedef kObject kAxAction;         --forward-declared in kFsDef.x.h

/** 
 * Given a kAxAction subclass type, gets a description of the class. 
 * 
 * This method should be overridden by subclasses to provide a description of each action type.
 *
 * @public              @memberof kAxAction
 * @param   type        Action type. 
 * @return              Type description. 
 */
 kFsFx(const kChar*) kAxAction_Description(kType type);

/** 
 * Factory constructor for action objects. 
 * 
 * This function can be used to construct any action type. 
 *
 * @public                  @memberof kAxAction
 * @param   action          Receives constructed action object. 
 * @param   type            Action class type. 
 * @param   name            Descriptive action instance name. 
 * @param   trigger         Action trigger source.
 * @param   actionEnviron   Object to provide external services to the action. 
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kAxAction_Construct(kAxAction* action, kType type, const kChar* name, kActionTrigger trigger, kAxEnviron actionEnviron, kAlloc allocator); 

/** 
 * Virtual method that can be overridden to perform XML settings validation.
 *
 * @public              @memberof kAxAction
 * @param   type        Action type.   
 * @param   settings    Action xml settings.
 * @param   alloc       Allocator that can be used for any intermediate objects.
 * @return              Operation status. 
 */
kFsFx(kStatus) kAxAction_Validate(kType type, kXml settings, kAlloc alloc);

/** 
 * Virtual method that can be overridden to perform custom processing when action is invoked. 
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.   
 * @param   input       Input to be sent to remote procedure (or kNULL).  
 * @param   output      Recieves output from remote produre (can be kNULL). 
 * @param   alloc       Memory allocator for output object (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kAxAction_Invoke(kAxAction action, kObject input, kObject* output, kAlloc alloc);

/** 
 * Gets a descriptive name for the action instance. 
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.  
 * @return              Descriptive instance name. 
 */
kFsFx(const kChar*) kAxAction_Name(kAxAction action);

/** 
 * Gets the action trigger source.
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.  
 * @return              Action trigger source. 
 */
kFsFx(kActionTrigger) kAxAction_Trigger(kAxAction action);

/** 
 * Returns the action's settings. 
 *
 * The settings object returned by this function should not be modified during Invoke processing.
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.   
 * @return              Internal action settings. 
 */
kFsFx(kXml) kAxAction_Settings(kAxAction action);

/** 
 * Gets a reference to a kNode object representing the local node. 
 *
 * The kNode object returned by this function is the same object as the kNode object made available 
 * through the system object returned by kSvNode_Client. Accordingly, concurrent access to this kNode 
 * object should be coordinated with the resident node application.
 *
 * The node object returned by this function may not be connected.  It is recommended to connect and
 * and refresh within the action, if needed. 
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.  
 * @return              Local node object.
 */
kFsFx(kNode) kAxAction_Node(kAxAction action);

/** 
 * Gets a reference to a kSystem object representing the collection of discovered nodes. 
 *
 * The kSystem object returned by this function is the same object as the kSystem object made available 
 * through kSvNode_Client. Accordingly, concurrent access to this kSystem object should be coordinated 
 * with the resident node application.
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.  
 * @return              System object.
 */
kFsFx(kSystem) kAxAction_System(kAxAction action);

/** 
 * For actions that are run in server context, gets a reference to the local application object (if set).
 *
 * Server-side actions are sometimes designed to work in concert with a local application object (set 
 * via kSvNode_SetApp). This method can be used to access the server-side application object.
 *
 * @public              @memberof kAxAction
 * @param   action      Action object.  
 * @return              Local application object, if set; otherwise kNULL. 
 */
kFsFx(kPointer) kAxAction_App(kAxAction action);

#include <kFireSync/Action/kAxAction.x.h>

#endif
