/**
* @file    GvMeshMsg.h
* @brief   Declares the GvMeshMsg class.
*
* Copyright (C) 2020-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GV_MESH_MSG_H
#define GV_MESH_MSG_H

#include <GoVision/Data/GvDataMsg.h>

#include <kVision/L3d/kL3dTransform3d.h>

/**
* Encapsulates a mesh message.
*
* @class       GvMeshMsg
* @extends     GvDataMsg
* @ingroup     GoVision-Data
*/

#define GV_MESH_MSG_NUM_OF_SYSTEM_CHANNEL 6
#define GV_MESH_MSG_NUM_OF_MAX_USER_CHANNEL 5

typedef GvDataMsg GvMeshMsg;

typedef k32s GvMeshMsgChannelId;

#define GV_MESH_MSG_CHANNEL_ID_SYSTEM_VERTEX             (0)
#define GV_MESH_MSG_CHANNEL_ID_SYSTEM_FACET              (1)
#define GV_MESH_MSG_CHANNEL_ID_SYSTEM_FACET_NORMAL       (2)
#define GV_MESH_MSG_CHANNEL_ID_SYSTEM_VERTEX_NORMAL      (3)
#define GV_MESH_MSG_CHANNEL_ID_SYSTEM_VERTEX_TEXTURE     (4)
#define GV_MESH_MSG_CHANNEL_ID_SYSTEM_VERTEX_CURVATURE   (5)

/*! Channel type definition.
    Values range between 0 and 6 are reserved for system use.
    Values range between 7 and 65535 (0xFFFF) are reserved for future system use. 
    Any value outside of reserved range can be used for user channel.
 */
typedef k32u GvMeshMsgChannelType;

#define GV_MESH_MSG_CHANNEL_TYPE_INVALID            (0)
#define GV_MESH_MSG_CHANNEL_TYPE_VERTEX             (1)
#define GV_MESH_MSG_CHANNEL_TYPE_FACET              (2)
#define GV_MESH_MSG_CHANNEL_TYPE_FACET_NORMAL       (3)
#define GV_MESH_MSG_CHANNEL_TYPE_VERTEX_NORMAL      (4)
#define GV_MESH_MSG_CHANNEL_TYPE_VERTEX_TEXTURE     (5)
#define GV_MESH_MSG_CHANNEL_TYPE_VERTEX_CURVATURE   (6)
#define GV_MESH_MSG_CHANNEL_TYPE_RESERVE_START      (GV_MESH_MSG_CHANNEL_TYPE_INVALID)
#define GV_MESH_MSG_CHANNEL_TYPE_RESERVE_END        (65535)

typedef k32s GvMeshMsgChannelState;

#define GV_MESH_MSG_CHANNEL_STATE_ERROR         (-1)
#define GV_MESH_MSG_CHANNEL_STATE_UNALLOCATED   (0)
#define GV_MESH_MSG_CHANNEL_STATE_ALLOCATED     (1)
#define GV_MESH_MSG_CHANNEL_STATE_EMPTY         (2)
#define GV_MESH_MSG_CHANNEL_STATE_PARTIAL       (3)
#define GV_MESH_MSG_CHANNEL_STATE_FULL          (4)

/** 
 * Constructs a GvMeshMsg object.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 Receives constructed GvMeshMsg object. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_Construct(GvMeshMsg* msg, kAlloc alloc);

/** 
 * Assign a GvMeshMsg object from an existing GvMeshMsg. 
 * Shallow copy of source object. Any Aggregated object's handle will also be copied.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source GvMeshMsg. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_Assign(GvMeshMsg msg, GvMeshMsg src);

/** 
 * Clear all allocated channel buffer and reset 'hasData' flag. This will not deallocate any channel itself.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearAll(GvMeshMsg msg);

/** 
 * Set whether GvMeshMsg allocated any channel buffer. 
 * User is responsible for setting this flag correct.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   data                Whether any channel buffer has been allocated. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetHasData(GvMeshMsg msg, kBool data);

/** 
 * Get last known cached information on whether GvMeshMsg allocated any channel buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Whether any channel buffer has been allocated according to the last cached information. 
 */
GvFx(kBool) GvMeshMsg_HasData(GvMeshMsg msg);

/** 
 * Set offset values. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   offset              Offset values to be set. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetOffset(GvMeshMsg msg, kPoint3d64f offset);

/** 
 * Get offset values. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Offset. 
 */
GvFx(kPoint3d64f) GvMeshMsg_Offset(GvMeshMsg msg);
 
/** 
 * Set range values. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   range               Range values to be set. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetRange(GvMeshMsg msg, kPoint3d64f range);

/** 
 * Get range values. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Range. 
 */
GvFx(kPoint3d64f) GvMeshMsg_Range(GvMeshMsg msg);
 
/** 
 * Set tranform values. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   tranform            Tranform values to be set. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetTransform(GvMeshMsg msg, kL3dTransform3d tranform);

/** 
 * Get transform values. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Transform. 
 */
GvFx(kL3dTransform3d) GvMeshMsg_Transform(GvMeshMsg msg);

// Vertex
/** 
 * Allocate system channel vertex buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   size                Number of Item. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateVertexData(GvMeshMsg msg, kSize size, kAlloc alloc);

/** 
 * Assign system channel vertex buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignVertexData(GvMeshMsg msg, kArray1 src);

/** 
 * Get system channel vertex buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_VertexData(GvMeshMsg msg);

/** 
 * Clear system channel vertex buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearVertexData(GvMeshMsg msg);

/** 
 * Get allocated system channel vertex buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedVertexDataCount(GvMeshMsg msg);

/** 
 * Set used system channel vertex buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexDataCount(GvMeshMsg msg, kSize count);

/** 
 * Get used system channel vertex buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_VertexDataCount(GvMeshMsg msg);

/** 
 * Get system channel vertex ID. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      channel ID. 
 */
GvFx(GvMeshMsgChannelId) GvMeshMsg_VertexId(GvMeshMsg msg);

/** 
 * Set system channel vertex state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexState(GvMeshMsg msg, GvMeshMsgChannelState state);

/** 
 * Get system channel vertex state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_VertexState(GvMeshMsg msg);

// Facet
/** 
 * Allocate system channel facet buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   size                Number of Item. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateFacetData(GvMeshMsg msg, kSize size, kAlloc alloc);

/** 
 * Assign system channel facet buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignFacetData(GvMeshMsg msg, kArray1 src);

/** 
 * Get system channel facet buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_FacetData(GvMeshMsg msg);

/** 
 * Clear system channel facet buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearFacetData(GvMeshMsg msg);

/** 
 * Get allocated system channel facet buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedFacetDataCount(GvMeshMsg msg);

/** 
 * Set used system channel facet buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetFacetDataCount(GvMeshMsg msg, kSize count);

/** 
 * Get used system channel facet buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_FacetDataCount(GvMeshMsg msg);

/** 
 * Get system channel facet ID. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      channel ID. 
 */
GvFx(GvMeshMsgChannelId) GvMeshMsg_FacetId(GvMeshMsg msg);

/** 
 * Set system channel facet state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetFacetState(GvMeshMsg msg, GvMeshMsgChannelState state);

/** 
 * Get system channel facet state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_FacetState(GvMeshMsg msg);

// Facet normal
/** 
 * Allocate system channel facet normal buffer. 
 * Number of items in this buffer is always the same as system channel facet buffer.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateFacetNormalData(GvMeshMsg msg, kAlloc alloc);

/** 
 * Assign system channel facet normal buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignFacetNormalData(GvMeshMsg msg, kArray1 src);

/** 
 * Get system channel facet normal buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_FacetNormalData(GvMeshMsg msg);

/** 
 * Clear system channel facet normal buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearFacetNormalData(GvMeshMsg msg);

/** 
 * Get allocated system channel facet normal buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedFacetNormalDataCount(GvMeshMsg msg);

/** 
 * Set used system channel facet normal buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetFacetNormalDataCount(GvMeshMsg msg, kSize count);

/** 
 * Get used system channel facet normal buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_FacetNormalDataCount(GvMeshMsg msg);

/** 
 * Get system channel facet normal ID. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      channel ID. 
 */
GvFx(GvMeshMsgChannelId) GvMeshMsg_FacetNormalId(GvMeshMsg msg);

/** 
 * Set system channel facet normal state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetFacetNormalState(GvMeshMsg msg, GvMeshMsgChannelState state);

/** 
 * Get system channel facet normal state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_FacetNormalState(GvMeshMsg msg);

// Vertex normal
/** 
 * Allocate system channel vertex normal buffer. 
 * Number of items in this buffer is always the same as system channel vertex buffer.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateVertexNormalData(GvMeshMsg msg, kAlloc alloc);

/** 
 * Assign system channel vertex normal buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignVertexNormalData(GvMeshMsg msg, kArray1 src);

/** 
 * Get system channel vertex normal buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_VertexNormalData(GvMeshMsg msg);

/** 
 * Clear system channel vertex normal buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearVertexNormalData(GvMeshMsg msg);

/** 
 * Get allocated system channel vertex normal buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedVertexNormalDataCount(GvMeshMsg msg);

/** 
 * Set used system channel vertex normal buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexNormalDataCount(GvMeshMsg msg, kSize count);

/** 
 * Get used system channel vertex normal buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_VertexNormalDataCount(GvMeshMsg msg);

/** 
 * Get system channel vertex normal ID. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      channel ID. 
 */
GvFx(GvMeshMsgChannelId) GvMeshMsg_VertexNormalId(GvMeshMsg msg);

/** 
 * Set system channel vertex normal state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexNormalState(GvMeshMsg msg, GvMeshMsgChannelState state);

/** 
 * Get system channel vertex normal state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_VertexNormalState(GvMeshMsg msg);

// Vertex texture
/** 
 * Allocate system channel vertex texture buffer. 
 * Number of items in this buffer is always the same as system channel vertex buffer.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateVertexTextureData(GvMeshMsg msg, kAlloc alloc);

/** 
 * Assign system channel vertex texture buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignVertexTextureData(GvMeshMsg msg, kArray1 src);

/** 
 * Get system channel vertex texture buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_VertexTextureData(GvMeshMsg msg);

/** 
 * Clear system channel vertex texture buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearVertexTextureData(GvMeshMsg msg);

/** 
 * Get allocated system channel vertex texture buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedVertexTextureDataCount(GvMeshMsg msg);

/** 
 * Set used system channel vertex texture buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexTextureDataCount(GvMeshMsg msg, kSize count);

/** 
 * Get used system channel vertex texture buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_VertexTextureDataCount(GvMeshMsg msg);

/** 
 * Get system channel vertex texture ID. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      channel ID. 
 */
GvFx(GvMeshMsgChannelId) GvMeshMsg_VertexTextureId(GvMeshMsg msg);

/** 
 * Set system channel vertex texture state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexTextureState(GvMeshMsg msg, GvMeshMsgChannelState state);

/** 
 * Get system channel vertex texture state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_VertexTextureState(GvMeshMsg msg);

// Vertex curvature
/** 
 * Allocate system channel vertex curvature buffer. 
 * Number of items in this buffer is always the same as system channel vertex buffer.
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateVertexCurvatureData(GvMeshMsg msg, kAlloc alloc);

/** 
 * Assign system channel vertex curvature buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignVertexCurvatureData(GvMeshMsg msg, kArray1 src);

/** 
 * Get system channel vertex curvature buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_VertexCurvatureData(GvMeshMsg msg);

/** 
 * Clear system channel vertex curvature buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearVertexCurvatureData(GvMeshMsg msg);

/** 
 * Get allocated system channel vertex curvature buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedVertexCurvatureDataCount(GvMeshMsg msg);

/** 
 * Set used system channel vertex curvature buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexCurvatureDataCount(GvMeshMsg msg, kSize count);

/** 
 * Get used system channel vertex curvature buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_VertexCurvatureDataCount(GvMeshMsg msg);

/** 
 * Get system channel vertex curvature ID. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      channel ID. 
 */
GvFx(GvMeshMsgChannelId) GvMeshMsg_VertexCurvatureId(GvMeshMsg msg);

/** 
 * Set system channel vertex curvature state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetVertexCurvatureState(GvMeshMsg msg, GvMeshMsgChannelState state);

/** 
 * Get system channel vertex curvature state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_VertexCurvatureState(GvMeshMsg msg);

// channel APIs
/** 
 * Get number of channels allocated and used , including both system channels and user channels. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of channels used. 
 */
GvFx(kSize) GvMeshMsg_ChannelCount(GvMeshMsg msg);

/** 
 * Get maximum number of user channels that can be allocated. Currently it is limited to 5 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Maximum number of user channels. 
 */
GvFx(kSize) GvMeshMsg_MaxUserChannelCount(GvMeshMsg msg);

/** 
 * Get number of user channels allocated and used. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @return                      Number of user channels used. 
 */
GvFx(kSize) GvMeshMsg_UserChannelCount(GvMeshMsg msg);

/** 
 * Add a new user channel. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id to be automatically assigned. 
 * @param   type                Channel type. 
 *                              Note that values between 0 and 65535 are reserved for current and future system use. 
 *                              User should only values outside of this range.
 *                              Please see GvMeshMsgChannelType definition for further detail.
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AddUserChannel(GvMeshMsg msg, GvMeshMsgChannelId* id, GvMeshMsgChannelType type);

/** 
 * Get channel type. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Channel type. 
 */
GvFx(GvMeshMsgChannelType) GvMeshMsg_ChannelType(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Set channel state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @param   state               Channel state. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetChannelState(GvMeshMsg msg, GvMeshMsgChannelId id, GvMeshMsgChannelState state);

/** 
 * Get channel state. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Channel state. 
 */
GvFx(GvMeshMsgChannelState) GvMeshMsg_ChannelState(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Set channel flag. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @param   flag                User defined channel flag. User can use flag to indicate how to deserialize channel data.
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetChannelFlag(GvMeshMsg msg, GvMeshMsgChannelId id, k32u flag);

/** 
 * Get channel flag. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      User defined channel flag. 
 *                              Returns 0 by default if not set.
 *                              When flag is invalid, k32U_NULL is returned.                     
 */
GvFx(k32u) GvMeshMsg_ChannelFlag(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Get channel allocated buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Number of allocated items. 
 */
GvFx(kSize) GvMeshMsg_AllocatedChannelDataCount(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Get channel used buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Number of used items. 
 */
GvFx(kSize) GvMeshMsg_UsedChannelDataCount(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Set channel used buffer size (in number of items). 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @param   count               Used item count. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_SetUsedChannelDataCount(GvMeshMsg msg, GvMeshMsgChannelId id, kSize count);

/** 
 * Get channel buffer item type. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Type of items in channel buffer allocated. 
 */
GvFx(kType) GvMeshMsg_ChannelDataType(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Get channel buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Channel buffer. 
 */
GvFx(kArray1) GvMeshMsg_ChannelData(GvMeshMsg msg, GvMeshMsgChannelId id);

/** 
 * Allocate channel buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @param   type                Item type. 
 * @param   size                Number of Item. 
 * @param   alloc               Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AllocateChannelData(GvMeshMsg msg, GvMeshMsgChannelId id, kType type, kSize size, kAlloc alloc);

/** 
 * Assign channel buffer from an existing buffer. 
 * Assigned object will increment source object's reference count automatically
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @param   src                 Source buffer. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_AssignChannelData(GvMeshMsg msg, GvMeshMsgChannelId id, kArray1 src);

/** 
 * Clear channel buffer. 
 *
 * @public                      @memberof GvMeshMsg 
 * @param   msg                 GvMeshMsg object. 
 * @param   id                  Channel id. 
 * @return                      Operation status. 
 */
GvFx(kStatus) GvMeshMsg_ClearChannelData(GvMeshMsg msg, GvMeshMsgChannelId id);

#include <GoVision/Data/GvMeshMsg.x.h>

#endif //GV_MESH_MSG_H
