#ifndef GV_FEATURE_MSG_H
#define GV_FEATURE_MSG_H

#include <GoVision/GvDef.h>
#include <GoVision/GvUtils.h>
#include <kFireSync/Data/kMsgSet.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArrayList.h>
#include <GoVision/Data/GvDataMsg.h>

/**
* Encapsulates a feature message.
*
* @class       GvFeatureMsg
* @extends     GvDataMsg
* @ingroup     GoVision-Data
*/
typedef GvDataMsg GvFeatureMsg;


/**
* Encapsulates a point feature message.
*
* @class       GvPointFeatureMsg
* @extends     GvFeatureMsg
* @ingroup     GoVision-Data
*/
typedef GvFeatureMsg GvPointFeatureMsg;

/** 
 * Sets the position component of the point feature.
 *
 * @public              @memberof GvPointFeatureMsg
 * @param   msg         Message object. 
 * @param   position    Pointer to position (x,y,z) of the feature.
 * @return              Operation status.
 */
GvFx(kStatus) GvPointFeatureMsg_SetPosition(GvPointFeatureMsg msg, const kPoint3d64f* position);

/** 
 * Gets the position component of the point feature.
 *
 * @public              @memberof GvPointFeatureMsg
 * @param   msg         Message object. 
 * @param   position    Pointer to updated position (x,y,z) of the feature.
 * @return              Operation status.
 */
GvFx(kStatus) GvPointFeatureMsg_Position(GvPointFeatureMsg msg, kPoint3d64f* position);

/**
* Encapsulates a line feature message.
*
* @class       GvLineFeatureMsg
* @extends     GvFeatureMsg
* @ingroup     GoVision-Data
*/
typedef GvFeatureMsg GvLineFeatureMsg;
/** 
 * Sets the point position component of the line feature.
 *
 * @public              @memberof GvLineFeatureMsg
 * @param   msg         Message object. 
 * @param   position    Pointer to a position on the line.
 * @return              Operation status.
 */
GvFx(kStatus) GvLineFeatureMsg_SetPosition(GvLineFeatureMsg msg, const kPoint3d64f* position);

/** 
 * Gets the point position component of the line feature.
 *
 * @public              @memberof GvLineFeatureMsg
 * @param   msg         Message object. 
 * @param   position    Pointer to updated point position of the line.
 * @return              Operation status.
 */
GvFx(kStatus) GvLineFeatureMsg_Position(GvLineFeatureMsg msg, kPoint3d64f* position);


/** 
 * Sets the direction component of the line feature.
 *
 * @public              @memberof GvLineFeatureMsg
 * @param   msg         Message object. 
 * @param   direction   Pointer to the direction vector of the line.
 * @return              Operation status.
 */
GvFx(kStatus) GvLineFeatureMsg_SetDirection(GvLineFeatureMsg msg, const kPoint3d64f* direction);

/** 
 * Gets the direction component of the line feature.
 *
 * @public              @memberof GvLineFeatureMsg
 * @param   msg         Message object. 
 * @param   direction      Pointer to updated direction vector of the line.
 * @return              Operation status.
 */
GvFx(kStatus) GvLineFeatureMsg_Direction(GvLineFeatureMsg msg, kPoint3d64f* direction);

/**
* Encapsulates a circle feature message.
*
* @class       GvCircleFeatureMsg
* @extends     GvFeatureMsg
* @ingroup     GoVision-Data
*/
typedef GvFeatureMsg GvCircleFeatureMsg;

/** 
 * Sets the position component of the circle feature.
 *
 * @public              @memberof GvCircleFeatureMsg
 * @param   msg         Message object. 
 * @param   position    Pointer to position (x,y,z) of the feature.
 * @return              Operation status.
 */
GvFx(kStatus) GvCircleFeatureMsg_SetPosition(GvCircleFeatureMsg msg, const kPoint3d64f* position);

/** 
 * Gets the position component of the circle feature.
 *
 * @public              @memberof GvCircleFeatureMsg
 * @param   msg         Message object. 
 * @param   position    Pointer to updated position (x,y,z) of the feature.
 * @return              Operation status.
 */
GvFx(kStatus) GvCircleFeatureMsg_Position(GvCircleFeatureMsg msg, kPoint3d64f* position);

/** 
 * Sets the normal component of the circle feature.
 *
 * @public              @memberof GvCircleFeatureMsg
 * @param   msg         Message object. 
 * @param   normal      Pointer to the normal vector to the circle plane.
 * @return              Operation status.
 */
GvFx(kStatus) GvCircleFeatureMsg_SetNormal(GvCircleFeatureMsg msg, const kPoint3d64f* normal);

/** 
 * Gets the normal component of the circle feature.
 *
 * @public              @memberof GvCircleFeatureMsg
 * @param   msg         Message object. 
 * @param   normal      Pointer to updated normal vector to the circle plane.
 * @return              Operation status.
 */
GvFx(kStatus) GvCircleFeatureMsg_Normal(GvCircleFeatureMsg msg, kPoint3d64f* normal);

/** 
 * Sets the radius component of the circle feature.
 *
 * @public              @memberof GvCircleFeatureMsg
 * @param   msg         Message object. 
 * @param   radius      Radius of the circle feature.
 * @return              Operation status.
 */
GvFx(kStatus) GvCircleFeatureMsg_SetRadius(GvCircleFeatureMsg msg, k64f radius);

/** 
 * Gets the radius component of the circle feature.
 *
 * @public              @memberof GvCircleFeatureMsg
 * @param   msg         Message object. 
 * @return              Circle radius.
 */
GvFx(k64f) GvCircleFeatureMsg_Radius(GvCircleFeatureMsg msg);

/**
* Encapsulates a plane feature message.
*
* @class       GvPlaneFeatureMsg
* @extends     GvFeatureMsg
* @ingroup     GoVision-Data
*/
typedef GvFeatureMsg GvPlaneFeatureMsg;

/** 
 * Sets the normal component of the plane feature.
 *
 * @public              @memberof GvPlaneFeatureMsg
 * @param   msg         Message object. 
 * @param   normal      Pointer to the normal vector [x,y,z] to the plane.
 * @return              Operation status.
 */
GvFx(kStatus) GvPlaneFeatureMsg_SetNormal(GvPlaneFeatureMsg msg, const kPoint3d64f* normal);

/** 
 * Gets the normal component of the plane feature.
 *
 * @public              @memberof GvPlaneFeatureMsg
 * @param   msg         Message object. 
 * @param   normal      Pointer to updated normal vector [x,y,z] to the plane.
 * @return              Operation status.
 */
GvFx(kStatus) GvPlaneFeatureMsg_Normal(GvPlaneFeatureMsg msg, kPoint3d64f* normal);

/** 
 * Sets the distance from the origin component of the plane feature.
 *
 * @public              @memberof GvPlaneFeatureMsg
 * @param   msg         Message object. 
 * @param   distance    Distance from origin of the plane.
 * @return              Operation status.
 */
GvFx(kStatus) GvPlaneFeatureMsg_SetOriginDistance(GvPlaneFeatureMsg msg, const k64f distance);

/** 
 * Gets the distance from origin component of the plane feature.
 *
 * @public              @memberof GvPlaneFeatureMsg
 * @param   msg         Message object. 
 * @return              Plane's distance from origin
 */
GvFx(k64f) GvPlaneFeatureMsg_OriginDistance(GvPlaneFeatureMsg msg);

/**
 * Sets the rendering object within the message.
 *
 * @public               @memberof GvFeatureMsg
 * @param    msg         Message object.
 * @param    object      Rendering object (GdkGraphic until Studio migrates to newer primitives).
 * @return               Operation status.
 */
GvFx(kStatus) GvFeatureMsg_SetRendering(GvFeatureMsg msg, kObject object);

/**
 * Retrieves the rendering object from the message.
 *
 * @public               @memberof GvFeatureMsg 
 * @param    msg         Message object.
 * @return               Rendering object (GdkGraphic until Studio migrates to newer primitives).
 */
GvFx(kObject) GvFeatureMsg_Rendering(GvFeatureMsg msg);

#include <GoVision/Data/GvFeatureMsg.x.h>

#endif
