var SessionLog = require('./SessionLog.js');

var State = {
    WAITING:    0,
    STARTING:   1,
    STARTED:    2,
    STOPPING:   3,
    STOPPED:    4
};

var Session = function(id, scenarioId, ip) {
    this.id = id;
    this.scenarioId = scenarioId;
    this.ip = ip;
    this.startAborted = false;
    this.state = State.WAITING;
    this.error = null;
    this.queuePosition = 0;
    this.emulator = null;
    this.slot = null;
    this.lastActivityCount = 0;
    this.activityTimeRemaining = -1;
    this.log = new SessionLog(id, scenarioId);
};

Session.State = State;

Session.prototype.isRunning = function() {
    return this.state == State.STARTED;
};

Session.prototype.isStarted = Session.prototype.isRunning;

Session.prototype.startConnectionTimer = function(timeout, callback) {
    this._connectionTimeout = timeout;
    this._timeoutCallback = callback;
    this.refreshConnection();
};

Session.prototype.stopConnectionTimer = function() {
    if (this._connectionTimer) {
        clearTimeout(this._connectionTimer);
        this._connectionTimer = null;
    }
};

Session.prototype.refreshConnection = function() {
    this.stopConnectionTimer();

    if (this._connectionTimeout) {
        var self = this;
        this._connectionTimer = setTimeout(function() {
            if (self._timeoutCallback) {
                self._timeoutCallback(self);
            }
        }, this._connectionTimeout);
    }
};

Session.prototype.setActivityTimeout = function(timeout, callback) {
    this._activityTimeout = timeout;
    this._activityTimeoutCallback = callback;
    this.activityTimeRemaining = timeout;
};

Session.prototype.setCrashCallback = function(callback) {
    this._crashCallback = callback;
};

Session.prototype.kickActivity = function() {
    this._lastActivityTime = Date.now();
    this.log.updateActivity(true);
};

Session.prototype.startMonitoring = function() {
    this.activityTimeRemaining = this._activityTimeout;
    this._lastActivityTime = Date.now();

    this.resumeMonitoring();
};

Session.prototype.resumeMonitoring = function() {
    this._lastActivityCount = null;

    var timerCallback = function() {
        this._checkCrash();
        this._checkActivity();
        this._updateActivity();
    }.bind(this);

    this._monitorTimer = setInterval(timerCallback, 500);
};

Session.prototype.pauseMonitoring = function() {
    if (this._monitorTimer) {
        clearInterval(this._monitorTimer);
        this._monitorTimer = null;
    }
};

Session.prototype.stopMonitoring = function() {
    this.pauseMonitoring();
    this.activityTimeRemaining = -1;
};

Session.prototype._checkCrash = function() {
    var isAlive = this.emulator.isAlive();

    if (!isAlive) {
        this.pauseMonitoring();
        if (this.error == null){
            this.error = "Crash";
        }
        this._crashCallback(this);
    }
};

Session.prototype._checkActivity = function() {
    var count = this.emulator.getActivityCount();

    if (this._lastActivityCount == null) {
        // First counter - do not treat as activity
        this._lastActivityCount = count;
        return;
    }

    if (count != this._lastActivityCount) {
        this._lastActivityCount = count;
        this.kickActivity();
    } else {
        this.log.updateActivity(false);
    }
};

Session.prototype._updateActivity = function() {
    var elapsedTime = Date.now() - this._lastActivityTime;
    var remainingTime = this._activityTimeout - Math.round(elapsedTime / 1000);

    if (this._activityTimeout >= 0){
        if (remainingTime < 0) {
            remainingTime = 0;
        }

        this.activityTimeRemaining = remainingTime;
        if (remainingTime == 0) {
            this.pauseMonitoring();
            this.error = "ActivityTimeout";

            // Add a brief grace period to allow the UI to poll a time of 0
            setTimeout(function() {
                if (this._activityTimeoutCallback) {
                    this._activityTimeoutCallback(this);
                }
                }.bind(this), 10000);
        }
    }
};

module.exports = Session;
