#!/usr/bin/python
# This file is part of the FireSync Project Generator.
# Copyright (C) 2008-2016 by LMI Technologies Inc.  All rights reserved.
# Distributed under the terms of the MIT License.
# Redistributed files must retain the above copyright notice.

import os
from . import Project
from . import WriterBase
from . import Utils

def GetTriplet():
    return 'arm-unknown-elf'

def MakeFileSuffix(platformName):
    return '-WrWbu' if platformName == 'Arm7u' else '-WrWb'

def GnuToolsVersion():
    return '4.9.4'

def GnuToolsPatch():
    return 'p23'

def ToolsPath(isWindows):
    if isWindows:
        return 'C:' + os.path.sep + 'tools'
    else:
        return os.path.sep + 'tools'

def ToolsPrefix(isWindows):
    return os.path.join(ToolsPath(isWindows), 'GccArm7_' + GnuToolsVersion() + '-' + GnuToolsPatch())

def WindRiverPrefix(isWindows):
    if isWindows:
        return 'C:' + os.path.sep + 'WrsVxw_6.9.4.6'
    else:
        return os.path.sep + 'WrsVxw_6.9.4.6'

def CrossToolsPrefix(isWindows):
    return os.path.join(ToolsPrefix(isWindows), GetTriplet(), 'bin', GetTriplet() + '-')

def PackageName():
    return 'vxworks-6.9'

def GeneratorEnvString():
    return 'K_REGENERATOR'

def DebugInfoEnvString():
    return 'K_ARM7_VXWORKS_DEBUG_INFO_ENABLED'

def ReGeneratorEnabled():
    try:
        return os.environ[GeneratorEnvString()] != '0'
    except KeyError:
        #TODO: enable by default, after feature review
        return False

def DebugInfoEnabled():
    try:
        return os.environ[DebugInfoEnvString()] != '0'
    except KeyError:
        return False

def GmakePath(projectFileName):
    herePath = os.path.dirname(os.path.realpath(__file__))
    gmakePath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'extern', 'Make', 'gmake.exe'))
    return os.path.relpath(gmakePath, os.path.dirname(projectFileName))

#
# Define how the projects are (re-)built
#
def BuildCmd(project, config):
    if config.UsePrimaryTools == '1':
        return '%s -p %s %s -f %s%s.mk config=%s' % \
            (os.path.join(WindRiverPrefix(True), 'wrenv.exe'),
            PackageName(),
            GmakePath(project.FileName),
            project.Name,
            MakeFileSuffix(config.Platform),
            config.Name)
    else:
        return '%s -f %s%s.mk config=%s' % \
            (GmakePath(project.FileName),
            project.Name,
            MakeFileSuffix(config.Platform),
            config.Name)

def RebuildCmd(project, config):
    if config.UsePrimaryTools == '1':
        return '%s -p %s %s -f %s%s.mk config=%s clean all' % \
            (os.path.join(WindRiverPrefix(True), 'wrenv.exe'),
            PackageName(),
            GmakePath(project.FileName),
            project.Name,
            MakeFileSuffix(config.Platform),
            config.Name)
    else:
        return '%s -f %s%s.mk config=%s clean all' % \
            (GmakePath(project.FileName),
            project.Name,
            MakeFileSuffix(config.Platform),
            config.Name)

def CleanCmd(project, config):
    if config.UsePrimaryTools == '1':
        return '%s -p %s %s -f %s%s.mk config=%s clean' % \
            (os.path.join(WindRiverPrefix(True), 'wrenv.exe'),
            PackageName(),
            GmakePath(project.FileName),
            project.Name,
            MakeFileSuffix(config.Platform),
            config.Name)
    else:
        return '%s -f %s%s.mk config=%s clean' % \
            (GmakePath(project.FileName),
            project.Name,
            MakeFileSuffix(config.Platform),
            config.Name)

class WrWbMkBase(WriterBase.WriterBase):

    _libDirs = [
        os.path.join('$(TOOLS_PATH)', GetTriplet(), 'lib', 'gcc', GetTriplet(), GnuToolsVersion()),
        os.path.join('$(TOOLS_PATH)', GetTriplet(), GetTriplet(), 'lib'),
    ]

    _compilerFlags = [
        #'-mfpu=neon',
        '-mfloat-abi=softfp',
        '-Wall',
        '-Wfloat-conversion',
        '-fno-zero-initialized-in-bss',
        '-Wno-missing-braces', # this is for GCC bug 53119 -- should be removed when GCC >= 5.x is abundant
    ]

    _cFlags = [
        '-std=gnu99',
        '-Wno-unused-variable',
        '-Wno-unused-parameter',
        '-Wno-unused-value',
    ]

    _cxx11Flags = [
        '-std=c++11',
        '-fexceptions',
        '-Wno-literal-suffix', # this is to suppress a C++-11 warning in a VxWorks header -- should be removed later
    ]

    _cxxFlags = [
        '-std=c++14',
        '-fexceptions',
        '-Wno-literal-suffix', # this is to suppress a C++-11 warning in a VxWorks header -- should be removed later
    ]

    _linkerFlags = [
        '-X',
        '-EL',
    ]

    _compilerFlagTemplates = {
        'Symbols': [
            '-g',
        ],

        'Optimize': [
            '-O2',
        ],

        'Optimize3': [
            '-O2', # don't trust aggressive optimizations using GCC.
        ],

        'CortexA9': [
            '-mcpu=cortex-a9',
            '-march=armv7-a',
        ],

        'LongCalls': [
            '-mlong-calls',
        ],

        'NoBuiltin': [
            '-fno-builtin',
        ],
    }

    def __init__(self, platforms, userPlatforms, projects, userProjects, name, suffix):
        WriterBase.WriterBase.__init__(self, platforms, userPlatforms, projects, userProjects)
        self.name = name
        self.suffix = suffix
        self.regeneratorEnabled = ReGeneratorEnabled()
        self.debugInfoEnabled = DebugInfoEnabled()

    @property
    def Name(self):
        return self.name

    @property
    def TempName(self):
        return self.name.lower()

    @property
    def SolutionExt(self):
        return '.mk'

    @property
    def ProjectExt(self):
        return '.mk'

    @property
    def ProjectSuffix(self):
        return self.Suffix + self.ProjectExt

    @property
    def SolutionSuffix(self):
        return self.Suffix + self.SolutionExt

    @property
    def Suffix(self):
        return self.suffix

    def ForceCxx11(self, config):
        return True if 'c++11' in config.Template.split('|') else False

    def SelectCompilerFlags(self, config, selectionString):
        compilerFlags = []
        components = selectionString.split('|')

        for component in components:
            for tmp in self._compilerFlagTemplates:
                if tmp == component:
                    for flag in self._compilerFlagTemplates[tmp]:
                        compilerFlags.append(flag)

        if 'NoNeon' not in components: # TODO: change default to no-NEON (if 'Neon' in components)
            compilerFlags.append('-mfpu=neon')
        else:
            compilerFlags.append('-mfpu=vfpv3')

        for flag in self._compilerFlags:
            compilerFlags.append(flag)

        return compilerFlags

    def SelectSourceCompilerFlags(self, config, sourceName):
        templates = config.Template.split('|')

        for sourceTemplate in config.SourceTemplates:
            if sourceTemplate.Name == sourceName:
                if sourceTemplate.Add != 'Exclude':
                    if sourceTemplate.Add != '':
                        for component in sourceTemplate.Add.split('|'):
                            templates.append(component)

                    if sourceTemplate.Remove != '':
                        for component in sourceTemplate.Remove.split('|'):
                            templates.remove(component)

        templateStr = '|'.join(templates)
        sourceCompilerFlags = ' '.join(self.SelectCompilerFlags(config, templateStr))
        defaultCompilerFlags = ' '.join(self.SelectCompilerFlags(config, config.Template))

        if sourceCompilerFlags != defaultCompilerFlags:
            return sourceCompilerFlags
        return ''

    def HasAsm(self, project, config):
        for src in self.FilterSources(project, config):
            source = src.Name
            extension = os.path.splitext(source)[1]
            if extension == '.s' or extension == '.S' or extension == '.asm':
                return True
        return False

    def IsSmp(self, project, config):
        return config.Platform == 'Arm7'

    def UseDebug(self, project, config):

        return True if 'Symbols' in config.Template.split('|') else False

    def StdCxxPath(self, project, config):
        herePath = os.path.dirname(os.path.realpath(__file__))
        stdExternPath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'extern', 'libstdc++-v3', 'arm7'))
        return os.path.relpath(stdExternPath, os.path.dirname(project.FileName))

    def SrcPath(self, project, config):
        herePath = os.path.dirname(os.path.realpath(__file__))
        wrExternPath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'extern', 'WindRiver'))
        return os.path.relpath(wrExternPath, os.path.dirname(project.FileName))

    def LsPath(self, project, config, scriptName):
        herePath = os.path.dirname(os.path.realpath(__file__))
        lscriptPath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'kHardwareZynq7k', 'kHardwareZynq7k', 'Link', scriptName))
        return os.path.relpath(lscriptPath, os.path.dirname(project.FileName))

    def LsPathDynamic(self, project, config):
        return self.LsPath(project, config, 'dynamic.ld')

    def LsPathFramework(self, project, config):
        return self.LsPath(project, config, 'framework.ld')

    def VsbPath(self, project, config):

        if self.IsSmp(project, config):
            return os.path.join(self.SrcPath(project, config), 'Zynq7kVsb')
        else:
            return os.path.join('$(VXW_PATH)', 'vxworks-6.9', 'target', 'lib')

    def VipPath(self, project, config):

        vsbName = 'Zynq7kImg' if self.IsSmp(project, config) else 'Zynq7kBootImg'
        suffix = 'd' if self.UseDebug(project, config) else ''

        return os.path.join(self.SrcPath(project, config), vsbName + suffix, vsbName + suffix + '.wpj')

    def SelectWpj(self, project, config):

        for component in config.Template.split('|'):
            subcomponents = component.split('=')
            if len(subcomponents) == 1 and subcomponents[0] == 'vxWorksPrj':
                return self.VipPath(project, config)
            elif len(subcomponents) == 2 and subcomponents[0] == 'vxWorksPrj':
                return subcomponents[1]

        return None

    def SelectFramework(self, project, config):

        for component in config.Template.split('|'):
            subcomponents = component.split('=')
            if len(subcomponents) == 1 and subcomponents[0] == 'FrameworkApp':
                return os.path.join(config.BinDir, 'kFrameworkZynq7k.kapp')
            elif len(subcomponents) == 2 and subcomponents[0] == 'Framework':
                return subcomponents[1]

        return None

    def FilterSources(self, project, config):
        sources = []
        for src in self.SelectSources(project):

            excluded = False
            for sourceTemplate in config.SourceTemplates:
                if sourceTemplate.Name == src.Name:
                    if sourceTemplate.Add == 'Exclude':
                        excluded = True

            if not excluded:
                sources.append(src)

        return sources

    def WriteProject(self, project):
        return self.WriteProjectEx(project)

    def WriteProjectEx(self, project, forceUpdate=False):

        fileName = '%s%s' % (os.path.splitext(project.FileName)[0], self.ProjectSuffix)

        with Utils.Output(fileName, Utils.EnvironmentStyle.Dollar, True, forceUpdate=forceUpdate) as output:

            utilsPath = os.path.normpath(os.path.join(os.path.dirname(os.path.realpath(__file__)), '..'))
            utilsRelPath = os.path.relpath(utilsPath, os.path.dirname(project.FileName))

            configurations = self.SelectConfigurations(project)

            if len(configurations) != 0:

                output.Format('\n')

                if self.regeneratorEnabled:
                    output.Format('export %s := 1\n' % (GeneratorEnvString()))
                    output.Format('export %s := %u\n' % (DebugInfoEnvString(), int(self.debugInfoEnabled)))
                    if configurations[0].UsePrimaryTools == '1':
                        output.Format('export K_LICENSE := 1\n')
                    output.Format('\n')

                output.Format('ifeq ($(OS)$(os), Windows_NT)\n')
                output.Format('\tVXW_PATH     := %s\n' % (WindRiverPrefix(True)))
                output.Format('\tTOOLS_PATH   := %s\n' % (ToolsPrefix(True)))
                output.Format('\tCROSS_PREFIX := %s\n' % (CrossToolsPrefix(True)))
                output.Format('\tCROSS_SUFFIX := .exe\n')
                output.Format('\tPYTHON       := python\n')
                output.Format('else\n')
                output.Format('\tVXW_PATH     := %s\n' % (WindRiverPrefix(False)))
                output.Format('\tTOOLS_PATH   := %s\n' % (ToolsPrefix(False)))
                output.Format('\tCROSS_PREFIX := %s\n' % (CrossToolsPrefix(False)))
                output.Format('\tCROSS_SUFFIX := \n')
                output.Format('\tPYTHON       := python3\n')
                output.Format('endif\n\n')

                output.Format('C_COMPILER := $(CROSS_PREFIX)gcc$(CROSS_SUFFIX)\n')
                output.Format('CXX_COMPILER := $(CROSS_PREFIX)g++$(CROSS_SUFFIX)\n')
                output.Format('LINKER := $(CROSS_PREFIX)ld$(CROSS_SUFFIX)\n')
                output.Format('ARCHIVER := $(CROSS_PREFIX)ar$(CROSS_SUFFIX)\n')
                output.Format('OBJ_NAMES := $(CROSS_PREFIX)nm$(CROSS_SUFFIX)\n')
                output.Format('\n')

                output.Format('TAR_GEN := $(PYTHON) %s\n' % os.path.join(utilsRelPath, 'kTarGen.py'))
                output.Format('MUNCH := $(PYTHON) %s\n' % os.path.join(utilsRelPath, 'kMunch.py'))
                output.Format('MKDIR_P := $(PYTHON) %s mkdir_p\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                output.Format('RM_F := $(PYTHON) %s rm_f\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                output.Format('RM_RF := $(PYTHON) %s rm_rf\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                output.Format('CP := $(PYTHON) %s cp\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                if self.regeneratorEnabled:
                    output.Format('KGENERATOR := $(PYTHON) %s\n' % os.path.join(utilsRelPath, 'kGenerator.py')) 
                output.Format('\n')

                output.Format('ifndef verbose\n')
                output.Format('\tSILENT := @\n')
                output.Format('endif\n\n')

                output.Format('ifndef config\n')
                output.Format('\tconfig := %s\n' % (configurations[0].Name))
                output.Format('endif\n\n')

                output.Format('ifeq ($(word 1, $(subst ., ,$(MAKE_VERSION))), 3)\n')
                output.Format('.PHONY: err\n')
                output.Format('err: ; $(error GNU Make version 3.x is not supported)\n')
                output.Format('endif\n\n')

                output.Format('# We require tools to be installed according to specific conventions (see manuals).\n')
                output.Format('# Tool prerequisites may change between major releases; check and report.\n')
                output.Format('ifeq ($(shell $(C_COMPILER) --version),)\n')
                output.Format('.PHONY: tc_err\n')
                output.Format('tc_err:\n')
                output.Format('\t$(error Cannot build because of missing prerequisite; please install)\n')
                output.Format('endif\n\n')

                for config in configurations:

                    compilerFlags = self.SelectCompilerFlags(config, config.Template)
                    isSmp = config.Platform == 'Arm7'
                    configType = config.Type
                    frameworkName = self.SelectFramework(project, config)

                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    if configType == 'Executable' or configType == 'TarExecutable':
                        targetName = os.path.join(config.BinDir, config.Target)
                        output.Format('\tTARGET := %s\n' % (targetName))

                        if frameworkName is not None:
                            kextName = os.path.splitext(targetName)[0] + '.kext'
                            output.Format('\tINTERMEDIATES := %s\n' % (kextName))

                        else:
                            output.Format('\tINTERMEDIATES := \n')

                    elif configType == 'Shared':
                        targetName = os.path.join(config.BinDir, config.Target)
                        output.Format('\tTARGET := %s\n' % (targetName))
                        output.Format('\tINTERMEDIATES := \n')

                    else:
                        targetName = os.path.join(config.LibDir, config.Target)
                        output.Format('\tTARGET := %s\n' % (targetName))
                        output.Format('\tINTERMEDIATES := \n')

                    output.Format('\tOBJ_DIR := %s/%s-%s-%s\n' % (config.TempDir, project.Name, self.TempName, config.Name))
                    output.Format('\tPREBUILD := %s\n' % (config.Prebuild))
                    output.Format('\tPOSTBUILD := %s\n' % (config.Postbuild))

                    output.Format('\tCOMPILER_FLAGS :=')
                    for flag in compilerFlags:
                        output.Format(' %s' % (flag))
                    output.Format('\n')

                    output.Format('\tC_FLAGS :=')
                    for flag in self._cFlags:
                        output.Format(' ')
                        output.Format('%s' % (flag))
                    output.Format('\n')

                    output.Format('\tINCLUDE_DIRS :=')

                    if config.UsePrimaryTools == '1':
                        output.Format(' -nostdinc')

                        for dir in [
                            os.path.join('$(TOOLS_PATH)', GetTriplet(), 'lib', 'gcc', GetTriplet(), GnuToolsVersion(), 'include'),
                            os.path.join(self.StdCxxPath(project, config), 'include'),
                            os.path.join('$(TOOLS_PATH)', GetTriplet(), GetTriplet(), 'include', 'c++', GnuToolsVersion()),
                            os.path.join('$(TOOLS_PATH)', GetTriplet(), GetTriplet(), 'include', 'c++', GnuToolsVersion(), GetTriplet()),
                        ]:
                            output.Format(' -I%s' % (dir))

                        for dir in [
                            os.path.join('$(VXW_PATH)', 'vxworks-6.9', 'target', 'h'),
                            os.path.join('$(VXW_PATH)', 'vxworks-6.9', 'target', 'h', 'wrn', 'coreip'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'ipcom', 'include'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'ipcom', 'config'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'ipcom', 'port', 'vxworks', 'include'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'ipcom', 'port', 'vxworks', 'config'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'ipappl', 'include'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'ipappl', 'config'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'vxcompat', 'include'),
                            os.path.join('$(VXW_PATH)', 'components', 'ip_net2-6.9', 'vxcompat', 'config'),
                        ]:
                            output.Format(' -I%s' % (dir))

                    for dir in config.IncludeDirs:
                        output.Format(' -I%s' % (dir))
                    output.Format('\n')

                    if self.HasAsm(project, config):
                        output.Format('\tASM_INCLUDE_DIRS :=')
                        for dir in config.IncludeDirs:
                            output.Format(' ')
                            output.Format('-Wa,-I%s' % (dir))
                        output.Format('\n')

                    output.Format('\tDEFINES :=')
                    if config.UsePrimaryTools == '1':

                        vsbPath = self.VsbPath(project, config)

                        output.Format(' -D%s' % ('CPU=_VX_ARMARCH7'))
                        output.Format(' -D%s' % ('TOOL_FAMILY=gnu'))
                        output.Format(' -D%s' % ('TOOL=gnu'))
                        output.Format(' -D%s' % ('_WRS_KERNEL'))
                        output.Format(' -D%s' % ('_VX_CPU=_VX_ARMARCH7'))

                        if isSmp:
                            output.Format(' -D%s' % ('_WRS_VX_SMP'))
                            output.Format(' -D%s' % ('_WRS_CONFIG_SMP'))

                        output.Format(' -D_VSB_CONFIG_FILE=\\"%s\\"' % (Utils.Format(os.path.join(vsbPath, 'h', 'config', 'vsbConfig.h'))), False)

                        output.Format(' -D%s' % ('ARMEL'))
                        output.Format(' -D%s' % ('ARM_USE_VFP'))
                        output.Format(' -D%s' % ('IP_PORT_VXWORKS=69'))

                    else:
                        output.Format(' -D%s' % ('_WRS_KERNEL'))

                    for define in config.Defines:
                        output.Format(' -D%s' % (define))
                    output.Format('\n')

                    output.Format('\tCXX_FLAGS :=')
                    cxxFlags = self._cxx11Flags if self.ForceCxx11(config) else self._cxxFlags
                    for flag in cxxFlags:
                        output.Format(' ')
                        output.Format('%s' % (flag))
                    output.Format('\n')

                    output.Format('\tLINKER_FLAGS :=')
                    for flag in self._linkerFlags:
                        output.Format(' %s' % (flag))
                    if not self.debugInfoEnabled:
                        output.Format(' --strip-debug')
                    output.Format('\n')

                    output.Format('\tLIB_DIRS :=')
                    for libDir in config.LibraryDirs:
                        output.Format(' -L%s' % (libDir))
                    for libDir in self._libDirs:
                        output.Format(' -L%s' % (libDir))
                    output.Format('\n')

                    output.Format('\tLIBS :=')
                    if len(config.Libs) != 0:
                        output.Format(' --whole-archive')
                        output.Format(' --start-group')
                        for lib in config.Libs:
                            output.Format(' -l%s' % (lib))
                        if self.SelectWpj(project, config) is not None and self.IsSmp(project, config):
                            output.Format(' -lstdc++')
                        output.Format(' --end-group')
                        output.Format(' --no-whole-archive')
                    output.Format('\n')

                    output.Format('\tOBJECTS := ')
                    first = True
                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.c' or extension == '.cpp' or extension == '.s' or extension == '.S' or extension == '.asm':
                            if not first:
                                output.Format(' \\\n\t', False)
                            first = False
                            output.Format('%s/%s-%s-%s/%s.o' % (config.TempDir, project.Name, self.TempName, config.Name, os.path.splitext(os.path.basename(src.Name))[0]))
                    output.Format('\n')

                    output.Format('\tDEP_FILES = ')
                    first = True
                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.c' or extension == '.cpp':
                            if not first:
                                output.Format(' \\\n\t', False)
                            first = False
                            output.Format('%s/%s-%s-%s/%s.d' % (config.TempDir, project.Name, self.TempName, config.Name, os.path.splitext(os.path.basename(src.Name))[0]))
                    output.Format('\n')

                    #
                    # List output files of dependent projects as target dependencies. 
                    # Targets from dependent projects are listed as dependencies here by matching the configuration name
                    # 
                    output.Format('\tTARGET_DEPS = ')
                    first = True
                    for dep in self.SelectDependencies(project):
                        depConfigs = self.SelectConfigurations(dep)
                        for depConfig in depConfigs:
                            if depConfig.Name == config.Name:

                                projectPath = os.path.relpath(os.path.dirname(dep.FileName), os.path.dirname(project.FileName))
                                target = os.path.join(projectPath, (depConfig.LibDir if depConfig.Type == 'Library' else depConfig.BinDir), depConfig.Target)

                                if not first:
                                    output.Format(' \\\n\t', False)
                                first = False
                                output.Format('%s' % (target))

                    output.Format('\n\n')
                    output.Format('endif\n\n')

                output.Format('.PHONY: all all-obj all-dep\n\n')

                output.Format('all: $(OBJ_DIR)\n')
                output.Format('\t$(PREBUILD)\n')
                output.Format('\t$(SILENT) $(MAKE) -f %s%s all-dep\n' % (project.Name, self.ProjectSuffix))
                output.Format('\t$(SILENT) $(MAKE) -f %s%s all-obj\n' % (project.Name, self.ProjectSuffix))
                output.Format('\n')

                output.Format('clean:\n')
                output.Format('\t$(info Cleaning $(OBJ_DIR))\n')
                output.Format('\t$(SILENT) $(RM_RF) $(OBJ_DIR)\n')
                output.Format('\t$(info Cleaning $(TARGET) $(INTERMEDIATES))\n')
                output.Format('\t$(SILENT) $(RM_F) $(TARGET) $(INTERMEDIATES)\n')
                output.Format('\n')

                output.Format('all-obj: $(OBJ_DIR) $(TARGET)\n')
                output.Format('all-dep: $(OBJ_DIR) $(DEP_FILES)\n')
                output.Format('\n')

                if self.regeneratorEnabled:
                    output.Format('$(OBJ_DIR): %s%s\n' % (project.Name, self.ProjectSuffix))
                    output.Format('\t$(info Cleanse $(OBJ_DIR))\n')
                    output.Format('\t$(SILENT) $(RM_RF) $(OBJ_DIR)\n')
                    output.Format('\t$(SILENT) $(MKDIR_P) $(OBJ_DIR)\n')
                    output.Format('\n')
                else:
                    output.Format('$(OBJ_DIR):\n')
                    output.Format('\t$(SILENT) $(MKDIR_P) $@\n')
                    output.Format('\n')

                if self.regeneratorEnabled:
                    solFileNameRel = os.path.relpath(project.Solution.FileName, os.path.dirname(project.FileName))
                    output.Format('ifeq ($(MAKECMDGOALS),)\n')
                    output.Format('ifeq ($(MAKE_RESTARTS),)\n')
                    output.Format('%s%s: %s.xml %s %s\n' % (project.Name, self.ProjectSuffix, project.Name, solFileNameRel, os.path.join(utilsRelPath, 'Generator', 'WrWbMk.py')))
                    output.Format('\t$(info Regen%s %s%s)\n' % (configurations[0].Platform, project.Name, self.ProjectSuffix))
                    output.Format('\t$(SILENT) $(KGENERATOR) --writers=WrWbMk --platforms=%s --project=%s %s\n' % (configurations[0].Platform, project.Name, solFileNameRel))
                    output.Format('endif\n')
                    output.Format('endif\n')
                    output.Format('\n')

                for config in configurations:

                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    configType = config.Type
                    frameworkName = self.SelectFramework(project, config)
                    wpjName = self.SelectWpj(project, config)

                    if configType == 'Executable' and wpjName is not None:
                        target = os.path.join(config.BinDir, config.Target)

                        tmpDir = '%s/%s-%s-%s' % (config.TempDir, project.Name, self.TempName, config.Name)
                        imageProject = os.path.join(tmpDir, '%s.wpj' % (project.Name))
                        imageOutput = os.path.join(tmpDir, 'default', 'vxWorks')
                        partialImage = '%s/%s-partial.o' % (tmpDir, project.Name)
                        linkerScript = self.LsPathFramework(project, config)

                        output.Format('%s: $(OBJECTS) $(TARGET_DEPS) %s\n' % (partialImage, linkerScript))
                        output.Format('\t$(info LdArm7 %s)\n' % (partialImage))
                        output.Format('\t$(SILENT) $(LINKER) $(LIB_DIRS) $(LINKER_FLAGS) -r -nostdlib -T%s -o %s $(OBJECTS) $(LIBS)\n' % (linkerScript, partialImage))
                        output.Format('\n')

                        output.Format('%s: %s %s\n' % (target, partialImage, imageProject))
                        output.Format('\t$(info VxpArm7 %s)\n' % (imageProject))
                        output.Format('\t$(SILENT) (cd %s && MAKEFLAGS= SUB_TARGETS="../%s-partial.o" vxprj build %s.wpj)\n' % (tmpDir, project.Name, project.Name))
                        output.Format('\t$(SILENT) $(CP) %s %s\n' % (imageOutput, target))
                        output.Format('\t$(POSTBUILD)\n')
                        output.Format('\n')

                    elif ((configType == 'Executable' or configType == 'TarExecutable') and frameworkName is not None):
                        target = os.path.join(config.BinDir, config.Target)
                        extName = os.path.splitext(target)[0] + '.kext'

                        tmpDir = '%s/%s-%s-%s' % (config.TempDir, project.Name, self.TempName, config.Name)
                        partialImage = '%s/%s-partial.o' % (tmpDir, project.Name)
                        munchSource = '%s/%s-munch.c' % (tmpDir, project.Name)
                        munchObj = '%s/%s-munch.o' % (tmpDir, project.Name)
                        mapFile = '%s/%s.map' % (tmpDir, os.path.splitext(os.path.basename(config.Target))[0])
                        linkerScript = self.LsPathDynamic(project, config)

                        output.Format('%s: %s %s\n' % (target, frameworkName, extName))
                        output.Format('\t$(info TarArm7 %s)\n' % (target))
                        output.Format('\t$(SILENT) $(TAR_GEN) %s %s %s\n' % (frameworkName, extName, target))
                        output.Format('\t$(POSTBUILD)\n')
                        output.Format('\n')

                        output.Format('%s: $(OBJECTS) $(TARGET_DEPS) %s %s\n' % (extName, munchObj, linkerScript))
                        output.Format('\t$(info LdArm7 %s)\n' % (extName))
                        output.Format('\t$(SILENT) $(LINKER) $(LINKER_FLAGS) -r -nostdlib -T%s -o %s %s %s --Map=%s\n' % (linkerScript,
                            extName, partialImage, munchObj, mapFile))
                        output.Format('\n')

                        output.Format('%s: %s\n' % (munchObj, munchSource))
                        output.Format('\t$(info GccArm7 %s)\n' % (munchSource))
                        output.Format('\t$(SILENT) $(C_COMPILER) $(COMPILER_FLAGS) $(C_FLAGS) -o %s -c %s -MMD -MP\n' % (munchObj, munchSource))
                        output.Format('\n')

                        output.Format('%s: %s\n' % (munchSource, partialImage))
                        output.Format('\t$(info MunchArm7 %s)\n' % (munchSource))
                        output.Format('\t$(SILENT) $(MUNCH) $(OBJ_NAMES) %s %s\n' % (partialImage, munchSource))
                        output.Format('\n')

                        output.Format('%s: $(OBJECTS) $(TARGET_DEPS)\n' % (partialImage))
                        output.Format('\t$(info LdArm7 %s)\n' % (partialImage))
                        output.Format('\t$(SILENT) $(LINKER) $(LIB_DIRS) $(LINKER_FLAGS) -r -nostdlib -o %s $(OBJECTS) $(LIBS)\n' % (partialImage))
                        output.Format('\n')

                    elif configType == 'Shared':

                        tmpDir = '%s/%s-%s-%s' % (config.TempDir, project.Name, self.TempName, config.Name)
                        target = os.path.join(config.BinDir, config.Target)
                        partialImage = '%s/%s-partial.o' % (tmpDir, project.Name)
                        munchSource = '%s/%s-munch.c' % (tmpDir, project.Name)
                        munchObj = '%s/%s-munch.o' % (tmpDir, project.Name)
                        mapFile = '%s/%s.map' % (tmpDir, os.path.splitext(os.path.basename(config.Target))[0])
                        linkerScript = self.LsPathDynamic(project, config)

                        output.Format('%s: $(OBJECTS) $(TARGET_DEPS) %s %s %s\n' % (target, partialImage, munchObj, linkerScript))
                        output.Format('\t$(info LdArm7 %s)\n' % (target))
                        output.Format('\t$(SILENT) $(LINKER) $(LINKER_FLAGS) -r -nostdlib -T%s -o %s %s %s --Map=%s\n' % (linkerScript,
                            target, partialImage, munchObj, mapFile))
                        output.Format('\t$(POSTBUILD)\n')
                        output.Format('\n')

                        output.Format('%s: %s\n' % (munchObj, munchSource))
                        output.Format('\t$(info GccArm7 %s)\n' % (munchSource))
                        output.Format('\t$(SILENT) $(C_COMPILER) $(COMPILER_FLAGS) $(C_FLAGS) -o %s -c %s -MMD -MP\n' % (munchObj, munchSource))
                        output.Format('\n')

                        output.Format('%s: %s\n' % (munchSource, partialImage))
                        output.Format('\t$(info MunchArm7 %s)\n' % (munchSource))
                        output.Format('\t$(SILENT) $(MUNCH) $(OBJ_NAMES) %s %s\n' % (partialImage, munchSource))
                        output.Format('\n')

                        output.Format('%s: $(OBJECTS) $(TARGET_DEPS)\n' % (partialImage))
                        output.Format('\t$(info LdArm7 %s)\n' % (partialImage))
                        output.Format('\t$(SILENT) $(LINKER) $(LIB_DIRS) $(LINKER_FLAGS) -r -nostdlib -o %s $(OBJECTS) $(LIBS)\n' % (partialImage))
                        output.Format('\n')

                    elif configType == 'Library':
                        target = os.path.join(config.LibDir, config.Target)

                        output.Format('%s: $(OBJECTS)\n' % (target))
                        output.Format('\t$(info ArArm7 %s)\n' % (target))
                        output.Format('\t$(SILENT) $(RM_F) %s\n' % (target))
                        output.Format('\t$(SILENT) $(ARCHIVER) rcs %s $(OBJECTS)\n' % (target))
                        output.Format('\t$(POSTBUILD)\n')
                        output.Format('\n')

                    else:
                        raise Exception('Unknown configuration type')

                    output.Format('endif\n\n')

                for config in configurations:

                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    tmpDir = '%s/%s-%s-%s' % (config.TempDir, project.Name, self.TempName, config.Name)
                    wpjName = self.SelectWpj(project, config)

                    if configType == 'Executable' and wpjName is not None:

                        wpjToName = os.path.join(tmpDir, '%s.wpj' % (project.Name))
                        wpjFromTempPath = os.path.relpath(os.path.join(os.path.dirname(project.FileName), wpjName),
                                            os.path.join(os.path.dirname(project.FileName), tmpDir))

                        output.Format('%s: %s\n' % (wpjToName, wpjName))
                        output.Format('\t$(info VxpArm7)\n')
                        output.Format('\t$(SILENT) (cd %s && MAKEFLAGS= vxprj copy %s %s.wpj)\n' % (tmpDir, wpjFromTempPath, project.Name))
                        output.Format('\n')

                    for src in self.SelectSources(project):
                        source = src.Name
                        extension = os.path.splitext(source)[1]

                        if extension == '.c' or extension == '.cpp':
                            ppdSource = '%s/%s.d' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])
                            objectFile = '%s/%s.o' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])

                            output.Format('%s %s: ' % (objectFile, ppdSource))
                            output.Format('%s\n' % (src.Name))

                            compilerFlags = self.SelectSourceCompilerFlags(config, src.Name)

                            if len(compilerFlags) == 0:
                                if extension == '.c':
                                    output.Format('\t$(info GccArm7 %s)\n' % src.Name)
                                    output.Format('\t$(SILENT) $(C_COMPILER) $(COMPILER_FLAGS) $(C_FLAGS) $(DEFINES) $(INCLUDE_DIRS) -o %s.o -c %s -MMD -MP\n' % \
                                        (os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                                else:
                                    output.Format('\t$(info GccArm7 %s)\n' % src.Name)
                                    output.Format('\t$(SILENT) $(CXX_COMPILER) $(COMPILER_FLAGS) $(CXX_FLAGS) $(DEFINES) $(INCLUDE_DIRS) -o %s.o -c %s -MMD -MP\n' % \
                                        (os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                            else:
                                if extension == '.c':
                                    output.Format('\t$(info Custom GccArm7 %s)\n' % src.Name)
                                    output.Format('\t$(SILENT) $(C_COMPILER) $(C_FLAGS) %s $(DEFINES) $(INCLUDE_DIRS) -o %s.o -c %s -MMD -MP\n' % \
                                        (compilerFlags, os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                                else:
                                    output.Format('\t$(info Custom GccArm7 %s)\n' % src.Name)
                                    output.Format('\t$(SILENT) $(CXX_COMPILER) $(CXX_FLAGS) %s $(DEFINES) $(INCLUDE_DIRS) -o %s.o -c %s -MMD -MP\n' % \
                                        (compilerFlags, os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                            output.Format('\n')

                        elif extension == '.s' or extension == '.S' or extension == '.asm':
                            objectFile = '%s/%s.o' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])

                            output.Format('%s: ' % (objectFile))
                            output.Format('%s\n' % (src.Name))

                            output.Format('\t$(info GccArm7 %s)\n' % src.Name)
                            output.Format('\t$(SILENT) $(C_COMPILER) $(COMPILER_FLAGS) -x assembler-with-cpp $(DEFINES) $(INCLUDE_DIRS) $(ASM_INCLUDE_DIRS) -o %s.o -c %s\n' % (
                                os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                            output.Format('\n')

                    output.Format('endif\n\n')

                output.Format('ifeq ($(MAKECMDGOALS),all-obj)\n\n')
                for config in configurations:
                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.c' or extension == '.cpp':
                            ppdSource = '%s/%s-%s-%s/%s.d' % (config.TempDir, project.Name, self.TempName, config.Name, os.path.splitext(os.path.basename(src.Name))[0])
                            output.Format('include %s\n' % (ppdSource))

                    output.Format('\n')
                    output.Format('endif\n\n')
                output.Format('endif\n\n')

    def WriteSolution(self, solution):

        fileName = '%s%s' % (os.path.join(os.path.dirname(solution.FileName), solution.Name), self.SolutionSuffix)
        
        with Utils.Output(fileName, Utils.EnvironmentStyle.Dollar, True) as output:

            output.Format('ifndef verbose\n')
            output.Format('\tSILENT = @\n')
            output.Format('endif\n\n')

            output.Format('.PHONY: all\n')
            output.Format('all: ')

            for project in self.SelectProjects(solution):
                output.Format('%s ' % (project.Name))
            output.Format('\n\n')

            for project in self.SelectProjects(solution):
                output.Format('.PHONY: %s\n' % (project.Name))
                output.Format('%s: ' % (project.Name))

                for dep in self.SelectDependencies(project):
                    output.Format('%s ' % (dep.Name))
                output.Format('\n')

                output.Format('\t$(SILENT) $(MAKE) -C %s -f %s%s\n\n' % (
                    os.path.relpath(os.path.dirname(project.FileName), os.path.dirname(solution.FileName)),
                    project.Name,
                    self.ProjectSuffix))

            output.Format('.PHONY: clean\n')
            output.Format('clean: ')

            for project in self.SelectProjects(solution):
                output.Format('%s-clean ' % (project.Name))
            output.Format('\n\n')

            for project in self.SelectProjects(solution):
                output.Format('.PHONY: %s-clean\n' % (project.Name))
                output.Format('%s-clean:\n' % (project.Name))

                output.Format('\t$(SILENT) $(MAKE) -C %s -f %s%s clean\n\n' % (
                    os.path.relpath(os.path.dirname(project.FileName), os.path.dirname(solution.FileName)),
                    project.Name,
                    self.ProjectSuffix))

            output.Format('\n')


class Writer(WriterBase.WriterBase):

    #
    # Supported platforms and project types
    #
    _platforms = [
        'Arm7',
        'Arm7u',
    ]

    _projects = [
        'C/CPP',
    ]

    def __init__(self, platforms, projects):
        WriterBase.WriterBase.__init__(self, self._platforms, platforms, self._projects, projects)
        self.__innerWriters = [

            WrWbMkBase(['Arm7'],
                platforms,
                self._projects,
                projects,
                'WrWbMk',
                '-WrWb'),

            WrWbMkBase(['Arm7u'],
                platforms,
                self._projects,
                projects,
                'WrWbMku',
                '-WrWbu'),
        ]

    @property
    def Name(self):
        return 'WrWbMk'

    def WriteProject(self, project):
        return self.WriteProjectEx(project)

    def WriteProjectEx(self, project, forceUpdate=False):
        for writer in self.__innerWriters:
            configurations = writer.SelectConfigurations(project)
            if len(configurations) != 0:
                writer.WriteProjectEx(project, forceUpdate)

    def WriteSolution(self, solution):
        for writer in self.__innerWriters:
            projects = writer.SelectProjects(solution)
            if len(projects) != 0:
                writer.WriteSolution(solution)
