#!/usr/bin/python
# This file is part of the FireSync Project Generator.
# Copyright (C) 2008-2016 by LMI Technologies Inc.  All rights reserved.
# Distributed under the terms of the MIT License.
# Redistributed files must retain the above copyright notice.

import re
import os
from . import Project
from . import WriterBase
from . import Utils

def GetTriplet():
    return 'tic6x-none-elf'

def CgToolsPath7():
    return os.path.join('$(TI_PATH)', 'cgtools_7.4.13')

def CgToolsPath8():
    return os.path.join('$(TI_PATH)', 'cgtools_8.3.0')

def Ccs6Path():
    return os.path.join('$(TI_PATH)', 'ccsv6')

def GnuToolsVersion():
    return '4.9.4'

def GnuToolsPatch():
    return 'p23'

def TiPrefix(isWindows):
    if isWindows:
        return 'C:' + os.path.sep + 'ti'
    else:
        return os.path.sep + 'ti'

def ToolsPath(isWindows):
    if isWindows:
        return 'C:' + os.path.sep + 'tools'
    else:
        return os.path.sep + 'tools'

def ToolsPrefix(isWindows):
    return os.path.join(ToolsPath(isWindows), 'GccC64x_' + GnuToolsVersion() + '-' + GnuToolsPatch())

def CrossToolsPrefix(isWindows):
    return os.path.join(ToolsPrefix(isWindows), GetTriplet(), 'bin', GetTriplet() + '-')
    
def BiosPath():
    return os.path.join('$(TI_PATH)', 'bios_6_42_03_35')

def XdcToolsPath():
    return os.path.join('$(TI_PATH)', 'xdctools_3_31_00_24_core')

def UseGnuCompiler():
    try:
        if os.environ['K_CCS_USE_GCC'] == '1':
            return True
    except KeyError:
        pass

    return False

def GeneratorEnvString():
    return 'K_REGENERATOR'

def ReGeneratorEnabled():
    try:
        return os.environ[GeneratorEnvString()] != '0'
    except KeyError:
        #TODO: enable by default, after feature review
        return False

def GmakePath(projectFileName):
    herePath = os.path.dirname(os.path.realpath(__file__))
    gmakePath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'extern', 'Make', 'gmake.exe'))
    return os.path.relpath(gmakePath, os.path.dirname(projectFileName))

#
# Define how the projects are (re-)built
#
def BuildCmd(project, config):
    return '%s -s -f %s-Ccs6.mk config=%s' % \
        (GmakePath(project.FileName),
        project.Name,
        config.Name)

def RebuildCmd(project, config):
    return '%s -s -f %s-Ccs6.mk config=%s clean all' % \
        (GmakePath(project.FileName),
        project.Name,
        config.Name)

def CleanCmd(project, config):
    return '%s -s -f %s-Ccs6.mk config=%s clean' % \
        (GmakePath(project.FileName),
        project.Name,
        config.Name)

class Writer(WriterBase.WriterBase):

    _platforms = [
        'C64x',
    ]

    _projects = [
        'C/CPP',
    ]

    _javaHome = os.path.join(Ccs6Path(), 'eclipse', 'jre')

    _tiCompilerCommonFlags = [
        '-mi1',
        '--mem_model:data=far',
        '-q',
    ]

    _tiCompilerCoffAbiFlags = [
        '--abi=coffabi',
    ]

    _tiCompilerEabiFlags = [
        '--abi=eabi',
    ]

    _tiCompilerCFlags = [
        '-pr',
        '-pdr',
        '-pden',
        '-pds179',
        '-pds190',
        '-pds195',
        '-pds238',
        '-pds552',
        '-pds880',
        '-pds1628',
        '-pds2142',
        '-pds2709',
    ]

    _tiCompilerCxxFlags = [
        '-pr',
        '-pden',
        '-pds179',
        '--exceptions',
    ]

    _tiCompilerFlagTemplates = {
        'Symbols': [
            '-g',
        ],

        'Optimize': [
            '-o2',
        ],

        'Optimize3': [
            '-o3',
        ],

        'Speed0': [
            '-ms0',
        ],

        'Speed1': [
            '-ms1',
        ],

        'Speed2': [
            '-ms2',
        ],

        'Speed3': [
            '-ms3',
        ],

        'C64x': [
            '-mv6400',
        ],

        'C64x+': [
            '-mv6400+',
        ],
    }

    _tiBiosIncludeDirs = [
        os.path.join(BiosPath(), 'packages'),
        os.path.join(XdcToolsPath(), 'packages'),
    ]

    _tiLinkerFlags = [
        '-q',
        '-a',
        '-c',
        '-x',
        '--diag_suppress=10189',
        '--diag_suppress=16002',
        '--diag_suppress=10068',
    ]

    _gnuCompilerFlags = [
        '-march=c64x+',
        '-msdata=none',
        '-ffunction-sections',
        '-fdata-sections',
    ]

    _gnuCompilerCFlags = [
        '-std=gnu99',
        '-Wall',
        '-Wfloat-conversion',
        '-Wno-missing-braces', # this is for GCC bug 53119 -- should be removed when GCC >= 5.x is abundant
        '-Wno-unused-variable',
        '-Wno-unused-parameter',
        '-Wno-unused-value',
    ]

    _gnuCompilerCxx11Flags = [
        '-std=c++11',
        '-fexceptions',
        '-Wall',
        '-Wfloat-conversion',
        '-Wno-missing-braces', # this is for GCC bug 53119 -- should be removed when GCC >= 5.x is abundant
    ]

    _gnuCompilerCxxFlags = [
        '-std=c++14',
        '-fexceptions',
        '-Wall',
        '-Wfloat-conversion',
        '-Wno-missing-braces', # this is for GCC bug 53119 -- should be removed when GCC >= 5.x is abundant
    ]

    _gnuCompilerFlagTemplates = {
        'Symbols': [
            '-g',
            '-gdwarf-2',
        ],

        'OptimizeDebug': [
            '-Og',
            '-mlong-calls',
        ],

        'Optimize': [
            '-O2',
            '-fno-partial-inlining',    # FSS-1336: Investigate recent C64x bootup issues after GoApi merge
            '-fno-reorder-functions',   # FSS-998: this optimization is disabled because it is suspected to result in invalid program code
        ],

        'Optimize3': [
            '-O2',                      # we don't trust aggressive optimizations using GCC on C64x.
            '-fno-partial-inlining',    # FSS-1336: Investigate recent C64x bootup issues after GoApi merge
            '-fno-reorder-functions',   # FSS-998: this optimization is disabled because it is suspected to result in invalid program code
        ],

        'LongCalls': [
            '-mlong-calls',
        ],

        'NoBuiltin': [
            '-fno-builtin',
        ],
    }

    _gnuLinkerFlags = [
        '-X',
        '-EL',
        '-r',
        '-nostdlib',
        '--allow-multiple-definition',
        '--whole-archive',
    ]

    def __init__(self, platforms, projects):
        WriterBase.WriterBase.__init__(self, self._platforms, platforms, self._projects, projects)
        self._useGnuCompiler = UseGnuCompiler()
        self.regeneratorEnabled = ReGeneratorEnabled()

    @property
    def Name(self):
        return 'Ccs6Mk'

    @property
    def TempName(self):
        return 'ccs6mk'

    @property
    def SolutionExt(self):
        return '.mk'

    @property
    def ProjectExt(self):
        return '.mk'

    @property
    def ProjectSuffix(self):
        return self.Suffix + self.ProjectExt

    @property
    def SolutionSuffix(self):
        return self.Suffix + self.SolutionExt

    @property
    def Suffix(self):
        return '-Ccs6'

    def CgToolsPath(self, config):
        # FSS-1221: Investigate recent C64x debug image build failures
        # Using Path7 for release builds, to avoid having to retest performance
        # Using Path8 for debug builds (only debug builds are affected by various issues)
        # Using CoffAbi (CCS3 ABI) for kAte releases
        if ((not self.UseEabi(config)) or (not self.IsDebug(config)) or self.IsLoader(config)):
            return CgToolsPath7()
        else:
            return CgToolsPath8()

    def LibDirs(self, config):
        return [
            os.path.join(self.CgToolsPath(config), 'lib'),
            os.path.join('$(TOOLS_PATH)', GetTriplet(), 'lib', 'gcc', GetTriplet(), GnuToolsVersion()),
            os.path.join('$(TOOLS_PATH)', GetTriplet(), GetTriplet(), 'lib'),
        ]

    def TiCompilerIncludeDirs(self, config):
        return [
            os.path.join(self.CgToolsPath(config), 'include'),
        ]

    def UseEabi(self, config):
        return True if 'CoffAbi' not in config.Template.split('|') else False

    def IsDebug(self, config):
        return True if 'Symbols' in config.Template.split('|') else False

    def IsLoader(self, config):
        return True if 'Loader' in config.Template.split('|') else False

    def ForceUseOfTic6x(self, config, sourceName):
        for sourceTemplate in config.SourceTemplates:
            if sourceTemplate.Name == sourceName:
                if sourceTemplate.Add == 'UseTic6x':
                    return True

        return False

    def ForceCxx11(self, config):
        return True if 'c++11' in config.Template.split('|') else False

    def SelectGnuCompilerFlags(self, config, selectionString):
        compilerFlags = []

        for component in selectionString.split('|'):
            for tmp in self._gnuCompilerFlagTemplates:
                if tmp == component:
                    for flag in self._gnuCompilerFlagTemplates[tmp]:
                        compilerFlags.append(flag)

        for flag in self._gnuCompilerFlags:
            compilerFlags.append(flag)

        return compilerFlags

    def SelectTiCompilerFlags(self, config, selectionString):
        compilerFlags = []

        for component in selectionString.split('|'):
            for tmp in self._tiCompilerFlagTemplates:
                if tmp == component:
                    for flag in self._tiCompilerFlagTemplates[tmp]:
                        compilerFlags.append(flag)

        if self.UseEabi(config):
            for flag in self._tiCompilerEabiFlags:
                compilerFlags.append(flag)

        else:
            for flag in self._tiCompilerCoffAbiFlags:
                compilerFlags.append(flag)

        for flag in self._tiCompilerCommonFlags:
            compilerFlags.append(flag)

        return compilerFlags

    def SelectSourceCompilerFlags(self, config, sourceName):
        templates = config.Template.split('|')

        for sourceTemplate in config.SourceTemplates:
            if sourceTemplate.Name == sourceName:
                if sourceTemplate.Add != 'Exclude':
                    if sourceTemplate.Add != '':
                        for component in sourceTemplate.Add.split('|'):
                            templates.append(component)

                    if sourceTemplate.Remove != '':
                        for component in sourceTemplate.Remove.split('|'):
                            templates.remove(component)

        templateStr = '|'.join(templates)
        sourceTiCompilerFlags = ' '.join(self.SelectTiCompilerFlags(config, templateStr))
        sourceGnuCompilerFlags = ' '.join(self.SelectGnuCompilerFlags(config, templateStr))
        defaultTiCompilerFlags = ' '.join(self.SelectTiCompilerFlags(config, config.Template))
        defaultGnuCompilerFlags = ' '.join(self.SelectGnuCompilerFlags(config, config.Template))

        if sourceTiCompilerFlags != defaultTiCompilerFlags or sourceGnuCompilerFlags != defaultGnuCompilerFlags:
            return sourceTiCompilerFlags, sourceGnuCompilerFlags
        return '', ''

    def HasAsm(self, project, config):
        for src in self.FilterSources(project, config):
            source = src.Name
            extension = os.path.splitext(source)[1]
            if extension == '.s' or extension == '.S':
                return True
        return False

    def XdcGccPath(self, project, config):
        herePath = os.path.dirname(os.path.realpath(__file__))
        headerPath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'extern', 'Ti', 'xdc', 'xdc_gcc.h'))
        return os.path.relpath(headerPath, os.path.dirname(project.FileName))

    def CfgPath(self, project, config):
        if self.IsLoader(config):
            return None
        herePath = os.path.dirname(os.path.realpath(__file__))
        cfgPath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'kHardwareDm647', 'kHardwareDm647', 'Tcf'))
        return os.path.relpath(cfgPath, os.path.dirname(project.FileName))

    def CmdPath(self, project, config):
        for src in self.SelectSources(project):
            source = src.Name
            extension = os.path.splitext(source)[1]
            if extension == '.cmd' and os.path.basename(source) != 'kHwDm647.cmd':
                return source

        return os.path.join(self.CfgPath(project, config), 'kHwDm647.cmd')

    def FilterSources(self, project, config):
        sources = []
        for src in self.SelectSources(project):

            excluded = False
            for sourceTemplate in config.SourceTemplates:
                if sourceTemplate.Name == src.Name:
                    if sourceTemplate.Add == 'Exclude':
                        excluded = True
            
            if not excluded:
                sources.append(src)

        return sources

    def WriteProject(self, project):
        return self.WriteProjectEx(project)

    def WriteProjectEx(self, project, forceUpdate=False):

        fileName = '%s%s' % (os.path.splitext(project.FileName)[0], self.ProjectSuffix)

        with Utils.Output(fileName, Utils.EnvironmentStyle.Dollar, True, forceUpdate=forceUpdate) as output:

            utilsPath = os.path.normpath(os.path.join(os.path.dirname(os.path.realpath(__file__)), '..'))
            utilsRelPath = os.path.relpath(utilsPath, os.path.dirname(project.FileName))

            configurations = self.SelectConfigurations(project)

            if len(configurations) != 0:

                output.Format('\n')

                if self.regeneratorEnabled:
                    output.Format('export %s := 1\n' % (GeneratorEnvString()))
                    output.Format('\n')

                output.Format('ifeq ($(OS)$(os), Windows_NT)\n')
                output.Format('\tTI_PATH      := %s\n' % (TiPrefix(True)))
                output.Format('\tTOOLS_PATH   := %s\n' % (ToolsPrefix(True)))
                output.Format('\tCROSS_PREFIX := %s\n' % (CrossToolsPrefix(True)))
                output.Format('\tCROSS_SUFFIX := .exe\n')
                output.Format('\tPYTHON       := python\n')
                output.Format('else\n')
                output.Format('\tTI_PATH      := %s\n' % (TiPrefix(False)))
                output.Format('\tTOOLS_PATH   := %s\n' % (ToolsPrefix(False)))
                output.Format('\tCROSS_PREFIX := %s\n' % (CrossToolsPrefix(False)))
                output.Format('\tCROSS_SUFFIX := \n')
                output.Format('\tPYTHON       := python3\n')
                output.Format('endif\n\n')

                output.Format('GNU_COMPILER := $(CROSS_PREFIX)gcc$(CROSS_SUFFIX)\n')
                output.Format('GNU_LINKER := $(CROSS_PREFIX)ld$(CROSS_SUFFIX)\n')
                output.Format('GNU_OBJ_NAMES := $(CROSS_PREFIX)nm$(CROSS_SUFFIX)\n')
                output.Format('LINKER := $(CROSS_PREFIX)ld$(CROSS_SUFFIX)\n')
                output.Format('ARCHIVER := $(CROSS_PREFIX)ar$(CROSS_SUFFIX)\n')
                output.Format('OBJ_NAMES := $(CROSS_PREFIX)nm$(CROSS_SUFFIX)\n')
                output.Format('XDC_XS := ' + XdcToolsPath() + '/xs$(CROSS_SUFFIX)\n')
                output.Format('\n')

                output.Format('AIS := $(PYTHON) %s\n' % os.path.join(utilsRelPath, 'kAis.py'))
                output.Format('MUNCH := $(PYTHON) %s\n' % os.path.join(utilsRelPath, 'kCrunch.py'))
                output.Format('MKDIR_P := $(PYTHON) %s mkdir_p\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                output.Format('RM_F := $(PYTHON) %s rm_f\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                output.Format('RM_RF := $(PYTHON) %s rm_rf\n' % os.path.join(utilsRelPath, 'kUtil.py'))
                if self.regeneratorEnabled:
                    output.Format('KGENERATOR := $(PYTHON) %s\n' % os.path.join(utilsRelPath, 'kGenerator.py'))
                output.Format('\n')

                output.Format('ifndef verbose\n')
                output.Format('\tSILENT := @\n')
                output.Format('endif\n\n')

                output.Format('ifndef config\n')
                output.Format('\tconfig := %s\n' % (configurations[0].Name))
                output.Format('endif\n\n')

                output.Format('# We require tools to be installed according to specific conventions (see manuals).\n')
                output.Format('# Tool prerequisites may change between major releases; check and report.\n')
                output.Format('ifeq ($(shell $(GNU_COMPILER) --version),)\n')
                output.Format('.PHONY: tc_err\n')
                output.Format('tc_err:\n')
                output.Format('\t$(error Cannot build because of missing prerequisite; please install)\n')
                output.Format('endif\n\n')

                for config in configurations:

                    tiCompilerFlags = self.SelectTiCompilerFlags(config, config.Template)
                    gnuCompilerFlags = self.SelectGnuCompilerFlags(config, config.Template)
                    cfgPath = self.CfgPath(project, config)
                    cmdFile = self.CmdPath(project, config)
                    configType = config.Type

                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    if configType == 'Executable':
                        targetName = os.path.join(config.BinDir, config.Target)
                        targetExt = os.path.splitext(targetName)[1]
                        outName = os.path.splitext(targetName)[0] + '.out'
                        output.Format('\tTARGET := %s\n' % (targetName))

                        if targetExt == '.out':
                            output.Format('\tINTERMEDIATES := \n')
                        else:
                            output.Format('\tINTERMEDIATES := %s\n' % (outName))

                    elif configType == 'Shared':
                        targetName = os.path.join(config.BinDir, config.Target)
                        output.Format('\tTARGET := %s\n' % (targetName))
                        output.Format('\tINTERMEDIATES := \n')

                    else:
                        targetName = os.path.join(config.LibDir, config.Target)
                        output.Format('\tTARGET := %s\n' % (targetName))
                        output.Format('\tINTERMEDIATES := \n')

                    output.Format('\tOBJ_DIR := %s/%s-%s-%s\n' % (config.TempDir, project.Name, self.TempName, config.Name))
                    output.Format('\tPREBUILD := %s\n' % (config.Prebuild))
                    output.Format('\tPOSTBUILD := %s\n' % (config.Postbuild))

                    output.Format('\tTI_COMPILER := %s\n' % os.path.join(self.CgToolsPath(config), 'bin', 'cl6x$(CROSS_SUFFIX)'))
                    output.Format('\tTI_LINKER := %s\n' % os.path.join(self.CgToolsPath(config), 'bin', 'lnk6x$(CROSS_SUFFIX)'))
                    output.Format('\tTI_ARCHIVER := %s\n' % os.path.join(self.CgToolsPath(config), 'bin', 'ar6x$(CROSS_SUFFIX)'))
                    output.Format('\tTI_OFDISP := %s\n' % os.path.join(self.CgToolsPath(config), 'bin', 'ofd6x$(CROSS_SUFFIX)'))

                    output.Format('\tTI_COMPILER_FLAGS :=')
                    for flag in tiCompilerFlags:
                        output.Format(' %s' % (flag))
                    output.Format('\n')

                    output.Format('\tTI_COMPILER_C_FLAGS :=')
                    for flag in self._tiCompilerCFlags:
                        output.Format(' ')
                        output.Format('%s' % (flag))
                    output.Format('\n')

                    output.Format('\tTI_COMPILER_CXX_FLAGS :=')
                    for flag in self._tiCompilerCxxFlags:
                        output.Format(' ')
                        output.Format('%s' % (flag))
                    output.Format('\n')

                    output.Format('\tTI_COMPILER_DEFINES :=')
                    for define in config.Defines:
                        output.Format(' -D%s' % (define))
                    output.Format('\n')

                    output.Format('\tTI_COMPILER_INCLUDE_DIRS :=')
                    for includeDirs in [ self._tiBiosIncludeDirs, self.TiCompilerIncludeDirs(config), config.IncludeDirs ]:
                        for dir in includeDirs:
                            output.Format(' -I%s' % (dir))
                    output.Format('\n')

                    output.Format('\tTI_LINKER_FLAGS :=')
                    for flag in self._tiLinkerFlags:
                        output.Format(' %s' % (flag))
                    if self.IsDebug(config):
                        output.Format(' --define=_DEBUG')
                    output.Format('\n')

                    output.Format('\tTI_LIB_DIRS :=')
                    for libDir in self.LibDirs(config):
                        output.Format(' -i%s' % (libDir))
                    for libDir in config.LibraryDirs:
                        output.Format(' -i%s' % (libDir))
                    if config.Type == 'Executable' and cfgPath is not None:
                        output.Format(' -i%s' % (cfgPath))
                    output.Format('\n')

                    output.Format('\tTI_LIBS :=')
                    for lib in config.Libs:
                        if (os.path.splitext(lib)[1] == ''):
                            output.Format(' -llib%s.a' % (lib))
                        else:
                            output.Format(' -l%s' % (lib))
                    if self.UseEabi(config): 
                        output.Format(' -llibgccmin.a')
                        output.Format(' -llibstdc++.a')
                        output.Format(' -llibcmin.a')
                    if config.Type == 'Executable':
                        output.Format(' %s' % (cmdFile))
                    output.Format('\n')

                    output.Format('\tGNU_COMPILER_FLAGS :=')
                    for flag in gnuCompilerFlags:
                        output.Format(' %s' % (flag))
                    output.Format('\n')

                    output.Format('\tGNU_COMPILER_C_FLAGS :=')
                    for flag in self._gnuCompilerCFlags:
                        output.Format(' %s' % (flag))
                    output.Format('\n')

                    output.Format('\tGNU_COMPILER_CXX_FLAGS :=')
                    gnuCompilerCxxFlags = self._gnuCompilerCxx11Flags if self.ForceCxx11(config) else self._gnuCompilerCxxFlags
                    for flag in gnuCompilerCxxFlags:
                        output.Format(' %s' % (flag))
                    output.Format('\n')

                    output.Format('\tGNU_COMPILER_DEFINES :=')
                    for define in config.Defines:
                        output.Format(' -D%s' % (define))
                    output.Format(' -Dxdc_target_types__=%s' % (self.XdcGccPath(project, config)))
                    output.Format('\n')

                    output.Format('\tGNU_COMPILER_INCLUDE_DIRS :=')
                    for includeDirs in [ self._tiBiosIncludeDirs, config.IncludeDirs ]:
                        for dir in includeDirs:
                            output.Format(' -I%s' % (dir))
                    output.Format('\n')

                    if self.HasAsm(project, config):
                        output.Format('\tGNU_COMPILER_ASM_INCLUDE_DIRS :=')
                        for dir in config.IncludeDirs:
                            output.Format(' ')
                            output.Format('-Wa,-I%s' % (dir))
                        output.Format('\n')

                    output.Format('\tGNU_LINKER_FLAGS :=')
                    for flag in self._gnuLinkerFlags:
                        output.Format(' %s' % (flag))
                    output.Format('\n')

                    output.Format('\tGNU_LIB_DIRS :=')
                    for libDir in self.LibDirs(config):
                        output.Format(' -L%s' % (libDir))
                    for libDir in config.LibraryDirs:
                        output.Format(' -L%s' % (libDir))
                    output.Format('\n')

                    output.Format('\tGNU_LIBS :=')
                    for lib in config.Libs:
                        if (os.path.splitext(lib)[1] == ''):
                            output.Format(' -l%s' % (lib))
                        else:
                            output.Format(' -l:%s' % (lib))

                    if self.UseEabi(config): 
                        output.Format(' -lgccmin')
                        output.Format(' -lstdc++')
                        output.Format(' -lcmin')
                    output.Format('\n')

                    output.Format('\tOBJECTS := ')
                    first = True
                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.c' or extension == '.cpp' or extension == '.s' or extension == '.S' or extension == '.s62' or extension == '.asm':
                            if not first:
                                output.Format(' \\\n\t', False)
                            first = False
                            output.Format('%s/%s-%s-%s/%s.obj' % (config.TempDir, project.Name, self.TempName, config.Name, os.path.splitext(os.path.basename(src.Name))[0]))
                    output.Format('\n')

                    output.Format('\tXS_FILES := ')
                    first = True
                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.cfg':
                            if not first:
                                output.Format(' \\\n\t', False)
                            first = False
                            output.Format('%s/configPkg/linker.cmd' % (os.path.dirname(src.Name)))
                    output.Format('\n')

                    output.Format('\tDEP_FILES := ')
                    first = True
                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.c' or extension == '.cpp':
                            if not first:
                                output.Format(' \\\n\t', False)
                            first = False
                            output.Format('%s/%s-%s-%s/%s.d' % (config.TempDir, project.Name, self.TempName, config.Name, os.path.splitext(os.path.basename(src.Name))[0]))
                    output.Format('\n')

                    #
                    # List output files of dependent projects as target dependencies. 
                    # Targets from dependent projects are listed as dependencies here by matching the configuration name
                    # 
                    output.Format('\tTARGET_DEPS := ')
                    first = True
                    for dep in self.SelectDependencies(project):
                        depConfigs = self.SelectConfigurations(dep)
                        for depConfig in depConfigs:
                            if depConfig.Name == config.Name:

                                projectPath = os.path.relpath(os.path.dirname(dep.FileName), os.path.dirname(project.FileName))
                                target = os.path.join(projectPath, (depConfig.LibDir if depConfig.Type == 'Library' else depConfig.BinDir), depConfig.Target)

                                if not first:
                                    output.Format(' \\\n\t', False)
                                first = False
                                output.Format('%s' % (target))

                    output.Format('\n\n')
                    output.Format('endif\n\n')

                output.Format('.PHONY: all all-obj all-dep\n\n')

                output.Format('all: $(OBJ_DIR)\n')
                output.Format('\t$(PREBUILD)\n')
                output.Format('\t$(SILENT) $(MAKE) -f %s%s all-dep\n' % (project.Name, self.ProjectSuffix))
                output.Format('\t$(SILENT) $(MAKE) -f %s%s all-obj\n' % (project.Name, self.ProjectSuffix))
                output.Format('\n')

                output.Format('clean:\n')
                output.Format('\t$(info Cleaning $(OBJ_DIR))\n')
                output.Format('\t$(SILENT) $(RM_RF) $(OBJ_DIR)\n')
                output.Format('\t$(info Cleaning $(TARGET) $(INTERMEDIATES))\n')
                output.Format('\t$(SILENT) $(RM_F) $(TARGET) $(INTERMEDIATES)\n')
                output.Format('\n')

                output.Format('all-obj: $(OBJ_DIR) $(TARGET)\n')
                output.Format('all-dep: $(OBJ_DIR) $(DEP_FILES)\n')
                output.Format('\n')

                if self.regeneratorEnabled:
                    output.Format('$(OBJ_DIR): %s%s\n' % (project.Name, self.ProjectSuffix))
                    output.Format('\t$(info Cleanse $(OBJ_DIR))\n')
                    output.Format('\t$(SILENT) $(RM_RF) $(OBJ_DIR)\n')
                    output.Format('\t$(SILENT) $(MKDIR_P) $(OBJ_DIR)\n')
                    output.Format('\n')
                else:
                    output.Format('$(OBJ_DIR):\n')
                    output.Format('\t$(SILENT) $(MKDIR_P) $@\n')
                    output.Format('\n')

                if self.regeneratorEnabled:
                    solFileNameRel = os.path.relpath(project.Solution.FileName, os.path.dirname(project.FileName))
                    output.Format('ifeq ($(MAKECMDGOALS),)\n')
                    output.Format('ifeq ($(MAKE_RESTARTS),)\n')
                    output.Format('%s%s: %s.xml %s %s\n' % (project.Name, self.ProjectSuffix, project.Name, solFileNameRel, os.path.join(utilsRelPath, 'Generator', self.Name + '.py')))
                    output.Format('\t$(info Regen%s %s%s)\n' % (configurations[0].Platform, project.Name, self.ProjectSuffix))
                    output.Format('\t$(SILENT) $(KGENERATOR) --writers=%s --platforms=%s --project=%s %s\n' % (self.Name, configurations[0].Platform, project.Name, solFileNameRel))
                    output.Format('endif\n')
                    output.Format('endif\n')
                    output.Format('\n')

                for config in configurations:

                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    if config.Type == 'Library':
                        target = os.path.join(config.LibDir, config.Target)

                        output.Format('%s: $(OBJECTS) $(XS_FILES)\n' % (target))
                        output.Format('\t$(info ArC64x %s)\n' % (target))
                        output.Format('\t$(SILENT) $(RM_F) %s\n' % (target))
                        output.Format('\t$(SILENT) $(ARCHIVER) rcs %s $(OBJECTS)\n' % (target))
                        output.Format('\t$(POSTBUILD)\n')
                        output.Format('\n')

                    elif config.Type == 'Shared':
                        target = os.path.join(config.BinDir, config.Target)

                        output.Format('%s: $(OBJECTS) $(XS_FILES)\n' % (target))
                        output.Format('\t$(info ArC64x %s)\n' % (target))
                        output.Format('\t$(SILENT) $(RM_F) %s\n' % (target))
                        output.Format('\t$(SILENT) $(ARCHIVER) rcs %s $(OBJECTS)\n' % (target))
                        output.Format('\t$(POSTBUILD)\n')
                        output.Format('\n')

                    elif config.Type == 'Executable':
                        target = os.path.join(config.BinDir, config.Target)
                        targetExt = os.path.splitext(target)[1]
                        outName = os.path.splitext(target)[0] + '.out'

                        tmpDir = '%s/%s-%s-%s' % (config.TempDir, project.Name, self.TempName, config.Name)
                        partialImage = '%s/%s-partial.o' % (tmpDir, project.Name)
                        munchSource = '%s/%s-munch.c' % (tmpDir, project.Name)
                        munchObj = '%s/%s-munch.o' % (tmpDir, project.Name)

                        mapName = '%s/%s-%s-%s/%s.map' % (
                            config.TempDir,
                            project.Name,
                            self.TempName,
                            config.Name,
                            os.path.splitext(os.path.basename(config.Target))[0])

                        ofdName = '%s/%s-%s-%s/%s.xml' % (
                            config.TempDir,
                            project.Name,
                            self.TempName,
                            config.Name,
                            os.path.splitext(os.path.basename(config.Target))[0])

                        if targetExt == '.out':
                            output.Format('%s: $(OBJECTS) $(TARGET_DEPS)\n' % (target))
                            output.Format('\t$(info TiClrC64x %s)\n' % (target))
                            output.Format('\t$(SILENT) $(TI_LINKER) $(TI_LINKER_FLAGS) -o%s -m%s $(TI_LIB_DIRS) $(TI_LIBS) $(OBJECTS)\n' % (target, mapName))
                            output.Format('\t$(POSTBUILD)\n')
                            output.Format('\n')

                        else:
                            output.Format('%s: %s %s\n' % (target, ofdName, outName))
                            output.Format('\t$(info AisC64x %s)\n' % (target))
                            if targetExt == '.bin':
                                output.Format('\t$(SILENT) $(AIS) --emif %s %s %s\n' % (ofdName, outName, target))
                            else:
                                output.Format('\t$(SILENT) $(AIS) %s %s %s\n' % (ofdName, outName, target))
                            output.Format('\t$(POSTBUILD)\n')
                            output.Format('\n')

                            output.Format('%s: %s\n' % (ofdName, outName))
                            output.Format('\t$(info OfdC64x %s)\n' % (ofdName))
                            output.Format('\t$(SILENT) $(TI_OFDISP) -x --obj_display=none,header,optheader,sections,symbols %s > %s\n' % (outName, ofdName))
                            output.Format('\n')

                            if self.IsLoader(config):
                                output.Format('%s: $(OBJECTS) $(TARGET_DEPS)\n' % (outName))
                                output.Format('\t$(info LdC64x %s)\n' % (outName))
                                output.Format('\t$(SILENT) $(TI_LINKER) $(TI_LINKER_FLAGS) -o%s -m%s $(TI_LIB_DIRS) $(TI_LIBS) $(OBJECTS)\n' % (outName, mapName))
                                output.Format('\n')

                            else:
                                output.Format('%s: $(OBJECTS) $(TARGET_DEPS) %s\n' % (outName, munchObj))
                                output.Format('\t$(info TiClrC64x %s)\n' % (outName))
                                output.Format('\t$(SILENT) $(TI_LINKER) $(TI_LINKER_FLAGS) -o%s -m%s $(TI_LIB_DIRS) $(TI_LIBS) $(OBJECTS) %s\n' % (outName, mapName, munchObj))
                                output.Format('\n')

                                output.Format('%s: %s\n' % (munchObj, munchSource))
                                output.Format('\t$(info GccC64x %s)\n' % (munchSource))
                                output.Format('\t$(SILENT) $(GNU_COMPILER) $(GNU_COMPILER_FLAGS) $(GNU_COMPILER_C_FLAGS) -o %s -c %s -MMD -MP\n' % (munchObj, munchSource))
                                output.Format('\n')
                                
                                output.Format('%s: %s\n' % (munchSource, partialImage))
                                output.Format('\t$(info MunchC64x %s)\n' % (munchSource))
                                output.Format('\t$(SILENT) $(MUNCH) $(GNU_OBJ_NAMES) %s %s\n' % (partialImage, munchSource))
                                output.Format('\n')

                                output.Format('%s: $(OBJECTS) $(TARGET_DEPS)\n' % (partialImage))
                                output.Format('\t$(info LdC64x %s)\n' % (partialImage))
                                output.Format('\t$(SILENT) $(GNU_LINKER) $(GNU_LIB_DIRS) $(GNU_LINKER_FLAGS) -o %s $(OBJECTS) $(GNU_LIBS)\n' % (partialImage))
                                output.Format('\n')

                    output.Format('endif\n\n')

                for config in configurations:

                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    for src in self.SelectSources(project):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        tmpDir = '%s/%s-%s-%s' % (config.TempDir, project.Name, self.TempName, config.Name)

                        if extension == '.c' or extension == '.cpp':
                            ppdSource = '%s/%s.d' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])
                            objectFile = '%s/%s.obj' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])

                            output.Format('%s %s: ' % (objectFile, ppdSource))
                            output.Format('%s\n' % (src.Name))

                            tiCompilerFlags, gnuCompilerFlags = self.SelectSourceCompilerFlags(config, src.Name)
                            isCxxSource = (extension == '.cpp')
                            useGnuCompiler = (isCxxSource or 
                                ((self._useGnuCompiler or config.UsePrimaryTools == '') and (not self.IsLoader(config)) and (not self.ForceUseOfTic6x(config, src.Name))))

                            if useGnuCompiler:
                                compilerFlags = '$(GNU_COMPILER_FLAGS)' if len(gnuCompilerFlags) == 0 else gnuCompilerFlags
                                languageFlags = '$(GNU_COMPILER_CXX_FLAGS)' if isCxxSource else '$(GNU_COMPILER_C_FLAGS)'

                            else:
                                compilerFlags = '$(TI_COMPILER_FLAGS)' if len(tiCompilerFlags) == 0 else tiCompilerFlags
                                languageFlags = '$(TI_COMPILER_CXX_FLAGS)' if isCxxSource else '$(TI_COMPILER_C_FLAGS)'

                            if useGnuCompiler:
                                output.Format('\t$(info GccC64x $<)\n')
                                output.Format('\t$(SILENT) $(GNU_COMPILER) %s %s $(GNU_COMPILER_DEFINES) $(GNU_COMPILER_INCLUDE_DIRS) -o %s.obj -c %s -MMD -MP\n' % \
                                    (compilerFlags, languageFlags, os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                            else:
                                output.Format('\t$(info TiClC64x $<)\n')
                                output.Format('\t$(SILENT) $(TI_COMPILER) %s %s $(TI_COMPILER_DEFINES) $(TI_COMPILER_INCLUDE_DIRS) -ppa -ppd=%s -fr=%s -fs=%s %s\n' % \
                                    (compilerFlags, languageFlags, ppdSource, tmpDir, tmpDir, src.Name))

                            output.Format('\n')

                        elif extension == '.s' or extension == '.S':
                            objectFile = '%s/%s.obj' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])

                            output.Format('%s: ' % (objectFile))
                            output.Format('%s\n' % (src.Name))

                            output.Format('\t$(SILENT) $(info GccC64x $<)\n')
                            output.Format('\t$(SILENT) $(GNU_COMPILER) $(GNU_COMPILER_FLAGS) -x assembler-with-cpp $(GNU_COMPILER_DEFINES) $(GNU_COMPILER_INCLUDE_DIRS) $(GNU_COMPILER_ASM_INCLUDE_DIRS) -o %s.obj -c %s\n' % (
                                os.path.join(tmpDir, os.path.splitext(os.path.basename(src.Name))[0]), src.Name))

                            output.Format('\n')

                        elif extension == '.s62' or extension == '.asm':
                            objectFile = '%s/%s.obj' % (tmpDir, os.path.splitext(os.path.basename(src.Name))[0])

                            output.Format('%s: ' % (objectFile))
                            output.Format('%s\n' % (src.Name))

                            output.Format('\t$(info TiClC64x $<)\n')
                            output.Format('\t$(SILENT) $(TI_COMPILER) $(TI_COMPILER_FLAGS) $(DEFINES) $(INCLUDE_DIRS) -fr=%s -fs=%s %s\n' % (tmpDir, tmpDir, src.Name))

                            output.Format('\n')

                        elif extension == '.cfg':

                            cfgSource = '%s/configPkg/linker.cmd' % (os.path.dirname(src.Name))

                            output.Format('%s: ' % (cfgSource))
                            output.Format('%s\n' % (src.Name))
                            output.Format('\t$(info TiXdcC64x $<)\n')
                            output.Format('\t$(SILENT) $(PYTHON) %s %s $(XDC_XS) %s %s %s\n' % \
                                (os.path.splitext(src.Name)[0] + 'Build.py',
                                self._javaHome,
                                BiosPath(),
                                Ccs6Path(),
                                CgToolsPath7()))
                            output.Format('\n')

                    output.Format('endif\n\n')

                output.Format('ifeq ($(MAKECMDGOALS),all-obj)\n\n')
                for config in configurations:
                    output.Format('ifeq ($(config),%s)\n\n' % (config.Name))

                    for src in self.FilterSources(project, config):
                        source = src.Name
                        extension = os.path.splitext(source)[1]
                        if extension == '.c' or extension == '.cpp':
                            ppdSource = '%s/%s-%s-%s/%s.d' % (config.TempDir, project.Name, self.TempName, config.Name, os.path.splitext(os.path.basename(src.Name))[0])
                            output.Format('include %s\n' % (ppdSource))

                    output.Format('\n')
                    output.Format('endif\n\n')
                output.Format('endif\n\n')

    def WriteSolution(self, solution):

        fileName = '%s%s' % (os.path.join(os.path.dirname(solution.FileName), solution.Name), self.SolutionSuffix)

        with Utils.Output(fileName, Utils.EnvironmentStyle.Dollar, True) as output:

            output.Format('ifndef verbose\n')
            output.Format('\tSILENT = @\n')
            output.Format('endif\n\n')

            output.Format('.PHONY: all\n')
            output.Format('all: ')

            for project in self.SelectProjects(solution):
                output.Format('%s ' % (project.Name))
            output.Format('\n\n')

            for project in self.SelectProjects(solution):
                output.Format('.PHONY: %s\n' % (project.Name))
                output.Format('%s: ' % (project.Name))

                for dep in self.SelectDependencies(project):
                    output.Format('%s ' % (dep.Name))
                output.Format('\n')

                output.Format('\t$(SILENT) $(MAKE) -C %s -f %s%s\n\n' % (
                    os.path.relpath(os.path.dirname(project.FileName), os.path.dirname(solution.FileName)),
                    project.Name,
                    self.ProjectSuffix))

            output.Format('.PHONY: clean\n')
            output.Format('clean: ')

            for project in self.SelectProjects(solution):
                output.Format('%s-clean ' % (project.Name))
            output.Format('\n\n')

            for project in self.SelectProjects(solution):
                output.Format('.PHONY: %s-clean\n' % (project.Name))
                output.Format('%s-clean:\n' % (project.Name))

                output.Format('\t$(SILENT) $(MAKE) -C %s -f %s%s clean\n\n' % (
                    os.path.relpath(os.path.dirname(project.FileName), os.path.dirname(solution.FileName)),
                    project.Name,
                    self.ProjectSuffix))

            output.Format('\n')
