/**
 * @file    kVsSpotExperimental.x.h
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_SPOT_EXPERIMENTAL_X_H
#define K_VISION_SPOT_EXPERIMENTAL_X_H

#include <kVision/Vs/kVsSpotExperimental.h>
#include <kVision/Common/kVision.h>

#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kImage.h>
#include <kApi/Data/kBytes.h>

/**
 * Get a Savitzky-Golay derivative window given parameters.
 *
 * @param   fitOrder    Polynomial order
 * @param   edgeWindow  Window size
 * @returns const pointer to derivative window of size edgeWindow
 */
k64f const * kVsSpotExp_DerivativeWindow(k32u fitOrder, k32u edgeWindow);

/**
 * Apply arbitrary filter to columns of input image.
 *
 * @param input      Input image must be of type k8u
 * @param filter     Pointer to filter kernel array
 * @param filterSize Filter (window) size
 * @param output     Output image must be of type k8s, k8u or k64f
 */
kVsFx(kStatus) kVsSpotExp_ColumnFilter(kImage input, const k64f* filter, k32u filterSize, kImage output);

/**
 * Find position of zero crossing per column.
 *
 * @param  input      Edge (gradient) Image of type k64f or k8s
 * @param  output     Array of type k64f. Interpolated zero-crossing location per column, k64F_NULL if none found
 * @param  threshold  Gradient must exceed this (+ve) value for zero crossing to be detected
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_YImageDecreasingZeroes(kImage input, kArray1 output, k32u threshold);

/**
 * Compute a balanced gradient (edge) across columns of the image
 *
 * @param input          Input image. Can be of type k8u or k64f
 * @param averageWindow  Number of columns to average over
 * @param edgeWindow     Size of 'sobel' gradient filter
 * @param output         kImage of type k64f. Will be populated with output.
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_YGradientBalanced(kImage input, k32u averageWindow, k32u edgeWindow, kImage output);

/**
 * Inner loop for Savitzky-Golay filtered Zero crossing spot detector.
 * Meant to be wrapped by kVsSpotExp_SavitzkyGolayZC
 *
 * @param  image       Input image. Must be of type k8u.
 * @param  edgeWindow  Window size to use for (Savitzky-Golay) best fit derivative filter
 * @param  fitOrder    Savitzky-Golay polynomial order
 * @param  threshold   Rising and falling edge threshold
 * @param  spots       kArrayList of kSpot, will be populated with detected spots
 * @param  edgeImage   Temp image of type k8u, must be same size as image
 * @param  zeroes      Temp array of k64f used for internal computation
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_SavitzkyGolayZCEx(
    kImage image, k32u edgeWindow, k32u fitOrder, k32u threshold, kArrayList spots, kImage edgeImage, kArray1 zeroes);


/**
 * Inner loop for Savitzky-Golay filtered center of gravity spot detector.
 * Meant to be wrapped by kVsSpotExp_SavitzkyGolayAccg
 *
 * @param  image           Input image. Must be of type k8u.
 * @param  edgeWindow      Window size to use for (Savitzky-Golay) best fit derivative filter
 * @param  fitOrder        Savitzky-Golay polynomial order
 * @param  threshold       Rising and falling edge threshold
 * @param  widthThreshold  Used for width calculation, only pixels with an intensity greater than baseIntensity + widthThreshold are accepted.
 * @param  minWidth        minimum spot width
 * @param  maxWidth        maximum spot width
 * @param  minSum          minimum spot sum
 * @param  spots           kArrayList of kSpot. Will be populated with detected spots
 * @param  bounds          kArrayList of kPoint32s, If not null, will be populated with column boundaries
 * @param  edgeImage       Temp image of type k8u, Must be same size as image
 * @param  stateArray      Temp array of per-column kSpotSobelAccgState
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_SavitzkyGolayAccgEx(
    kImage image, k32u edgeWindow, k32u fitOrder, k32u threshold,
    k32u widthThreshold, k32u minWidth, k32u maxWidth, k32u minSum,
    kArrayList spots, kArrayList bounds, kImage edgeImage, kArray1 stateArray);


/**
 * Inner loop for Sobel filtered Zero-Crossing edge detection
 * Meant to be wrapped by kVsSpotExp_SobelZC
 *
 * @param  image          Input image, can be of type k8u or k64f
 * @param  averageWindow  Number of columns to average over for Sobel filter
 * @param  edgeWindow     Window size to use for (Savitzky-Golay) best fit derivative filter
 * @param  threshold      Rising and falling edge threshold
 * @param  spots          kArrayList of kSpot, Will be populated with detected spots
 * @param  edgeImage      Temp kImage of type k8s
 * @param  balance        Whether gradients should be balanced
 * @param  zeroes         Temp array of k64f used for internal computation
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_SobelZCEx(
    kImage image, k32u averageWindow, k32u edgeWindow, k32u threshold,
    kArrayList spots, kImage edgeImage, kBool balance, kArray1 zeroes);

#endif
