/**
* @file     kVsImageStats.h
* @brief    Declares the kVsImageStats class
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef K_VISION_IMAGE_STATS_H
#define K_VISION_IMAGE_STATS_H

#include <kVision/Common/kVision.h>

/**
* @class       kVsImageStats
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       Container class for image statistics this class contain all image statistics.
*              These value includes:
*              Raw/Sorted pixSize: the sum of pixel count of all spots
*              Raw/Sorted SpotPixSum: the sum of all pixel values of all spots
*              Sorted centreSum: the sum of all spot centroid value
*              PixSum: the sum of all pixel values
*              SpotIntCgSum: the integer part of the all spot centroid sums
*              Total Raw/Sorted spots, pixels count: the number of all Raw/sorted spots, pixels
*
*/
typedef kObject kVsImageStats;

/**
* Constructs a kVsImageStats object.
*
* @public               @memberof kVsImageStats
* @param    imageStats  Destination for the constructed object handle.
* @return               Operation status.
*/
kVsFx(kStatus) kVsImageStats_Construct(kVsImageStats* imageStats, kAlloc allocator);

/**
* Sets the rawSpotPixelCount of image statistics, this is the number of pixels in each of the unsorted reported spots.
*
* @public                     @memberof kVsImageStats
* @param   imageStats         ImageStats object
* @param   rawSpotPixelCount  RawSpotSizeSum of the imageStats
* @return                     Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetRawSpotPixelCount(kVsImageStats imageStats, kSize rawSpotPixelCount);

/**
* Gets the rawSpotPixelCount of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              RawSpotSizeSum.
*/
kVsFx(kSize) kVsImageStats_RawSpotPixelCount(kVsImageStats imageStats);

/**
* Sets the sortedSpotPixelCount of image statistics, this is the sum of the number of pixels in each of the sorted reported spots.
*
* @public                      @memberof kVsImageStats
* @param   imageStats          ImageStats object
* @param   sortedSpotSizeSum   SortedSpotSizeSum of the imageStats
* @return                      Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetSortedSpotPixelCount(kVsImageStats imageStats, kSize sortedSpotPixelCount);

/**
* Gets the sortedSpotPixelCount of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              SortedSpotSizeSum.
*/
kVsFx(kSize) kVsImageStats_SortedSpotPixelCount(kVsImageStats imageStats);

/**
* Sets the rawSpotPixelSum of image statistics, this is the sum of all pixel values in each of the unsorted reported spots.
*
* @public                  @memberof kVsImageStats
* @param   imageStats      ImageStats object
* @param   rawSpotPixSum   RawSpotPixSum of the imageStats
* @return                  Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetRawSpotPixelSum(kVsImageStats imageStats, kSize rawSpotPixelSum);

/**
* Gets the rawSpotPixelSum of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              RawSpotPixSum.
*/
kVsFx(kSize) kVsImageStats_RawSpotPixelSum(kVsImageStats imageStats);

/**
* Sets the sortedSpotPixelSum of image statistics, this is the sum of all pixel values in each of the sorted reported spots.
*
* @public                      @memberof kVsImageStats
* @param   imageStats          ImageStats object
* @param   sortedSpotPixSum    SortedSpotPixSum of the imageStats
* @return                      Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetSortedSpotPixelSum(kVsImageStats imageStats, kSize sortedSpotPixelSum);

/**
* Gets the sortedSpotPixelSum of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              SortedSpotPixSum.
*/
kVsFx(kSize) kVsImageStats_SortedSpotPixelSum(kVsImageStats imageStats);

/**
* Sets the pixelSum of image statistics, this is the sum of all pixel values in the frame.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @param   pixSum      PixSum of the imageStats
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetPixelSum(kVsImageStats imageStats, kSize pixelSum);

/**
* Gets the pixelSum of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              PixSum.
*/
kVsFx(kSize) kVsImageStats_PixelSum(kVsImageStats imageStats);

/**
* Sets the rawSpotCount count of image statistics, this is the number of unsorted spots reported in the frame.
*
* @public                   @memberof kVsImageStats
* @param   imageStats       ImageStats object
* @param   totalRawSpots    totalRawSpots of the imageStats
* @return                   Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetRawSpotsCount(kVsImageStats imageStats, kSize rawSpotsCount);

/**
* Gets the rawSpotCount of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              TotalRawSpots.
*/
kVsFx(kSize) kVsImageStats_RawSpotsCount(kVsImageStats imageStats);

/**
* Sets the sortedSpotsCount of image statistics, this is the number of sorted spots reported in the frame.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @param   totalSpots  TotalSortedSpots of the imageStats
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetSortedSpotsCount(kVsImageStats imageStats, kSize sortedSpotsCount);

/**
* Gets the SortedSpotsCount of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              TotalSortedSpots.
*/
kVsFx(kSize) kVsImageStats_SortedSpotsCount(kVsImageStats imageStats);

/**
* Sets the framePixelCount of image statistics, this is the number of pixels in the frame.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @param   totalPix    TotalPixels of the imageStats
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetFramePixelsCount(kVsImageStats imageStats, kSize framePixCount);

/**
* Gets the framePixelCount of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              TotalPixels.
*/
kVsFx(kSize) kVsImageStats_FramePixelsCount(kVsImageStats imageStats);

/**
* Sets the pseudoMedian index of image statistics, this is the spot index (Nth spot) that will be used as median value 
* and will be configured before executing algorithm when finding median value with median position
* for FPGA 6.1.2 and below.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @param   median      PseudoMedian of the imageStats
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetPseudoMedianIndex(kVsImageStats imageStats, kSize median);

/**
* Gets the pseudoMedian index of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              PseudoMedian.
*/
kVsFx(kSize) kVsImageStats_PseudoMedianIndex(kVsImageStats imageStats);

/**
* Sets the sortedSpotAverage of image statistics, this is the average of integer part of spot centroid sum.
* (spotIntCgSum/TotalSpots). For FPGA 6.1.3 and above, either average or the median value is selected to
* represent the spot centroid statistics.
*
* @public                       @memberof kVsImageStats
* @param   imageStats           ImageStats object
* @param   sortedSpotAverage    SortedSpotAverage of the imageStats
* @return                       Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetSortedSpotCentreAverage(kVsImageStats imageStats, kSize sortedSpotAverage);

/**
* Gets the sortedSpotAverage of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              SortedSpotAverage.
*/
kVsFx(kSize) kVsImageStats_SortedSpotCentreAverage(kVsImageStats imageStats);

/**
* Sets the rawSpotAverage of image statistics, this is the average of integer part of spot centroid sum.
* (spotIntCgSum/TotalSpots). For FPGA 6.1.3 and above, either average or the median value is selected to
* represent the spot centroid statistics.
* 
*
* @public                   @memberof kVsImageStats
* @param   imageStats       ImageStats object
* @param   rawSpotAverage   RawSpotAverage of the imageStats
* @return                   Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetRawSpotCentreAverage(kVsImageStats imageStats, kSize rawCentreAverage);

/**
* Gets the rawSpotAverage of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              RawSpotAverage.
*/
kVsFx(kSize) kVsImageStats_RawSpotCentreAverage(kVsImageStats imageStats);

/**
* Sets the rawMedian of image statistics, this will be the centroid of the spot of pseudo median index (row for ACCG/column for ARCG)\
* where CG count reaches provided median (for CGs 5.1.2.x and below).
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @param   rawMedian   RawMedianPostion of the imageStats
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetRawMedian(kVsImageStats imageStats, kSize rawMedian);

/**
* Gets the rawMedian of image statistic.
*
* @public                    @memberof kVsImageStats
* @param   rawMedianPostion  ImageStats object
* @return                    RawMedianPostion.
*/
kVsFx(kSize) kVsImageStats_RawMedian(kVsImageStats imageStats);

/**
* Sets the sortedMedianPosition of image statistics, this will be the centroid of the spot of pseudo median index (row for ACCG/column for ARCG)\
* where CG count reaches provided median (for CGs 5.1.2.x and below).
*
* @public                        @memberof kVsImageStats
* @param   imageStats            ImageStats object
* @param   sortedMedianPosition  SortedMedianPostion of the imageStats
* @return                        Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetSortedMedian(kVsImageStats imageStats, kSize sortedMedian);

/**
* Gets the sortedSpotAverage of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              SortedMedianPostion.
*/
kVsFx(kSize) kVsImageStats_SortedMedian(kVsImageStats imageStats);


/**
* Sets the rawSpotAverageSum of image statistics, this is average sum of raw spot pixel value.
*
* @public                   @memberof kVsImageStats
* @param   imageStats       ImageStats object
* @param   rawAverageSum    RawSpotPixAverageSum of the imageStats
* @return                   Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetRawSpotAverageSum(kVsImageStats imageStats, kSize rawAverageSum);

/**
* Gets the rawSpotAverageSum of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              RawSpotPixAverageSum.
*/
kVsFx(kSize) kVsImageStats_RawSpotAverageSum(kVsImageStats imageStats);

/**
* Sets the sortedSpotAverageSum of image statistics, this is average sum of sorted spot pixel value.
*
* @public                       @memberof kVsImageStats
* @param   imageStats           ImageStats object
* @param   sortedAverageSum     SortedSpotPixAverageSum of the imageStats
* @return                       Operation status.
*/
kVsFx(kStatus) kVsImageStats_SetSortedSpotAverageSum(kVsImageStats imageStats, kSize sortedAvergeSum);

/**
* Gets the sortedSpotAverageSum of image statistic.
*
* @public              @memberof kVsImageStats
* @param   imageStats  ImageStats object
* @return              SortedSpotPixAverageSum.
*/
kVsFx(kSize) kVsImageStats_SortedSpotAverageSum(kVsImageStats imageStats);

#include <kVision/Vs/kVsImageStats.x.h>

#endif /* #ifndef K_VISION_IMAGE_STATS_H */


