/**
* @file    kVsImageNoiseAlg.h
* @brief   Declares the kVsImageNoiseAlg class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kVS_IMAGE_NOISE_ALG_H
#define kVS_IMAGE_NOISE_ALG_H

#include <kVision/Common/kVision.h>

/**
* @class       kVsImageNoiseAlg
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       Provides a set of utility functions to perform temporal and spatial noise characterization on a set of images. 
*/
typedef kObject kVsImageNoiseAlg;

/**
* Constructs a kVsImageNoiseAlg object
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_Construct(kVsImageNoiseAlg* alg, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kVsImageNoiseAlg_MeasureSpatial, kVsImageNoiseAlg_MeasureTemporal).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kVsImageNoiseAlg_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_Setup(kVsImageNoiseAlg alg);

/**
* Measures average intensity and the smooth ground image from the individual samples
*
* @public                       @memberof kVsImageNoiseAlg
* @param   alg                  Algorithm object
* @param   images               Input image samples
* @param   count                Number of input image samples
* @param   averageIntensity     Optional output returning the average intensity of the input images. Can be used to normalize the output of spatial and temporal
*                               noise calculations to the absolute intensity of the image
* @param   averageImage         Optional output for the averaged ground image, which can be used in spatial noise processing (kVsImageNoiseAlg_MeasureSpatial)
*                               The output image matches the input image size and has pixel type of k64f
* @param   groundImage          Optional output for the smooth ground image (smoothness determined by kVsImageNoiseAlg_SetGroundSmoothWindow)
*                               The output image matches the input image size and has pixel type of k64f
* @return                       Operation status.
*/

kVsFx(kStatus) kVsImageNoiseAlg_MeasureGround(kVsImageNoiseAlg alg, kImage* images, kSize count, k64f* averageIntensity, kImage* averageImage, kImage* groundImage);

/**
* Measures the spatial standard deviation of a set of images. The steps are as follows:
*       - The individual sample images are averaged to produce a single image
*       - Optionally, the underlying smooth ground pattern is removed through the use of a large smoothing window (TBD)
*       - Standard deviation is calculated on the difference between the average image and the smooth ground image
*
* @public                       @memberof kVsImageNoiseAlg
* @param   alg                  Algorithm object
* @param   averageImage         Input image after averaging several raw image samples with pixel type of k64f
* @param   groundImage          Optional input image of smooth ground of the raw images to be subtracted prior to calculating spatial noise.
*                               Expected to be equal to the size of the image with pixel type of k64f.
*                               Can be set to kNULL to ignore smooth ground variation
* @param   intensityStdev       Spatial standard deviation of the average image samples
* @param   noiseImage           Risidual noise after subtracting the ground image from the average of the input samples. Pixel type is k64f. Used to characterize 
*                               the spatial standard deviation of the samples. The parameter is optional and can be set to kNULL.
* @return                       Operation status.
*/

kVsFx(kStatus) kVsImageNoiseAlg_MeasureSpatial(kVsImageNoiseAlg alg, kImage averageImage, kImage groundImage, k64f* intensityStdev, kImage* noiseImage);

/**
* Measures temporal standard deviation of a set of images. The steps are as follows:
*        - For each pixel position calculate the standard deviation across the collected image samples
*        - Calculate the cumulative standard deviation using average standard deviation of the individual pixels
*
* @public                       @memberof kVsImageNoiseAlg
* @param   alg                  Algorithm object
* @param   images               Input image samples
* @param   count                Number of input image samples
* @param   intensityStdev       Temporal standard deviation
* @param   stdevImage           Output image containing per-pixel standard deviation values. Pixel type is k64f. 
*                               The parameter is optional and can be set to kNULL
* @return                       Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_MeasureTemporal(kVsImageNoiseAlg alg, kImage* images, kSize count, k64f* intensityStdev, kImage* stdevImage);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_SetImageSize(kVsImageNoiseAlg alg, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_ImageSize(kVsImageNoiseAlg alg, kSize* imageWidth, kSize* imageHeight);

/**
* Set the size of the region of interest as a fraction of the full image size. Full image size is used by default, but 
* often it is desired to eliminate the periphery of the images due to non-uniformity of the lighting/optical setup
* By default, full image size is used
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @param   window      Algorithm window size (px)
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_SetRoi(kVsImageNoiseAlg alg, k64f roi);

/**
*  Get the size of the region of interest as a fraction of the full image size.
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @return              Size of the region of interest as a fraction of the full image size.
*/
kVsFx(k64f) kVsImageNoiseAlg_Roi(kVsImageNoiseAlg alg);

/**
* Set the size of the smoothing window for the ground calculation as a fraction of the full image width
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @param   window      Algorithm window size (0..1)
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseAlg_SetGroundSmoothWindow(kVsImageNoiseAlg alg, k64f window);

/**
*  Get the size of the smoothing window for the ground calculation as a fraction of the full image size
*
* @public              @memberof kVsImageNoiseAlg
* @param   alg         Algorithm object
* @return              Size of the smoothing window for the ground calculation as a fraction of the full image width (0..1)
*/
kVsFx(k64f) kVsImageNoiseAlg_GroundSmoothWindow(kVsImageNoiseAlg alg);

#include <kVision/Vs/kVsImageNoiseAlg.x.h>

#endif  /* kVS_IMAGE_NOISE_ALG_H */
