/** 
 * @file    kVsAdaptiveNorm.h
 * @brief   Declares the kVsAdaptiveNorm class
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.
 */

#ifndef K_VISION_ADAPTIVE_NORM_H
#define K_VISION_ADAPTIVE_NORM_H

#include <kVision/Vs/kVsImageFilter.h>

/**
* @class   kVsAdaptiveNorm
* @extends kObject
* @ingroup kVision-Vs
* @brief   An image filter class for performing an optimized adaptive normalization on images or 2D arrays using 
*          a specified window size. The class supports k8u, k16s, k32s, and k64f item types. Only support
*          data ranges from 0 tp 255. See kVsImageFilter for setting additional algoritm parameters 
*          and use of kVsImageFilter_Setup and kVsImageFilter_Apply functions.
*/
typedef kVsImageFilter kVsAdaptiveNorm;

/**
* Constructs a kVsAdaptiveNorm object
*
* @public              @memberof kVsAdaptiveNorm
* @param   filter      Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kVsAdaptiveNorm_Construct(kVsAdaptiveNorm* filter, kAlloc allocator);

/**
* Set the intensity truncation value for the image filter output (from 1 to 127). Using the maximum of 127 counts
* disables the truncation. The field is set to maximum by default.
*
* @public               @memberof kVsAdaptiveNorm
* @param   filter       Filter object
* @param   filterCap    Image filter truncation value
* @return               Operation status.
*/
kVsFx(kStatus) kVsAdaptiveNorm_SetFilterCap(kVsAdaptiveNorm filter, kSize filterCap);

/**
*  Get the image filter truncation value
*
* @public              @memberof kVsAdaptiveNorm
* @param   filter      Filter object
* @return              Image filter truncation value
*/
kVsFx(kSize) kVsAdaptiveNorm_FilterCap(kVsAdaptiveNorm filter);

/**
* Set the minimum amount of contrast within the filtering window to consider a patch valid. Intensity range lower than 
* that results in the pixel centered at that window to be invalidated (set to filter cap). Default is
* kVS_ADAPTIVE_NORM_DEFAULT_MIN_CONTRAST.
*
* @public               @memberof kVsAdaptiveNorm
* @param   filter       Filter object
* @param   minContrast  Minimum contrast
* @return               Operation status.
*/
kVsFx(kStatus) kVsAdaptiveNorm_SetMinContrast(kVsAdaptiveNorm filter, kSize minContrast);

/**
*  Get the minimum contrast
*
* @public              @memberof kVsAdaptiveNorm
* @param   filter      Filter object
* @return              Minimum contrast value
*/
kVsFx(kSize) kVsAdaptiveNorm_MinContrast(kVsAdaptiveNorm filter);

#include <kVision/Vs/kVsAdaptiveNorm.x.h>

#endif  /* K_VISION_ADAPTIVE_NORM_H */

