/**
* @file    kS3dVolumeCheckPlotter.h
* @brief   Declares the kS3dVolumeCheckPlot class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKPLOTTER_H
#define KVISION_KS3D_VOLUMECHECKPLOTTER_H

#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

#include <kFireSync/Data/kPlot.h>
#include <kApi/Utils/kUtils.h>
#include <kFireSync/Data/kGraphic.h>

/**
* @class       kS3dVolumeCheckPlotter
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Provides some utility functions to generate, manage and edit of a height map.
*/
typedef kObject kS3dVolumeCheckPlotter;

/**
* Constructs a kS3dVolumeCheckPlotter object
*
* @public              @memberof kS3dVolumeCheckPlotter
* @param   plotter     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckPlotter_Construct(kS3dVolumeCheckPlotter* plotter, kAlloc allocator);

/**
* Get display image. The image size is the same as the height map array.
*
* @public              @memberof kS3dVolumeCheckPlotter
* @param   plotter     kS3dVolumeCheckPlotter object.
* @return              display image.
*/
kVsFx(kImage)  kS3dVolumeCheckPlotter_ViewImage(kS3dVolumeCheckPlotter plotter);


/**
* Repaints the display image.
*
* @public              @memberof kS3dVolumeCheckPlotter
* @param   plotter     kS3dVolumeCheckPlotter object.
* @param   data        Height map of kArray2 <k32s>    
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckPlotter_UpdateViewImage(kS3dVolumeCheckPlotter plotter, kArray2 data);

/**
* Generating a kPlot object for the representation of height map.
*
* @public              @memberof kS3dVolumeCheckPlotter.
* @param   plotter     kS3dVolumeCheckPlotter object.
* @param   outPlot     Destination for the constructed handle of kPlot object. 
* @param   data        Height map of kArray2 <k32s>   
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckPlotter_PlotViewImage(kS3dVolumeCheckPlotter plotter, kPlot* outPlot, kArray2 data);

/**
* Set a boolean value to determine whether the color scale should be drawn.
*
* @public              @memberof kS3dVolumeCheckPlotter.
* @param   plotter     kS3dVolumeCheckPlotter object.
* @param   drawScale   Determines whether the color scale should be drawn. default = kTRUE
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckPlotter_SetDrawScale(kS3dVolumeCheckPlotter plotter, kBool drawScale);

/**
* Set the unit of the height map. default = "mm"
*
* @public              @memberof kS3dVolumeCheckPlotter.
* @param   plotter     kS3dVolumeCheckPlotter object.
* @param   zUnit       unit of the height map.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckPlotter_SetZUnit(kS3dVolumeCheckPlotter plotter, const kChar* zUnit);

/**
* Set scale of the  height map in mm. 
*
* @public              @memberof kS3dVolumeCheckPlotter.
* @param   plotter     kS3dVolumeCheckPlotter object.
* @param   scale       Scaling of the height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckPlotter_SetScale(kS3dVolumeCheckPlotter plotter, kPoint3d64f scale);


/**
* Set the draw range in z direction.
*
* @public              @memberof kS3dVolumeCheckPlotter.
* @param   plotter     kS3dVolumeCheckPlotter object.
* @param   minZ        Minimum limit of the color coding. 
* @param   maxZ        Maximum limit of the color coding. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckPlotter_SetViewRange(kS3dVolumeCheckPlotter plotter, k64f minZ, k64f maxZ);

#include <kVision/S3d/kS3dVolumeCheckPlotter.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKPLOTTER_H */
