/**
* @file    kS3dVolumeCheckAlign.h
* @brief   Declares the kS3dVolumeCheckAlign class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKALIGN_H
#define KVISION_KS3D_VOLUMECHECKALIGN_H

#include <kVision/S3d/kS3dVolumeCheckDef.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

/**
* @class       kS3dVolumeCheckAlign
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Class for upscaling input data to internal kArray2 < kPoint3d32s > 
*              aligment and rotation of the surface and segmenting  the data.
*/
typedef kObject kS3dVolumeCheckAlign;

/**
* Constructs a kS3dVolumeCheckAlign object
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_Construct(kS3dVolumeCheckAlign* align, kAlloc allocator);


/**
* Set input data for the VolumeCheck measurement.
* The accepted kArray2 type are:
*     kArray2 < kPoint3d16s >   xyScale == zScale = range resolution in mm
*     kArray2 < kPoint3d32s >   xyScale == zScale = range resolution in mm
*     kArray2 < kPoint3d32f >   xyScale == zScale = 1
*     kArray2 < kPoint3d64f >   xyScale == zScale = 1
*     kArray2 < k16s >          xyScale= pixel pitch in mm, zScale = z resolution in mm
*     kArray2 < k32s >          xyScale= pixel pitch in mm, zScale = z resolution in mm
*     kArray2 < k32f >          xyScale= pixel pitch in mm, zScale = 1
*     kArray2 < k64f >          xyScale= pixel pitch in mm, zScale = 1
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @param   input       input data of kArray2.
* @param   xyScale     pixel pitch in mm.
* @param   zScale      z resolution or range resolution in mm.
* @param   mode        Rotation mode of the input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_SetInputData(kS3dVolumeCheckAlign align, kArray2 input, k64f xyScale, k64f zScale, kS3dVolumeCheckSensorRotationMode mode);

/**
* Get maximum iterative steps for the surface alignment. 
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              maximum iterative steps.
*/
kVsFx(k16s) kS3dVolumeCheckAlign_MaxRepeat(kS3dVolumeCheckAlign align);

/**
* Set maximum iterative steps for the surface alignment.
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @param   maxRepeat   Maximale  The alignment is performed iteratively
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_SetMaxRepeat(kS3dVolumeCheckAlign align, k16s maxRepeat);


/**
* Alignment of the point clouds to the upper reference plane
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_MaxZeroPlaneAlign(kS3dVolumeCheckAlign align);

/**
* Segment chamber area, calculate the chamber center and shift the point clouds to the center.
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @param   offset      offset values between the defined coordinate origin and calculated roughly chamber center in mm.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_SegmentChamberCenter(kS3dVolumeCheckAlign align, kPoint64f offset);

/**
* Makes a 2D transformation of the point cloud.
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @param   angle       Rotation angle in degree.
* @param   x0          Movement in x direction in mm.
* @param   y0          Movement in y direction in mm.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_Transform2D(kS3dVolumeCheckAlign align, k64f angle, k64f x0, k64f y0);

/**
* Get the range rasolution of the point cloud with kArray2 < kPoint3d32s >
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Range rasolution.
*/
kVsFx(k64f) kS3dVolumeCheckAlign_RangeScale(kS3dVolumeCheckAlign align);

/**
* Get the point cloud data of kArray2 < kPoint3d32s >
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              point cloud data of kArray2 < kPoint3d32s >.
*/
kVsFx(kArray2) kS3dVolumeCheckAlign_Data(kS3dVolumeCheckAlign align);

/**
* Get the point cloud data of kArray2 < kPoint3d32s >
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              valid map of kArray2 < k8u >.
*/
kVsFx(kArray2) kS3dVolumeCheckAlign_Valid(kS3dVolumeCheckAlign align);

/**
* Get the slope angle of the plane surface normal vector corresponding to the upper reference plane
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Slope angle of the surface normal vector.
*/
kVsFx(k64f) kS3dVolumeCheckAlign_SlopeAngle(kS3dVolumeCheckAlign align);

/**
* Get the orientation angle of the plane surface normal vector corresponding to the upper reference plane
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Orientation angle of the surface normal vector.
*/
kVsFx(k64f) kS3dVolumeCheckAlign_PolarAngle(kS3dVolumeCheckAlign align);

/**
* Get the actual center of the combustion chamber by averaging the valid point position 
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Actual center of the combustion chamber.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlign_Center(kS3dVolumeCheckAlign align);

/**
* Return warning hints 
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Warning hints.
*/
kVsFx(k32s) kS3dVolumeCheckAlign_WarningStatus(kS3dVolumeCheckAlign align);

/**
* Eliminates the invalid outliers from the record.
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlign_CheckValidRange(kS3dVolumeCheckAlign align);

/**
* Get the integer height threshold used in the calss for filling invalid points with the template.
*
* @public              @memberof kS3dVolumeCheckAlign
* @param   align       kS3dVolumeCheckAlign object.
* @return              The threshold used in the calss.
*/
kVsFx(k32s) kS3dVolumeCheckAlign_Threshold(kS3dVolumeCheckAlign align);

#include <kVision/S3d/kS3dVolumeCheckAlign.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKALIGN_H */
