/**
* @file    kS3dStereoProfilerBase.h
* @brief   Declares the kS3dStereoProfilerBase class.
*
* @internal
* Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef K_VISION_S3D_STEREO_PROFILER_BASE_H
#define K_VISION_S3D_STEREO_PROFILER_BASE_H

#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArray3.h>
#include <kApi/Data/kMath.h>

#include <kFireSync/Data/kSpot.h>

#include <kVision/L3d/kL3dUtilities.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/Vs/kVsJobQueue.h>
#include <kVision/S3d/kS3dStereoCal.h>

//////////////////////////////////////////////////////////////////////////
// Input params struct
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dStereoProfilerParams
{
    kL3dCameraWindow window[2];
    kL3dTransform3d transform;  // No longer needed during initialization. Initialize to identity and remove

    k64f resolution;
    k64f phaseMergeThreshold;   // Unused and should be removed

    //expansion correction parameters
    k64f viewDepth;         //Distance from the FOV midrange to the baseline. Should be negative, mm.
    k64f opticalCentre;     //Distance from the FOV midrange to the optical center (center ray convergence) Negative is closer to sensor.
    k64f expansionCoeff;    //Expansion coefficient (CTE) of the spine material.

} kS3dStereoProfilerParams;

//////////////////////////////////////////////////////////////////////////
// Rectification context passed to point rectification function 
// allocated in host or device
//////////////////////////////////////////////////////////////////////////
//typedef kPointer kS3dStereoProfilerRectContext;

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
* @class       kS3dStereoProfilerBase
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs 3D point generation. Base class for kS3dStereoProfilerHost & kS3dStereoProfilerCuda (CPU & GPU implementations)
*/
typedef kObject kS3dStereoProfilerBase;

/**
* Calculate the internal size.
* Should probably deprecate this in favor of VSize() 
* Host and Cuda should overwrite this, a.t.m. they don't have additional allocations and Cuda has GPU allocations so this is not trivial as two numbers should be reported 
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Operation status
*/

kVsFx(kSize) kS3dStereoProfilerBase_MemoryUsage(kS3dStereoProfilerBase profiler); // deprecate in favor of VSize()

//////////////////////////////////////////////////////////////////////////
// Get/Set 
//////////////////////////////////////////////////////////////////////////

/**
* Set lookup type for kS3dSurfaceGenerator
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   type        kS3dStripeLookupType
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_SetLookupType(kS3dStereoProfilerBase profiler, kS3dStripeLookupType type);

/**
* Get lookup type for kS3dSurfaceGenerator
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              kS3dStripeLookupType 
*/

kVsFx(kS3dStripeLookupType) kS3dStereoProfilerBase_LookupType(kS3dStereoProfilerBase profiler);

/**
* Set point transform applied during kS3dStereoProfilerBase_LookupMap()
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   transform   Type
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_SetTransform(kS3dStereoProfilerBase profiler, const kL3dTransform3d* transform);

/**
* Get point transform applied during kS3dStereoProfilerBase_LookupMap()
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Transformation
*/

kVsFx(const kL3dTransform3d*) kS3dStereoProfilerBase_Transform(kS3dStereoProfilerBase profiler);

/**
* Set ROI applied during kS3dStereoProfilerBase_LookupMap()
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   roi         Region if interest 
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_SetActiveAreaRoi(kS3dStereoProfilerBase profiler, const kRect3d64f* roi);

/**
* Get ROI applied during kS3dStereoProfilerBase_LookupMap()
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Region if interest
*/

kVsFx(const kRect3d64f*) kS3dStereoProfilerBase_ActiveAreaRoi(kS3dStereoProfilerBase profiler);

/**
* Get reference temperature.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Reference temperature
*/

kVsFx(k64f) kS3dStereoProfilerBase_ReferenceTemperature(kS3dStereoProfilerBase profiler);

/**
* Set sensor temperature.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   temperature Sensor temperature
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_SetTemperature(kS3dStereoProfilerBase profiler, k64f temperature);

/**
* Get reference temperature.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Temperature
*/

kVsFx(k64f) kS3dStereoProfilerBase_Temperature(kS3dStereoProfilerBase profiler);

/**
* Get expansion correction enabled flag.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Flag
*/

kVsFx(kBool) kS3dStereoProfilerBase_ExpansionCorrectionEnabled(kS3dStereoProfilerBase profiler);

/**
* Set expansion correction enabled flag.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   enabled     Flag
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_EnableExpansionCorrection(kS3dStereoProfilerBase profiler, kBool enabled);

/**
* Returns the view index of the sensor stereo pair with lower X coordinate relative to the FOV of the sensor. For current sensor models with symmetric
* camera configuration the function can be used directly to determine how camera indices map to physical positions of the cameras.
*
* The function should be used to determine the direction of the Z axis relative to image column coordinates. Assuming standard individual camera
* orientation (Columns increasing with increasing X. No other orientation is supported), foreground points appear left of the background on the
* 'left' camera and right of the background on the 'right' camera. Existing correspondence methodology (kS3dStripeProcessor_ProcessPhaseMap) uses
* this information to eliminate duplicate correspondences
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Returns the view index of the stereo pair with the lower X coordinate relative to the field of view of the sensor.
*/

kVsFx(kSize) kS3dStereoProfilerBase_LeftViewIndex(kS3dStereoProfilerBase profiler);

/**
* Get min disparity value.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Disparity min
*/

kVsFx(k32s) kS3dStereoProfilerBase_DisparityMin(kS3dStereoProfilerBase profiler);

/**
* Get max disparity value.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Disparity max
*/

kVsFx(k32s) kS3dStereoProfilerBase_DisparityMax(kS3dStereoProfilerBase profiler);

/**
* Set 3D LUT table resolution.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   resolution  Resolution
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_SetResolution(kS3dStereoProfilerBase profiler, k64f resolution);

/**
* Get 3D LUT table resolution.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Resolution
*/

kVsFx(k64f) kS3dStereoProfilerBase_Resolution(kS3dStereoProfilerBase profiler);

/**
* Set 2D LUT table resolution.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   resolution  Projection resolution
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_SetProjectionResolution(kS3dStereoProfilerBase profiler, k64f resolution);

/**
* Get 2D LUT table resolution.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Resolution
*/

kVsFx(k64f) kS3dStereoProfilerBase_ProjectionResolution(kS3dStereoProfilerBase profiler);

/**
* Returns total pre-transformation range volume of the constructed lookup tables. Output is a 3D rectangle with mm as units. Can be taken advantage of by
* processing algorithms which set their parameters based on a percentage of the total operating volume.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Total pre-transformation range volume of the constructed lookup tables
*/

kVsFx(const kRect3d64f*) kS3dStereoProfilerBase_CalibrationVolume(kS3dStereoProfilerBase profiler);

/**
* Update the window params for each camera. 
* Used by rectification. 
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @param   window0     Window 0
* @param   window1     Window 1
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_UpdateWindow(kS3dStereoProfilerBase profiler, const kL3dCameraWindow* window0, const kL3dCameraWindow* window1);

/**
* Query thermal expansion parameters.
*
* @public                  @memberof kS3dStereoProfilerBase
* @param   profiler        Profiler object
* @param   viewDepth       Distance from the FOV midrange to the baseline. Should be negative, mm.
* @param   opticalCentre   Distance from the FOV midrange to the optical center (center ray convergence) Negative is closer to sensor.
* @param   expansionCoeff  Expansion coefficient (CTE) of the spine material.
* @return                  Operation status
*/

kVsFx(kStatus) kS3dStereoProfilerBase_TermalExpansionParams(kS3dStereoProfilerBase profiler, k64f* viewDepth, k64f* opticalCentre, k64f* expansionCoeff);

/**
* Query if camera image was transposed.
*
* @public                  @memberof kS3dStereoProfilerBase
* @param   profiler        Profiler object
* @param   viewIndex       Camera index (0, 1)
* @return                  kTRUE if images were transposed
*/
kVsFx(kBool) kS3dStereoProfilerBase_TransposeAt(kS3dStereoProfilerBase profiler, kSize viewIndex);

//////////////////////////////////////////////////////////////////////////
//
// Hardware specific context object 
// Allocated on host or device
//
//////////////////////////////////////////////////////////////////////////

typedef kPointer kS3dStereoProfilerHwContext;

/**
* Acquire hardware context for executing certain profiler functions without the profiler object at hand.
*
* @public              @memberof kS3dStereoProfilerBase
* @param   profiler    Profiler object
* @return              Pointer to the hardware context object allocated on host.
*/

kVsFx(kS3dStereoProfilerHwContext) kS3dStereoProfilerBase_HwContext(kS3dStereoProfilerBase profiler);

/**
* Rectifies fractional image coordinates. Both coordinates must be scaled to kSPOT_CENTRE_SCALE
*
* @public              @memberof kS3dStereoProfilerHwContext
* @param   hwContext   kS3dStereoProfilerHwContext object
* @param   viewIndex   Index of the camera
* @param   x           Coordinate x
* @param   y           Coordinate y
* @param   xProj       Output coordinate x
* @param   yProj       Output coordinate x
* @return              Operation status
*/

kCudaInlineFx(void) kS3dStereoProfilerHwContext_Rectify(kS3dStereoProfilerHwContext hwContext, kSize viewIndex, k32s x, k32s y, k16s* xProj, k16s* yProj);

/**
* Rectifies fractional image coordinates. Both coordinates must be scaled to kSPOT_CENTRE_SCALE, however, Y values must fall on integer rows.
* No interpolation across rows is performed
*
* @public              @memberof kS3dStereoProfilerHwContext
* @param   hwContext   kS3dStereoProfilerHwContext object
* @param   viewIndex   Index of the camera
* @param   x           Coordinate x
* @param   y           Coordinate y
* @param   xProj       Output coordinate x
* @param   yProj       Output coordinate x
* @return              Operation status
*/

kCudaInlineFx(void) kS3dStereoProfilerHwContext_RectifyLinear(kS3dStereoProfilerHwContext hwContext, kSize viewIndex, k32s x, k32s y, k16s* xProj, k16s* yProj);

/**
* Converts x0 and x1 projections to 3D point.
*
* @public              @memberof kS3dStereoProfilerHwContext
* @param   hwContext   kS3dStereoProfilerHwContext object
* @param   xProj0      Phase projection 0
* @param   xProj1      Phase projection 1
* @param   yProj       Phase projection y
* @param   result      Out 3D point
* @return              Operation status
*/

kCudaInlineFx(void) kS3dStereoProfilerHwContext_LookupDisparity(kS3dStereoProfilerHwContext hwContext, k16s xProj0, k16s xProj1, k16s yProj, kPoint3d16s* result);


/**
* Converts xProj and yProj to 3D point.
*
* @public              @memberof kS3dStereoProfilerHwContext
* @param   hwContext   kS3dStereoProfilerHwContext object
* @param   viewIndex   Camera index
* @param   xProj       Point projection x
* @param   xProj       Point projection y 
* @param   phase       Phase value
* @param   result      Out 3D point
* @return              Operation status
*/

kCudaInlineFx(void) kS3dStereoProfilerHwContext_LookupPhase(kS3dStereoProfilerHwContext hwContext, kSize viewIndex, k16s xProj, k16s yProj, k32s phase, kPoint3d16s* result);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dStereoProfilerBase.x.h>

#endif // K_VISION_S3D_STEREO_PROFILER_BASE_H
