/**
* @file    kS3dStereoProfiler.h
* @brief   Declares the kS3dStereoProfiler class.
*
* @internal
* Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef K_VISION_S3D_STEREO_PROFILER_H
#define K_VISION_S3D_STEREO_PROFILER_H

#include <kVision/S3d/kS3dStereoProfilerBase.h>

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef enum
{
    kS3D_STEREO_PROFILER_HOST = 0,
    kS3D_STEREO_PROFILER_CUDA = 1,
    kS3D_STEREO_PROFILER_CUDA_FORCE = 2,

} kS3dStereoProfilerType;


/**
* @class       kS3dStereoProfiler
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs 3D point generation. This is a wrapper class for the CPU/GPU implementations of the stereo profiler class
*/
typedef kObject kS3dStereoProfiler;

/**
* Constructs a kS3dStereoProfiler object
*
* @public               @memberof kS3dStereoProfilerHost
* @param   profiler     Profiler object
* @param   profilerType Defines if CPU or GPU profiler is used (HOST/CUDA)
* @param   params       Profiler params
* @param   cal          Stereo calibration object
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/

kVsFx(kStatus) kS3dStereoProfiler_Construct(kS3dStereoProfiler* profiler, kS3dStereoProfilerType profilerType, const kS3dStereoProfilerParams* params, kS3dStereoCal cal, kAlloc alloc);

/**
* Rectifies fractional image coordinates. Both coordinates must be scaled to kSPOT_CENTRE_SCALE
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   viewIndex   Index of the camera 
* @param   x           Coordinate x
* @param   y           Coordinate y  
* @param   xProj       Output coordinate x
* @param   yProj       Output coordinate x
* @return              Operation status
*/

kInlineFx(void) kS3dStereoProfiler_Rectify(kS3dStereoProfiler profiler, kSize viewIndex, k32s x, k32s y, k16s* xProj, k16s* yProj);

/**
* Rectifies fractional image coordinates. Both coordinates must be scaled to kSPOT_CENTRE_SCALE, however, Y values must fall on integer rows. 
* No interpolation across rows is performed
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   viewIndex   Index of the camera
* @param   x           Coordinate x
* @param   y           Coordinate y
* @param   xProj       Output coordinate x
* @param   yProj       Output coordinate x
* @return              Operation status
*/

kInlineFx(void) kS3dStereoProfiler_RectifyLinear(kS3dStereoProfiler profiler, kSize viewIndex, k32s x, k32s y, k16s* xProj, k16s* yProj);

/**
* Converts x0 and x1 projections to 3D point.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   xProj0      Phase projection 0
* @param   xProj1      Phase projection 1
* @param   yProj       Phase projection y
* @param   result      Out 3D point
* @return              Operation status
*/

kInlineFx(void) kS3dStereoProfiler_LookupDisparity(kS3dStereoProfiler profiler, k16s xProj0, k16s xProj1, k16s yProj, kPoint3d16s* result);

/**
* Converts xProj and yProj to 3D point.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   viewIndex   Camera index
* @param   xProj       Point projection x
* @param   xProj       Point projection y
* @param   phase       Phase value
* @param   result      Out 3D point
* @return              Operation status
*/

kInlineFx(void) kS3dStereoProfiler_LookupPhase(kS3dStereoProfiler profiler, kSize viewIndex, k16s xProj, k16s yProj, k32s phase, kPoint3d16s* result);

/**
* Calculate the internal size.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Operation status
*/

kInlineFx(kSize) kS3dStereoProfiler_MemoryUsage(kS3dStereoProfiler profiler); // deprecate in favor of VSize()

//////////////////////////////////////////////////////////////////////////
// Get/Set 
//////////////////////////////////////////////////////////////////////////

/**
* Set lookup type for kS3dStereoProfiler_LookupMap()
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   type        kS3dStripeLookupType
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_SetLookupType(kS3dStereoProfiler profiler, kS3dStripeLookupType type);

/**
* Get lookup type for kS3dStereoProfiler_LookupMap()
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              kS3dStripeLookupType
*/

kVsFx(kS3dStripeLookupType) kS3dStereoProfiler_LookupType(kS3dStereoProfiler profiler);

/**
* Set point transform applied during kS3dStereoProfiler_LookupMap()
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   transform   Type
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_SetTransform(kS3dStereoProfiler profiler, const kL3dTransform3d* transform);

/**
* Get point transform applied during kS3dStereoProfiler_LookupMap()
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Transformation
*/

kVsFx(const kL3dTransform3d*) kS3dStereoProfiler_Transform(kS3dStereoProfiler profiler);

/**
* Set ROI applied during kS3dStereoProfiler_LookupMap()
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   roi         Region if interest
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_SetActiveAreaRoi(kS3dStereoProfiler profiler, const kRect3d64f* roi);

/**
* Get ROI applied during kS3dStereoProfiler_LookupMap()
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Region if interest
*/

kVsFx(const kRect3d64f*) kS3dStereoProfiler_ActiveAreaRoi(kS3dStereoProfiler profiler);

/**
* Get reference temperature.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Reference temperature
*/

kVsFx(k64f) kS3dStereoProfiler_ReferenceTemperature(kS3dStereoProfiler profiler);

/**
* Set sensor temperature.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   temperature Sensor temperature
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_SetTemperature(kS3dStereoProfiler profiler, k64f temperature);

/**
* Get reference temperature.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Temperature
*/

kVsFx(k64f) kS3dStereoProfiler_Temperature(kS3dStereoProfiler profiler);

/**
* Get expansion correction enabled flag.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Flag
*/

kVsFx(kBool) kS3dStereoProfiler_ExpansionCorrectionEnabled(kS3dStereoProfiler profiler);

/**
* Set expansion correction enabled flag.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   enabled     Flag
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_EnableExpansionCorrection(kS3dStereoProfiler profiler, kBool enabled);

/**
* Returns the view index of the sensor stereo pair with lower X coordinate relative to the FOV of the sensor. For current sensor models with symmetric
* camera configuration the function can be used directly to determine how camera indices map to physical positions of the cameras.
*
* The function should be used to determine the direction of the Z axis relative to image column coordinates. Assuming standard individual camera
* orientation (Columns increasing with increasing X. No other orientation is supported), foreground points appear left of the background on the
* 'left' camera and right of the background on the 'right' camera. Existing correspondence methodology (kS3dStripeProcessor_ProcessPhaseMap) uses
* this information to eliminate duplicate correspondences
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Returns the view index of the stereo pair with the lower X coordinate relative to the field of view of the sensor.
*/

kVsFx(kSize) kS3dStereoProfiler_LeftViewIndex(kS3dStereoProfiler profiler);

/**
* Get min disparity value.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Disparity min
*/

kVsFx(k32s) kS3dStereoProfiler_DisparityMin(kS3dStereoProfiler profiler);

/**
* Get max disparity value.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Disparity max
*/

kVsFx(k32s) kS3dStereoProfiler_DisparityMax(kS3dStereoProfiler profiler);

/**
* Set 3D LUT table resolution.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   resolution  Resolution
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_SetResolution(kS3dStereoProfiler profiler, k64f resolution);

/**
* Get 3D LUT table resolution.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Resolution
*/

kVsFx(k64f) kS3dStereoProfiler_Resolution(kS3dStereoProfiler profiler);

/**
* Set 2D LUT table resolution.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   resolution  Projection resolution
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_SetProjectionResolution(kS3dStereoProfiler profiler, k64f resolution);

/**
* Get 2D LUT table resolution.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Resolution
*/

kVsFx(k64f) kS3dStereoProfiler_ProjectionResolution(kS3dStereoProfiler profiler);

/**
* Returns total pre-transformation range volume of the constructed lookup tables. Output is a 3D rectangle with mm as units. Can be taken advantage of by
* processing algorithms which set their parameters based on a percentage of the total operating volume.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Total pre-transformation range volume of the constructed lookup tables
*/

kVsFx(const kRect3d64f*) kS3dStereoProfiler_CalibrationVolume(kS3dStereoProfiler profiler);

/**
* Update the window params for each camera.
* Used by rectification.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @param   window0     Window 0
* @param   window1     Window 1
* @return              Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_UpdateWindow(kS3dStereoProfiler profiler, const kL3dCameraWindow* window0, const kL3dCameraWindow* window1);

/**
* Acquire hardware context for executing certain profiler functions without the profiler object at hand.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Pointer to the hardware context object allocated on host.
*/

kVsFx(kS3dStereoProfilerHwContext) kS3dStereoProfiler_HwContext(kS3dStereoProfiler profiler);

/**
* Acquire device ready (Cuda GPU) hardware context for executing certain profiler functions without the profiler object at hand.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              Pointer to the hardware context object allocated on device.
*/

kVsFx(kS3dStereoProfilerHwContext) kS3dStereoProfiler_HwContextCuda(kS3dStereoProfiler profiler);

/**
* Determine if Cuda is enabled and supported by the current profiler object.
*
* @public              @memberof kS3dStereoProfiler
* @param   profiler    Profiler object
* @return              kTRUE if Cuda is on.
*/

kVsFx(kBool) kS3dStereoProfiler_IsCuda(kS3dStereoProfiler profiler);

/**
* Query thermal expansion parameters.
*
* @public                  @memberof kS3dStereoProfiler
* @param   profiler        Profiler object
* @param   viewDepth       Distance from the FOV midrange to the baseline. Should be negative, mm.
* @param   opticalCentre   Distance from the FOV midrange to the optical center (center ray convergence) Negative is closer to sensor.
* @param   expansionCoeff  Expansion coefficient (CTE) of the spine material.
* @return                  Operation status
*/

kVsFx(kStatus) kS3dStereoProfiler_TermalExpansionParams(kS3dStereoProfiler profiler, k64f* viewDepth, k64f* opticalCentre, k64f* expansionCoeff);

/**
* Query if camera image was transposed.
*
* @public                  @memberof kS3dStereoProfiler
* @param   profiler        Profiler object
* @param   viewIndex       Camera index (0, 1)
* @return                  kTRUE if images were transposed
*/
kVsFx(kBool) kS3dStereoProfiler_TransposeAt(kS3dStereoProfiler profiler, kSize viewIndex);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dStereoProfiler.x.h>

#endif // K_VISION_S3D_STEREO_PROFILER_H
