/**
* @file    kS3dSpeckleDisparityFilter.h
* @brief   Declares the kS3dSpeckleDisparityFilter class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef KS3D_SPECKLE_DISPARITY_FILTER_H
#define KS3D_SPECKLE_DISPARITY_FILTER_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dSpeckleDisparityFilter
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Provides a set of utility functions to perform filtering of disparity images output from texture matching algorithms (BM, SGBM). 
*/
typedef kObject kS3dSpeckleDisparityFilter;

/**
* Constructs a kS3dSpeckleDisparityFilter object
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_Construct(kS3dSpeckleDisparityFilter* filter, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dSpeckleDisparityFilter_AdaptiveNorm, kS3dSpeckleDisparityFilter_StretchHistogram).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dSpeckleDisparityFilter_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_Setup(kS3dSpeckleDisparityFilter filter);

/**
* Removes outliers from a disparity map. Outliers are defined as small self contained regions, or groups of points within a large region
* separated from other values within the region by a large step in the disparity value. Minimum size of a valid region is configured through 
* kS3dSpeckleDisparityFilter_SetRegionMinCount. Maximum difference in adjacent disparity values to be considered a part of the same region
* is configured through kS3dSpeckleDisparityFilter_SetRegionStepThreshold function.
*
* @public               @memberof kS3dSpeckleDisparityFilter
* @param   filter       Filter object
* @param   disparityIn  Input disparity map
* @param   disparityOut Output disparity map (can be the same as the input) 
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_RemoveOutliers(kS3dSpeckleDisparityFilter filter, kArray2 disparityIn, kArray2 disparityOut);

/**
* Removes regions of low texture from the disparity map. These regions are determined by examining the input images for values equal to
* the filter cap (filtered 'zero' value). This is needed for algorithms such as SGBM, where this operation isn't done implicitly. 
*
* @public               @memberof kS3dSpeckleDisparityFilter
* @param   filter       Filter object
* @param   image0       Left input image
* @param   image1       Right input image
* @param   disparityIn  Input disparity map
* @param   disparityOut Output disparity map (can be same as the input)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_FilterTexture(kS3dSpeckleDisparityFilter filter, kImage image0, kImage image1, kArray2 disparityIn, kArray2 disparityOut);

/**
* Applies a moving average filter to the disparity map. The size of the moving average window is set using kS3dSpeckleDisparityFilter_SetSmootWindow function. 
* The user can also control filling and extrapolation around null values by adjusting the minimum fill ratio within the smoothing area. 
* See kS3dSpeckleDisparityFilter_SetSmoothMinFillRatio for further details. 
* 
* This function is useful when performing texture matching at multiple resolutions. Smoothed 'coarse' disparity maps can be used as templates for finer steps. 
*
* @public               @memberof kS3dSpeckleDisparityFilter
* @param   filter       Filter object
* @param   disparityIn  Input disparity map
* @param   disparityOut Output disparity map (can be the same as the input)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_MovingAverage(kS3dSpeckleDisparityFilter filter, kArray2 disparityIn, kArray2 disparityOut);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetImageSize(kS3dSpeckleDisparityFilter filter, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_ImageSize(kS3dSpeckleDisparityFilter filter, kSize* imageWidth, kSize* imageHeight);

/**
* Set the value of the filter cap used during image pre-filtering. For the description of the filter cap see kS3dSpeckleDisparityFilter class
* description. The value must be set for the low texture removal (kS3dSpeckleDisparityFilter_FilterTexture) to function correctly
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   filterCap   Filter cap value
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetFilterCap(kS3dSpeckleDisparityFilter filter, k32u filterCap);

/**
* Gets the filter cap value
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Filter cap value
*/
kVsFx(k32u) kS3dSpeckleDisparityFilter_FilterCap(kS3dSpeckleDisparityFilter filter);

/**
* Set output disparity bit depth. The value must match the setting used to calculate the disparity (see kS3dSpeckleMatchAlg_SetDisparityBitDepth) 
* for low texture removal (kS3dSpeckleDisparityFilter_FilterTexture) to function correctly 
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   bitDepth    Bit depth for disparity calculation and output
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetDisparityBitDepth(kS3dSpeckleDisparityFilter filter, k32u bitDepth);

/**
* Get disparity bit depth setting
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Bit depth for disparity calculation and output
*/
kVsFx(k32u) kS3dSpeckleDisparityFilter_DisparityBitDepth(kS3dSpeckleDisparityFilter filter);

/**
* Set the minimum number of values within a continuous region in a disparity map for that region to be considered valid 
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   count       Minimium number of disparity values in a valid continuous region
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetRegionMinCount(kS3dSpeckleDisparityFilter filter, k32u count);

/**
* Get mnimium number of disparity values in a valid continuous region
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Mnimium number of disparity values in a valid continuous region
*/
kVsFx(k32u) kS3dSpeckleDisparityFilter_RegionMinCount(kS3dSpeckleDisparityFilter filter);

/**
* Set maximum difference in adjacent disparity values to be considered part of the same region
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   threshold   Maximum difference in adjacent disparity values to be considered part of the same region
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetRegionStepThreshold(kS3dSpeckleDisparityFilter filter, k32u threshold);

/**
* Get maximum difference in adjacent disparity values to be considered part of the same region
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Maximum difference in adjacent disparity values to be considered part of the same region
*/
kVsFx(k32u) kS3dSpeckleDisparityFilter_RegionStepThreshold(kS3dSpeckleDisparityFilter filter);

/**
* Sets the size of the smoothing window used in kS3dSpeckleDisparityFilter_MovingAverage
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   window      Size of the moving average window (px)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetSmootWindow(kS3dSpeckleDisparityFilter filter, k32u window);

/**
* Gets the size of the smoothing window used in kS3dSpeckleDisparityFilter_MovingAverage
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Size of the moving average window (px)
*/
kVsFx(k32u) kS3dSpeckleDisparityFilter_SmoothWindow(kS3dSpeckleDisparityFilter filter);

/**
* Sets the minimum fill ratio within a smoothing window used in kS3dSpeckleDisparityFilter_MovingAverage. If the fraction
* of valid values within the smoothing window area does not reach the fill ratio, the output value at the centre of the window
* is set to null. Using a low minimum fill ratio can be used to 'pad' the smoothed disparity beyond the original borders
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @param   fillRatio   Minimum fill ratio within a smoothing window to produce a valid smoothed output (range 0...1)      
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleDisparityFilter_SetSmoothMinFillRatio(kS3dSpeckleDisparityFilter filter, k64f fillRatio);

/**
* Gets the minimum fill ratio within a smoothing window
*
* @public              @memberof kS3dSpeckleDisparityFilter
* @param   filter      Filter object
* @return              Minimum fill ratio within a smoothing window
*/
kVsFx(k64f) kS3dSpeckleDisparityFilter_SmoothMinFillRatio(kS3dSpeckleDisparityFilter filter);

#include <kVision/S3d/kS3dSpeckleDisparityFilter.x.h>

#endif  /* KS3D_SPECKLE_DISPARITY_FILTER_H */
