/**
* @file    kS3dRangeFilter.h
* @brief   Declares the kS3dRangeFilter class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_RANGE_FILTER_H
#define kS3D_RANGE_FILTER_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dRangeFilter
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs filtering operations on maps of 3D range data
*/
typedef kObject kS3dRangeFilter;

/**
* Constructs a kS3dRangeFilter object
*
* @public              @memberof kS3dRangeFilter
* @param   filter     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_Construct(kS3dRangeFilter* filter, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dRangeFilter_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_Setup(kS3dRangeFilter filter);

/**
* Cleans boundaries around large steps in Z values and eliminates small patches. Operation is performed in place on the input range data. 
* The criteria for continuity and minimum path size are specified via kS3dRangeFilter_SetMinRunLength and kS3dRangeFilter_SetRunStepThreshold
*
* @public                   @memberof kS3dRangeFilter
* @param   filter           Filter object
* @param   map              Input range map. Expected kArray2<kPoint3d16s> of size specified via kS3dRangeFilter_SetMapSize
*                           filtered result
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_RemoveOutliers(kS3dRangeFilter filter, kArray2 map);

/**
* Sets the size of the expected input maps.
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @param   mapWidth    Width of the input maps (dimension 1)
* @param   mapHeight   Height of the input maps (dimension 0)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_SetMapSize(kS3dRangeFilter filter, kSize mapWidth, kSize mapHeight);

/**
* Gets the size of input maps
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @param   mapWidth    Output pointer to width of the input maps (dimension 1)
* @param   mapHeight   Output pointer to height of the input maps (dimension 0)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_MapSize(kS3dRangeFilter filter, kSize* mapWidth, kSize* mapHeight);

/**
* Sets minimum number of adjacent points along the width of the map constituting a valid run. Any runs of fewer points
* than specified are discared.
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @param   length      Minimum valid run length
* @return              Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_SetMinRunLength(kS3dRangeFilter filter, kSize length);

/**
* Gets minimum number of adjacent points along the width of the map constituting a valid run. 
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @return              Minimum valid run length
*/
kVsFx(kSize) kS3dRangeFilter_MinRunLength(kS3dRangeFilter filter);

/**
* Sets the maximum step between Z values of adjacent points along the width of the heightmap which 
* does not constitute a break in continuity. A difference of more than the specified threshold
* results in break in the run. The value is specified in units of the range values
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @param   threshold   Step threshold
* @return              Operation status.
*/
kVsFx(kStatus) kS3dRangeFilter_SetRunStepThreshold(kS3dRangeFilter filter, kSize threshold);

/**
* Gets maximum step between Z values of adjacent points along the width of the heightmap which
* does not constitute a break in continuity.
*
* @public              @memberof kS3dRangeFilter
* @param   filter      Filter object
* @return              Step threshold
*/
kVsFx(kSize) kS3dRangeFilter_RunStepThreshold(kS3dRangeFilter filter);

#include <kVision/S3d/kS3dRangeFilter.x.h>

#endif  /* kS3D_RANGE_FILTER_H */
