/**
* @file    kS3dProjectorAdjust.h
* @brief   Declares the kS3dProjectorAdjust class. 
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/


#ifndef KVISION_S3D_PROJECTOR_ADJUST_H
#define KVISION_S3D_PROJECTOR_ADJUST_H

#include <kVision/S3d/kS3dCommon.h>

/**
 * @class   kS3dProjectorAdjust
 * @extends kObject
 * @ingroup kVision-S3d
 * @brief   Class of kS3dProjectorAdjust
 */

typedef kObject kS3dProjectorAdjust;

#define kS3D_PROJECTOR_ADJUST_FILE_NAME         "ProjectorAdjust.kdat6"

/**
 * Constructs a kS3dProjectorAdjust object
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Destination of constructed kS3dProjectorAdjust handle
 * @param   alloc           Memory allocator (or kNULL for default)
 * @return                  Operation status
 */
kVsFx(kStatus) kS3dProjectorAdjust_Construct(kS3dProjectorAdjust* adjust, kAlloc alloc);

/**
 * Load the projector adjustment from local storage
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Destination of class object handle
 * @param   path            File location to read projector adjustment from
 * @param   alloc           Memory allocator (or kNULL for default)
 * @return                  Operation status 
 */
kVsFx(kStatus) kS3dProjectorAdjust_Load(kS3dProjectorAdjust* adjust, const kChar* path, kAlloc alloc);

/**
 * Load the projector adjustment from a stream
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Destination of class object handle
 * @param   stream          Input stream handle to read from
 * @param   alloc           Memory allocator (or kNULL for default)
 * @return                  Operation status 
 */
kVsFx(kStatus) kS3dProjectorAdjust_LoadStream(kS3dProjectorAdjust* adjust, kStream stream, kAlloc alloc);

/**
 * Save the projector adjustment to local storage
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @param   path            File location to save projector adjustment to
 * @return                  Operation status 
 */
kVsFx(kStatus) kS3dProjectorAdjust_Save(kS3dProjectorAdjust adjust, const kChar* path);

/**
 * Save the projector adjustment to stream
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @param   stream          Output stream handle to save projector adjustment to
 * @return                  Operation status 
 */
kVsFx(kStatus) kS3dProjectorAdjust_SaveStream(kS3dProjectorAdjust adjust, kStream stream);

/**
 * Sets the sensor serial number
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @param   id              Sensor serial number
 * @return                  Operation status
 */
kVsFx(kStatus) kS3dProjectorAdjust_SetSensorId(kS3dProjectorAdjust adjust, k32u id);

/**
 * Gets the sensor serial number
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @return                  Sensor serial number
 */
kVsFx(k32u) kS3dProjectorAdjust_SensorId(kS3dProjectorAdjust adjust);

/**
 * Sets adjustment timestamp 
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @param   timestamp       Timestamp
 * @return                  Operation status
 */
kVsFx(kStatus) kS3dProjectorAdjust_SetTimestamp(kS3dProjectorAdjust adjust, const kChar* timestamp);

/**
 * Gets adjustment timestamp 
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @return                  Adjustment timestamp
 */
kVsFx(kChar*) kS3dProjectorAdjust_Timestamp(kS3dProjectorAdjust adjust);

/**
 * Sets the projector uniformity column correction
 * 
 * The input is expected in the form of 1D array for projector uniformity column correction scales in range (0,1)
 * 
 * During the assignment, the adjustment class makes a copy of input array. The caller is responsible for destroying the input.
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @param   xCorrection     Collection of projector uniformity correction per column in kArray1<k64f>
 * @return                  Operation status
 */
kVsFx(kStatus) kS3dProjectorAdjust_SetXCorrection(kS3dProjectorAdjust adjust, kArray1 xCorrection);

/**
 * Gets the projector uniformity column correction 
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @return                  kArray1<k64f> of projector uniformity correction per column
 */
kVsFx(kArray1) kS3dProjectorAdjust_XCorrection(kS3dProjectorAdjust adjust);

/**
 * Sets the projector uniformity row correction
 * 
 * The input is expected in the form of 1D array for projector uniformity row correction scales in range (0,1)
 * 
 * During the assignment, the adjustment class makes a copy of input array. The caller is responsible for destroying the input.
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @param   yCorrection     Collection of projector uniformity correction per row in kArray1<k64f>
 * @return                  Operation status
 */
kVsFx(kStatus) kS3dProjectorAdjust_SetYCorrection(kS3dProjectorAdjust adjust, kArray1 yCorrection);

/**
 * Gets the projector uniformity row correction 
 * 
 * @public                  @memberof kS3dProjectorAdjust
 * @param   adjust          Class object
 * @return                  kArray1<k64f> of projector uniformity correction per row
 */
kVsFx(kArray1) kS3dProjectorAdjust_YCorrection(kS3dProjectorAdjust adjust);

#include <kVision/S3d/kS3dProjectorAdjust.x.h>

#endif