/**
* @file    kS3dPhaseSegment.h
* @brief   Declares the kS3dPhaseSegment class. 
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef kS3D_PHASE_SEGMENT_H
#define kS3D_PHASE_SEGMENT_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dPhaseSegment
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Performs a segmentation operation. The continuous phase area will be grouped.
*/
typedef kObject kS3dPhaseSegment;

/**
* Constructs a kS3dPhaseSegment object
*
* @public              @memberof kS3dPhaseSegment
* @param   segment     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSegment_Construct(kS3dPhaseSegment* segment, kAlloc allocator);

/**
*  Runs the segmentation operation.  
*
* @public              @memberof kS3dPhaseSegment
* @param   segment     kS3dPhaseSegment object.
* @param   map         Segment map will be filled. kArray2 <k32s>. 
*                      The both width and height of the array should be larger than 8 pixel.   
* @param   data        Phase map of kArray2 <k32s>. 
*                      The both width and height of the array should be identical to the segment map array.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSegment_Run(kS3dPhaseSegment segment, kArray2 map, kArray2 data);

/**
* Set the threshold. The phase points will be grouped together when the phase difference 
* between the adjacent points is not greater than this threshold value.
*
* @public              @memberof kS3dPhaseSegment
* @param   segment     kS3dPhaseSegment object.
* @param   threshold   Threshold
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSegment_SetThreshold(kS3dPhaseSegment segment, kSize threshold);

/**
* Get the threshold. The phase points will be grouped together when the phase difference
* between the adjacent points is not greater than this threshold value.
*
* @public              @memberof kS3dPhaseSegment
* @param   segment     kS3dPhaseSegment object.
* @return              Threshold
*/
kVsFx(kSize) kS3dPhaseSegment_Threshold(kS3dPhaseSegment segment);

/**
* Set the repeat number of filtering
*
* @public              @memberof kS3dPhaseSegment
* @param   segment     kS3dPhaseSegment object.
* @param   repeat      Repeat number
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSegment_SetRepeat(kS3dPhaseSegment segment, kSize repeat);

/**
* Get the repeat number of filtering
*
* @public              @memberof kS3dPhaseSegment
* @param   segment     kS3dPhaseSegment object.
* @return              Repeat number
*/
kVsFx(kSize) kS3dPhaseSegment_Repeat(kS3dPhaseSegment segment);

kVsFx(kStatus) kS3dPhaseSegment_SetViewIndex(kS3dPhaseSegment segment, kSize index);

#include <kVision/S3d/kS3dPhaseSegment.x.h>

#endif
