/**
* @file    kS3dPhaseMedian.h
* @brief   Declares the kS3dPhaseMedian class. 
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef kS3D_GRAYCODE_MEDIAN_H
#define kS3D_GRAYCODE_MEDIAN_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

#define kS3D_GRAYCODEMEDIAN_MAX_WIDTH 32

/**
* @class       kS3dPhaseMedian
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs median filtering operation of phase map with 5 x 5 pixel.
               Only the phase order but not the phase value will be correct. 
*/
typedef kObject kS3dPhaseMedian;

/**
* Constructs a kS3dPhaseMedian object
*
* @public              @memberof kS3dPhaseMedian
* @param   median      Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseMedian_Construct(kS3dPhaseMedian* median, kAlloc allocator);

/**
* Runs the filtering operation. 
*
* @public              @memberof kS3dPhaseMedian
* @param   median      kS3dPhaseMedian object.
* @param   data        Phase map of kArray2 <k32s>. The both width and height of the array should be larger than 8 pixel.
* @param   temp        Temporary data buffer for the operation.  kArray2 <k32s> 
*                      The both width and height of the array should be identical to the phase map array.  
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseMedian_Run(kS3dPhaseMedian median, kArray2 data, kArray2 temp);

/**
* Set the repeat number of filtering
*
* @public              @memberof kS3dPhaseMedian
* @param   median      kS3dPhaseMedian object.
* @param   repeat      Repeat number
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseMedian_SetRepeat(kS3dPhaseMedian median, kSize repeat);

/**
* Get the repeat number of filtering
*
* @public              @memberof kS3dPhaseMedian
* @param   median      kS3dPhaseMedian object.
* @return              Repeat number.
*/
kVsFx(kSize) kS3dPhaseMedian_Repeat(kS3dPhaseMedian median);

/**
* Set the digital count of one phase period.
*
* @public              @memberof kS3dPhaseMedian
* @param   median      kS3dPhaseMedian object.
* @param   periodt     Digital count of one phase period.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseMedian_SetPeriod(kS3dPhaseMedian median, kSize period);

/**
* Get the digital count of one phase period.
*
* @public              @memberof kS3dPhaseMedian
* @param   median      kS3dPhaseMedian object.
* @return              Digital count of one phase period.
*/
kVsFx(kSize) kS3dPhaseMedian_Period(kS3dPhaseMedian median);

/**
* Set the Half width of the filtering window.
*
* @public              @memberof kS3dPhaseMedian
* @param   median      kS3dPhaseMedian object.
* @param   width       Half width of the filtering window.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseMedian_SetWindow(kS3dPhaseMedian median, kSize width);

#include <kVision/S3d/kS3dPhaseMedian.x.h>

#endif
