/**
* @file    kS3dPhaseCorrection.h
* @brief   Declares the kS3dPhaseCorrection class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kS3D_PHASE_CORRECTION_H
#define KVISION_kS3D_PHASE_CORRECTION_H

#include <kApi/Data/kMath.h>
#include <kVision/S3d/kS3dPhaseSegment.h>
#include <kVision/S3d/kS3dPhaseMedian.h>
#include <kVision/S3d/kS3dPhaseExpand.h>
#include <kVision/S3d/kS3dPhaseOutlier.h>

typedef struct kS3dStripePhasePixel
{
    k16s phase;
    k8u code;
    k8u intensity;
} kS3dStripePhasePixel;

/**
* @class       kS3dPhaseCorrection
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Trying to correct the phase order by comparing neighboring points under continuity assumption. 
*              The operation should only be used for some specific applications.
*/
typedef kObject kS3dPhaseCorrection;

/**
* Constructs a kS3dPhaseCorrection object
*
* @public               @memberof kS3dPhaseCorrection
* @param   phaseCorr    Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_Construct(kS3dPhaseCorrection* phaseCorr, kAlloc allocator);

/**
* Set the digital count of one phase period.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   period      Digital count of one phase period.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetPhasePeriod(kS3dPhaseCorrection phaseCorr, kSize period);

/**
* Get the digital count of one phase period.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @return              Digital count of one phase period.
*/
kVsFx(kSize) kS3dPhaseCorrection_PhasePeriod(kS3dPhaseCorrection phaseCorr);

/**
* Set a boolean value to describe the phase curve. If the value is true, is the phase angle decreasing.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   isInvers    Is the phase angle decreasing?.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetInversPhase(kS3dPhaseCorrection phaseCorr, kBool isInvers);

/**
* Get the boolean value to determine the phase curve direction.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @return              Is the phase angle decreasing?
*/
kVsFx(kBool) kS3dPhaseCorrection_InversPhase(kS3dPhaseCorrection phaseCorr);

/**
* Set a operating mode to decide which combination of the filtering should be used.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   mode        Currently: kS3D_PHASE_CORRECTION_TYPE_MULTI_REFLECTION is for correcting the phase order from multi reflection e.g.Combustion chamber.
*                                 kS3D_PHASE_CORRECTION_TYPE_TRANSLUCENCY is for correcting the phase order on transparent surface, e.g. skin surface measurement.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetFilterMode(kS3dPhaseCorrection phaseCorr, kS3dPhaseCorrectionType mode);

/**
* Get the operating mode.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @return              The operating mode.
*/
kVsFx(kSize) kS3dPhaseCorrection_FilterMode(kS3dPhaseCorrection phaseCorr);

/**
* Set the View index. For storing intermediate results  if necessary.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   index       View index: 0 or 1.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetViewIndex(kS3dPhaseCorrection phaseCorr, kSize index);


/**
* Start the phase correction according to 2D segmentation methods. 
*
* @public                 @memberof kS3dPhaseCorrection
* @param   phaseCorr      kS3dPhaseCorrection object.
* @param   stripePhaseMap Input data. kArray2<kS3dStripePhasePixel>.
*                         The both width and height of the array should be larger than 8 pixel.     
* @return                 Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_Apply(kS3dPhaseCorrection phaseCorr, kArray2 stripePhaseMap);

/**
* Start the phase correction according to 2D segmentation methods.
*
* @public                 @memberof kS3dPhaseCorrection
* @param   phaseCorr      kS3dPhaseCorrection object.
* @param   stripePhaseMap Input data. kArray2<kPhasePixel2>.
*                         The both width and height of the array should be larger than 8 pixel.
* @return                 Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_ApplyPhasePixel2(kS3dPhaseCorrection phaseCorr, kSize phasePeriod, kArray2 phaseMap);

/**
* Set a width value. Only the Fragments, which maximum width in a line less than the value, will be treated. 
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   width       A width value to limit the correction range.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetValidWidthLimit(kS3dPhaseCorrection phaseCorr, kSize width);

/**
* Set a number value. Fragments are sorted by area size downwards. The areas up to this number are not corrected.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   number      A number value to limit the correction range.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetValidAreaNumber(kS3dPhaseCorrection phaseCorr, kSize number);

/**
* Set a step value. Only the Fragments will be corrected, which have a phase jump to its neighbor greater than this value.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   step        Jump step. Default: 1
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetMinimumJumpingStep(kS3dPhaseCorrection phaseCorr, kSize step);

/**
* Set a window. Only the points within this defined area will be corrected.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   window      Correction window.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetCorrectionWindow(kS3dPhaseCorrection phaseCorr, kRect32s window);

/**
* Set a area size. Fragments whose area is smaller than the specified size, are not considered as a group during the correction.
*
* @public              @memberof kS3dPhaseCorrection
* @param   phaseCorr   kS3dPhaseCorrection object.
* @param   area        minimum valid area size.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetOutlierFragmentLimit(kS3dPhaseCorrection phaseCorr, kSize area);

/**
* Set a boolean value to decide, whether the main area should be preferred.
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @param   preferMainArea    The main area should be preferred?
* @return                    Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetPreferMainArea(kS3dPhaseCorrection phaseCorr, kSize preferMainArea);

/**
* Set the Half width of window for the anti reflection filtering.
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @param   width             Half width of the filtering window.
* @return                    Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetReflectionCorrWidth(kS3dPhaseCorrection phaseCorr, kSize width);

/**
* Get the Half width of window for the anti reflection filtering.
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @return                    Half width of the filtering window.
*/
kVsFx(kSize) kS3dPhaseCorrection_ReflectionCorrWidth(kS3dPhaseCorrection phaseCorr);

/**
* Set the repeat count for the anti reflection filtering.
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @param   repeat            Repeat count.
* @return                    Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetReflectionCorrRepeat(kS3dPhaseCorrection phaseCorr, kSize repeat);

/**
* Get the repeat count for the anti reflection filtering.
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @return                    Repeat count.
*/
kVsFx(kSize) kS3dPhaseCorrection_ReflectionCorrRepeat(kS3dPhaseCorrection phaseCorr);

/**
* Set a relative threshold for checking the phase continuity between adjacent points.
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @param   factor            The relative threshold refers to the pattern period.
* @return                    Operation status.
*/
kVsFx(kStatus) kS3dPhaseCorrection_SetSegmentLimitFactor(kS3dPhaseCorrection phaseCorr, k64f factor);

/**
* Get the relative threshold for checking the phase continuity between adjacent points
*
* @public                    @memberof kS3dPhaseCorrection
* @param   phaseCorr         kS3dPhaseCorrection object.
* @return                    The relative threshold.
*/
kVsFx(k64f) kS3dPhaseCorrection_SegmentLimitFactor(kS3dPhaseCorrection phaseCorr);

//kVsFx(kStatus) kS3dPhaseCorrection_CopyToViewMap(kS3dPhaseCorrection phaseCorr, kArray2 viewMap, k32s phasePeriod, k32s grayWidth, k32s projectorWidth, k64f offset, kBool inversPhase, k32s camIndex);

#include <kVision/S3d/kS3dPhaseCorrection.x.h>

#endif
