/** 
 * @file    kS3dPhaseActionInfo.h
 * @brief   Declares the kS3dPhaseActionInfo class. 
 * 
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef kS3D_PHASE_ACTION_INFO_H
#define kS3D_PHASE_ACTION_INFO_H

#include <kVision/S3d/kS3dCommon.h>
#include <kApi/Data/kXml.h>
#include <kFireSync/kNodeDef.h> // kCameraPhaseDecoderFormat

/**
 * @class  kS3dPhaseActionInfo
 * @ingroup kVision
 * @brief   Settings for kS3dPhaseAction.
 */
typedef kObject kS3dPhaseActionInfo; 

/**
* Constructs a kS3dPhaseActionInfo object
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Destination for the constructed object handle.
* @param   allocator             Memory allocator (or kNULL for default).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Construct(kS3dPhaseActionInfo* info, kAlloc allocator);

/**
* Read items from xml file and copy values to a new constructed kS3dPhaseActionInfo object
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Destination for the constructed object handle.
* @param   xml                   xml file object which contains the value of the parameters
* @param   allocator             Memory allocator (or kNULL for default).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Load(kS3dPhaseActionInfo* info, kXml xml, kAlloc allocator);

/**
* Copy the parameters from kS3dPhaseActionInfo object to a new constructed xml file object to store it 
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   xml                   Destination for the constructed xml file object handle with values from info
* @param   allocator             Memory allocator (or kNULL for default).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Store(kS3dPhaseActionInfo info, kXml* xml, kAlloc allocator);

/**
* Write the parameters from kS3dPhaseActionInfo object to an xml object
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   xml                   Xml file object to write value to
* @param   item                  Item of xml file to write value to
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Write(kS3dPhaseActionInfo info, kXml xml, kXmlItem item);

/**
* Read the parameters to kS3dPhaseActionInfo object from an xml object
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   xml                   Xml file object to read value from
* @param   item                  Item of xml file to read value from
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Read(kS3dPhaseActionInfo info, kXml xml, kXmlItem item);

/**
* Set all parameters of kS3dPhaseActionInfo object to default values
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Clear(kS3dPhaseActionInfo info); 

/**
* Validate the kS3dPhaseActionInfo object passed in
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_Validate(kS3dPhaseActionInfo info); 

/**
* Sets the intensity of projector pattern. Multiplier applied to the generated pattern intensity (0..1)
* (default being 1)
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   intensity             Intensity of projector pattern (0..1)
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPatternIntensity(kS3dPhaseActionInfo info, k64f phaseIntensity);

/**
* Gets the intensity of projector pattern.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Pattern intensity
*/
kVsFx(k64f) kS3dPhaseActionInfo_PatternIntensity(kS3dPhaseActionInfo info);

/**
* Sets the parameters of uniformity of projector stripe pattern. Intensity scaling factor to compensate for projection lateral non-uniformity (0..1). 
* The maximum intensity of the center of the generated pattern is attenuated by this factor relative to the boundaries.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   patternUniformity     Uniformity of projector stripe pattern
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPatternUniformity(kS3dPhaseActionInfo info, k64f patternUniformity);

/**
* Gets the parameters of uniformity of projector stripe pattern
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   patternUniformity     Uniformity of projector stripe pattern
* @return                        Operation status.
*/
kVsFx(k64f) kS3dPhaseActionInfo_PatternUniformity(kS3dPhaseActionInfo info);

/**
* Sets the parameters of phase period of phase decoder. The phase values expected by the kS3dPhaseViewProcessor_Run function must be scaled such that
* 0 to 2Pi phase offset is mapped to 0 to period
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   phasePeriod           Phase period of phase decoder
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPhasePeriod(kS3dPhaseActionInfo info, kSize phasePeriod);

/**
* Gets the parameters of phase period of phase decoder
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Operation status.
*/
kVsFx(kSize) kS3dPhaseActionInfo_PhasePeriod(kS3dPhaseActionInfo info);

/**
* Sets parameter of contrast threshold for stripe decoding. The difference between stripe image pixel intensity and reference pixel intensity
* must be higher or equal to this value in order for the decoding at that pixel to be valid.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   stripeThreshold       Contrast threshold for stripe decoding
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetStripeThreshold(kS3dPhaseActionInfo info, kSize stripeThreshold);

/**
* Gets parameter of contrast threshold for stripe decoding.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Operation status.
*/
kVsFx(kSize) kS3dPhaseActionInfo_StripeThreshold(kS3dPhaseActionInfo info);

/**
* Sets the parameter of contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   phaseThreshold        Contrast threshold for phase decoding
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPhaseThreshold(kS3dPhaseActionInfo info, kSize phaseThreshold);

/**
* Gets the parameter of contrast threshold for phase decoding.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   phaseThreshold        Contrast threshold for phase decoding
* @return                        Operation status.
*/
kVsFx(kSize) kS3dPhaseActionInfo_PhaseThreshold(kS3dPhaseActionInfo info);

/**
* Enables image input. 
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   enable                Flag specifying whether image input should be enabled
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_EnableImageInput(kS3dPhaseActionInfo info, kBool enable);

/**
* Gets the value of the flag specifying whether raw image input is enabled.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Flag specifying whether image input is enabled
*/
kVsFx(kBool) kS3dPhaseActionInfo_ImageInputEnabled(kS3dPhaseActionInfo info);

/**
* Sets encode type (kS3dEncodeType)
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   encodeType            Encode type (kS3dEncodeType)
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetEncodeType(kS3dPhaseActionInfo info, kS3dEncodeType encodeType);

/**
* Gets encode type (kS3dEncodeType)
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Encode type (kS3dEncodeType)
*/
kVsFx(kS3dEncodeType) kS3dPhaseActionInfo_EncodeType(kS3dPhaseActionInfo info);

/**
* Sets phase decoder format (kCameraPhaseDecoderFormat)
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   format                Format (kCameraPhaseDecoderFormat)
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPhaseDecoderFormat(kS3dPhaseActionInfo info, kCameraPhaseDecoderFormat format);

/**
* Gets phase decoder format (kCameraPhaseDecoderFormat)
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Format (kCameraPhaseDecoderFormat)
*/
kVsFx(kCameraPhaseDecoderFormat) kS3dPhaseActionInfo_PhaseDecoderFormat(kS3dPhaseActionInfo info);

/**
* Sets phase prediction for compression only (DecoderFormat == kCAMERA_PHASE_DECODER_FORMAT_DELTA_I8_P24 == 8).
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   prediction            Phase prediction 
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPhasePrediction(kS3dPhaseActionInfo info, kSize prediction);

/**
* Gets phase prediction.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Phase prediction.
*/
kVsFx(kSize) kS3dPhaseActionInfo_PhasePrediction(kS3dPhaseActionInfo info);

/**
* Sets the number of subframes to split the phase for compression (DecoderFormat == kCAMERA_PHASE_DECODER_FORMAT_DELTA_I8_P24 == 8).
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   subframeCount         Subframe count
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetSubframeCount(kS3dPhaseActionInfo info, kSize subframeCount);

/**
* Gets phase prediction.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Subframe count
*/
kVsFx(kSize) kS3dPhaseActionInfo_SubframeCount(kS3dPhaseActionInfo info);

/**
* Sets the lengths of the Gray code and phase subsequences. At most one reference image is supported.
* If no stripe sequence is used (phase period occupies full projection)
* the number of reference images must also be zero. Total number of expected images is the sum of the individual
* sub-sequence counts
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   referenceImageCount   Number of reference images in the image sequence (<= 1)
* @param   stripeImageCount      Number of stripe images in the image sequence (<= 7)
* @param   phaseImageCount       Number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetStripeSequenceInfo(kS3dPhaseActionInfo info, kSize referenceImageCount, kSize stripeImageCount, kSize phaseImageCount);

/**
* Gets the lengths of the Gray code and phase subsequences.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   referenceImageCount   Output pointer to the number of reference images in the image sequence
* @param   stripeImageCount      Output pointer to the number of stripe images in the image sequence
* @param   phaseImageCount       Output pointer to the number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_StripeSequenceInfo(kS3dPhaseActionInfo info, kSize* referenceImageCount, kSize* stripeImageCount, kSize* phaseImageCount);

/**
* Sets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   coefficients          Array of coefficients describing embedded periods (k64f).
* @param   stepCounts            Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   coefficientCount      Size of the "coefficients" and "stepCounts" arrays.
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetEmbeddedPhaseSequenceInfo(kS3dPhaseActionInfo info, const k64f* coefficients, const kSize* stepCounts, kSize coefficientCount);

/**
* Gets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @param   outCoefficients       Array of coefficients describing embedded periods (k64f).
* @param   outStepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   outCoefficientCount   Number of values in the "coefficients" and "stepCounts" arrays returned (outCoefficientCount <= kS3D_EMBEDDED_PHASE_DECODER_PERIOD_COUNT).
* @param   capacity              Size of "coefficients" and "stepCounts" arrays passed in (capacity should be >= coefficientCount).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_EmbeddedPhaseSequenceInfo(kS3dPhaseActionInfo info, k64f* outCoefficients, kSize* outStepCounts, kSize capacity);

/**
* Get the number of frequencies in EPS.
* 
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Number of coefficients.
*/

kVsFx(kSize) kS3dPhaseActionInfo_EmbeddedPhaseCoefficientCount(kS3dPhaseActionInfo info);

/**
* Get the number of images in EPS.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  Info object
* @return                        Number of images.
*/

kVsFx(kSize) kS3dPhaseActionInfo_EmbeddedPhaseImageCount(kS3dPhaseActionInfo info);

/**
* Stripe alg phase period length.
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @param   period                Phase period in pixels
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseActionInfo_SetPeriodLength(kS3dPhaseActionInfo info, k64f period);

/**
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                     
*/
kVsFx(k64f) kS3dPhaseActionInfo_PeriodLength(kS3dPhaseActionInfo info);

#include <kVision/S3d/kS3dPhaseActionInfo.x.h>

#endif
