/**
* @file    kS3dOmcFile.h
* @brief   Declares the kS3dOmcFile class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_S3D_OMC_H
#define KVISION_S3D_OMC_H

#include <kApi/Data/kImage.h>
#include <kVision/S3d/kS3dVolumeCheckDef.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dVolumeCheckSubPixLut.h>
#include <kFireSync/Data/kPlot.h>
#include <kApi/Utils/kUtils.h>
#include <kFireSync/Data/kGraphic.h>

/**
* @class       kS3dOmcFile
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Provides some utility functions to generate, manage and edit of a height map.
*/
typedef kObject kS3dOmcFile;

/**
* Constructs a kS3dOmcFile object
*
* @public              @memberof kS3dOmcFile
* @param   omc     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_Construct(kS3dOmcFile* omc, kAlloc allocator);

/**
* Constructs a kS3dOmcFile object through loading a OMC file
*
* @public              @memberof kS3dOmcFile
* @param   omc         Destination for the constructed object handle.
* @param   fileName    File name.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_Load(kS3dOmcFile* omc, const kChar *fileName);

/**
* Constructs a kS3dOmcFile object through importing data from a kArray2 object.
*
* @public                @memberof kS3dOmcFile
* @param   omc           Destination for the constructed object handle.
* @param   inputArray    Source data Array.
* @param   latScale      Pixel distance in mm. 
* @param   zResolution   Digital resolution of depth map in mm.
* @param   type          Type of data to be imported. 
* @return                Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_Array2ToOmc(kS3dOmcFile *omc, kArray2 inputArray, k64f latScale, k64f zResolution, kS3dArray2DataType type);


/**
* Constructs a kS3dOmcFile object through importing data from a kArray2 object with a given range.
*
* @public                @memberof kS3dOmcFile
* @param   omc           Destination for the constructed object handle.
* @param   inputArray    Source data Array.
* @param   latScale      Pixel distance in mm.
* @param   zResolution   Digital resolution of depth map in mm.
* @param   type          Type of data to be imported.
* @return                Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_Array2ToOmcRoi(kS3dOmcFile *omc, kArray2 inputArray, k64f latScale, k64f zResolution, kRect3d64f roi, kS3dArray2DataType type);

/**
* Get height map array
*
* @public              @memberof kS3dOmcFile
* @param   omc         kS3dOmcFile object.
* @return              Height map of kAarray2 < k32s >
*/
kVsFx(kArray2) kS3dOmcFile_Data(kS3dOmcFile omc);

/**
* Get pattern contrast array
*
* @public              @memberof kS3dOmcFile
* @param   omc         kS3dOmcFile object.
* @return              Pattern contrast map of kAarray2 < k8u >
*/
kVsFx(kArray2) kS3dOmcFile_Pow(kS3dOmcFile omc);

/**
* Stores the height map to a file.
*
* @public              @memberof kS3dOmcFile.
* @param   omc         kS3dOmcFile object.
* @param   fileName    File name.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_Save(kS3dOmcFile omc, const kChar *fileName);

/**
* Stores a kArray2 object to a file.
*
* @public               @memberof kS3dOmcFile.
* @param   array        Input height map or point cloud.
* @param   fileName     File name.
* @param   latScale     Pixel pitch of the to be saved height map.
* @param   zResolution  Digital resolutuion of the to be saved height map.
* @param   type         Array type and the to be saved component.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_SaveArray2ToOmc(kArray2 array, const kChar* fileName, k64f latScale, k64f zResolution, kS3dArray2DataType type);

/**
* Get the digital resolutuion of the height map of kArray2 < k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @return               digital resolutuion.
*/
kVsFx(k64f) kS3dOmcFile_Zresolution(kS3dOmcFile omc);

/**
* Set the digital resolutuion of the height map of kArray2 < k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @param   zResolution  Digital resolutuion.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_SetZresolution(kS3dOmcFile omc, k64f zResolution);

/**
* Get the pixel pitch of the of the height map of kArray2 <k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @return               Pixel pitch.
*/
kVsFx(k64f) kS3dOmcFile_LateralScale(kS3dOmcFile omc);

/**
* Set the pixel pitch of the of the height map of kArray2 <k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @param   latScale     Pixel pitch.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_SetLateralScale(kS3dOmcFile omc, k64f latScale);

/**
* Get the image width  of the height map of kArray2 <k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @return               image width.
*/
kVsFx(k32s) kS3dOmcFile_DepthSizeX(kS3dOmcFile omc);

/**
* Set the image width  of the height map of kArray2 <k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @param   depthSizeX   image width.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_SetDepthSizeX(kS3dOmcFile omc, k32s depthSizeX);

/**
* Get the image height  of the height map of kArray2 <k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @return               Image height.
*/
kVsFx(k32s) kS3dOmcFile_DepthSizeY(kS3dOmcFile omc);

/**
* Set the image height  of the height map of kArray2 <k32s >.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @param   depthSizeY   Image height.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_SetDepthSizeY(kS3dOmcFile omc, k32s depthSizeY);

/**
* Updates working buffers of the height map with current configuration.
*
* @public               @memberof kS3dOmcFile.
* @param   omc          kS3dOmcFile object.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_Setup(kS3dOmcFile omc);

/**
* Stores a kImage object to a file.
*
* @public               @memberof kS3dOmcFile.
* @param   image        Input image.
* @param   fileName     File name.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dOmcFile_SaveImageToOmc(kImage image, const kChar* fileName);


kVsFx(kStatus) kS3dOmcFile_UpdateViewImage(kS3dOmcFile omcFile, kImage* viewImage, kArray1 palette, k64f zmin, k64f zmax, kBool flip);


kVsFx(kStatus) kS3dOmcFile_PlotViewImage(kS3dOmcFile omcFile, kPlot* outPlot, kArray1 palette, k64f zmin, k64f zmax, kBool drawScale, const kChar* zUnit, kBool flip, kArrayList contour);

#include <kVision/S3d/kS3dOmcFile.x.h>

#endif /* #ifndef KVISION_S3D_OMC_H */
