/**
* @file    kS3dDecompressor.h
* @brief   Declares the kS3dDecompressor type.
*
* @internal
* Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef K_VISION_S3D_DECOMPRESSOR_H
#define K_VISION_S3D_DECOMPRESSOR_H

#include <kVision/S3d/kS3dCommon.h>

/**
* @class   kS3dDecompressor
* @extends kObject
* @ingroup kVision-S3d
* @brief   Unpacks and decodes compressed phase and intensity data. 
* 
* This class provides an optimized implementation of phase/intensity decompression -- unpacking and decoding
* a list of compressed subframes to produce uncompressed phase and intensity arrays. The implementation may 
* make use of multiple threads and/or GPU resources. 
* 
* The kS3dDecompressor_Execute method can be used to perform all decompression steps in a single call. 
* Alternatively, the BeginExecute, AddSubframe, and EndExecute methods can be used to add one 
* subframe at a time. The latter may be useful if subframes are received one at a time (e.g., over the network),
* enabling processing to begin before all subframes have been received. 
* 
* This class is intended to be reused to decompress multiple frames (where each frame consists of one or more 
* subframes). In doing to, the aim would be to improve execution efficiency by reusing intermediate data 
* processing structures between invocations. In order to avoid increased execution times due to memory 
* allocation in the first few frames, the Reserve method can optionally be used to preallocate internal buffers. 
* 
@code {.c}

kStatus PrepareDecompressor(kS3dDecompressor* decompressor, kSize frameSize, kSize subframeCount, kAlloc* allocator)
{
    kCheck(kS3dDecompressor_Construct(decompressor, allocator));

    // Enable phase/intensity for decompression as each subframe can have both
    kCheck(kS3dDecompressor_EnablePhase(decompressor, kTRUE));
    kCheck(kS3dDecompressor_EnableIntensity(decompressor, kFALSE));

    //Memory reservation is optional, but can reduce variability in execution times. 
    //The frame size is just the total number of elements in the uncompressed phase image, while 
    //the subframe count is the expected number of compressed subframes.
    kCheck(kS3dDecompressor_Reserve(decompressor, kTRUE, frameSize, subframeCount));

    return kOK;
}

kStatus Decompress(kS3dDecompressor decompressor, kArrayList subframeList, kArray2 phaseOutput, kArray2 intensityOutput)
{
    //The subframe list is a list of received kCompressedPhase objects, where each object represents 
    //one subframe within the same frame. Results written to the phaseOutput as kPhasePixel2 format.
    //The Execute method performs all work in a single call. Alternatively, the BeginExecute, AddSubframe, 
    //and EndExecute methods can be used to add subframes one at a time, as they are received. The implementation 
    //of the Execute method demonstrates how to use those methods.
    kCheck(kS3dDecompressor_Execute(decompressor, subframeList, phaseOutput));

    return kOK;
}
 
@endcode
* 
*/
typedef kObject kS3dDecompressor;

/**
* Constructs a kS3dDecompressor object.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Destination for the constructed object handle.
* @param   allocator        Memory allocator (or kNULL for default).
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dDecompressor_Construct(kS3dDecompressor* decompressor, kAlloc allocator);

/**
* Reserves memory capacity for internal buffers. 
* 
* kS3dDecompressor grows its internal memory buffers as needed during processing. After processing 
* a few frames, it will usually reach a steady-state where no more allocations are required. This 
* is beneficial for reducing future execution times. 
* 
* To speed up the process of reaching this steady state, the Reserve method can optionally be used to make 
* better-informed initial guesses about required buffer sizes. 
*
* @public                     @memberof kS3dDecompressor
* @param   decompressor       Decompressor object. 
* @param   intensityExpected  Will intensity streams be decompressed (in addition to phase)? 
* @param   frameSize          Expected maximum uncompressed size of frames (pixel count). 
* @param   subframeCount      Expected subframe count.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dDecompressor_Reserve(kS3dDecompressor decompressor, kBool intensityExpected, kSize frameSize, kSize subframeCount);

/**
* Performs decompression. 
* 
* This method is equivalent to calling BeginExecute, followed by one or more calls to AddSubframe, 
* followed by a final call to EndExecute. 
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object. 
* @param   subframeList     List of all compressed subframes associated with a single frame, in subframe-index order (kArrayList<kCompressedPhase>).
* @param   output           Output array to receive phase/intensity data. 
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dDecompressor_Execute(kS3dDecompressor decompressor, kArrayList subframeList, kArray2 output);

/**
* Begins decompression. 
* 
* After beginning decompression, call AddSubframe to add data, followed by EndExecute.
* 
* Ownership (lifetime management responsibility) of the phaseOutput and intensityOutput arguments is not transferred.  
* However, the decompressor may write asynchronously to the phaseOutput and intensityOutput arrays at any time between calling 
* BeginExecute and EndExecute. Accordingly, please ensure that EndExecute is called before attempting to access the phaseOutput
* or intensityOutput arrays.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object. 
* @param   output           Output array to receive phase/intensity data. 
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dDecompressor_BeginExecute(kS3dDecompressor decompressor, kArray2 phaseOutput);

/**
* Adds a subframe to be processed. 
*
* Subframes should be added in subframe-index order until all of the subframes associated with a particular frame 
* have been added.
* 
* Decompression work may continue in background threads after calling this method. The EndExecute method
* must be called after all subframes have been added in order to complete processing activities.
* 
* Ownership (lifetime management responsibility) of the subframe argument is not transferred.  
* However, the decompressor may read asynchronously from the subframe any time between calling 
* AddSubframe and calling EndExecute. Accordingly, please ensure that EndExecute is called before attempting to 
* modify or destroy the subframe object.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object. 
* @param   subframe         Compressed subframe. 
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dDecompressor_AddSubframe(kS3dDecompressor decompressor, kCompressedPhase subframe);

/**
* Ends decompression. 
* 
* This method will block until processing is complete. When this method returns, the decompressed
* results will be available in the phaseOutput and intensityOutput arguments that were previously 
* passed to BeginExecute.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object. 
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dDecompressor_EndExecute(kS3dDecompressor decompressor);

/**
* Sets the intensity flag determining if intensity data should be decoded.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object.
* @param   intensity        Intensity flag.
* @return                   Operation status.
*/
kInlineFx(kStatus) kS3dDecompressor_EnableIntensity(kS3dDecompressor decompressor, kBool intensity);

/**
* Gets the intensity flag determining if intensity data should be decoded.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object.
* @return                   Intensity flag.
*/
kInlineFx(kBool) kS3dDecompressor_IsIntensity(kS3dDecompressor decompressor);

/**
* Sets the phase flag determining if phase data should be decoded.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object.
* @param   intensity        Phase flag.
* @return                   Operation status.
*/
kInlineFx(kStatus) kS3dDecompressor_EnablePhase(kS3dDecompressor decompressor, kBool phase);

/**
* Gets the phase flag determining if phase data should be decoded.
*
* @public                   @memberof kS3dDecompressor
* @param   decompressor     Decompressor object.
* @return                   Phase flag.
*/
kInlineFx(kBool) kS3dDecompressor_IsPhase(kS3dDecompressor decompressor);

#include <kVision/S3d/kS3dDecompressor.x.h>

#endif
