#ifndef K_VISION_L3D_LUTWRITER_H
#define K_VISION_L3D_LUTWRITER_H

#include <kVision/L3d/kL3dSensorCal.h>
#include <kVision/L3d/kL3dTransform2d.h>
#include <kFireSync/Client/Camera/kCameraInfo.h>
#include <kFireSync/Client/kCamera.h>
#include <kVision/L3d/kL3dLutWriterImpl.h>
#include <kVision/L3d/kL3dLutWriterLegacyImpl.h>

typedef kObject kL3dLutWriter;

typedef struct kL3dLutWriterRunParams 
{
    k64f xResolution;
    k64f zResolution;

    k32u intensityMultiplier;
} kL3dLutWriterRunParams;

/**
 * @brief Construct a Lut Writer object
 * 
 * @param writer    Handler to lut writer object
 * @param camera    Camera object to get image dimensions and subsampling intervals
 * @param cal       Calibration file to generate luts
 * @param useLegacy Optional: set this flag as kTRUE to revert to legacy LUT generation algorithm using sparse tables and omitting double extrapolation, Default: kTRUE
 * @param caps      Optional: if camera is kNULL kRangeLutCaps can be passed in to specify default camera parameters, Default: kNULL
 * @param allocator Optional: create object with a specific memory allocator (e.g. unit tests) Default: kNULL
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriter_Construct(kL3dLutWriter* writer, kCamera camera, kL3dSensorCal cal, kBool useLegacy = kTRUE, kRangeLutCaps* caps = kNULL, kAlloc allocator = kNULL);

/**
 * @brief Get the window subdivision used in the lut
 * 
 * @param writer    Handler to lut writer object
 * @return k32u     Value of the window subdivision used in the lut
 */
kVsFx(k32u)    kL3dLutWriter_WindowSubdivision(kL3dLutWriter writer);

/**
 * @brief Given an FOV or active area and transform, determine the x and z resolution (mm/k16s)
 * 
 * @param writer        Handler to lut writer object
 * @param roi           Rect object that describes the FOV or active area desired
 * @param transform     Transform to compensate sensor orientations
 * @param xResolution   Pointer to output x resolution
 * @param zResolution   Pointer to output z resolution
 * @return kStatus      kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriter_CalculateResolutions(kL3dLutWriter writer, const kRect64f* roi, const kL3dTransform2d* transform, k64f* xResolution, k64f* zResolution);

/**
 * @brief Given the runtime parameters (FOV and intensity multiplier) and transform, recreate the look up table and write to the FPGA
 * 
 * @param writer            Handler to lut writer object
 * @param transform         Transform to compensate sensor orientations
 * @param params            Runtime parameters (Resolutions and intensity multiplier)
 * @param exportSettings    Flag to enable lut settings export
 * @return kStatus 
 */
kVsFx(kStatus) kL3dLutWriter_Refresh(kL3dLutWriter writer, const kL3dTransform2d* transform, const kL3dLutWriterRunParams* params, kBool exportSettings);

/**
 * @brief Getter functions that return the calibration tables at various stages of processing
 * 
 * Cal: Dense table without extrapolation
 * CalExtrapolated: Dense table with extrapolation, legacy lut will return dense table without extrapolation
 * Array: Subsampled Cal table with extrapolation
 * Lut: Array table scaled to k16s
 * Valid: Table that flags which entires in the table are valid
 */
kVsFx(kArray2) kL3dLutWriter_XCal(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_ZCal(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_XCalExtrapolated(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_ZCalExtrapolated(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_XLut(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_ZLut(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_ValidLut(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_XArray(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_ZArray(kL3dLutWriter writer);
kVsFx(kArray2) kL3dLutWriter_ValidArray(kL3dLutWriter writer);

/**
 * @brief Given a centre and slice (x,y) test whether a valid range look up can be performed
 * 
 * @param writer    Handler to lut writer object
 * @param slice     spot slice value (y)
 * @param centre    spot centre value (x)
 * @return kBool    kTRUE if valid, kFALSE if failed or invalid
 */
kVsFx(kBool) kL3dLutWriter_ValidateCentroid(kL3dLutWriter writer, k32u slice, k64f centre);

/**
 * @brief Given spot slice and centre, perform FPGARL and get world space x and z values
 * 
 * @param writer    Handler to lut writer object
 * @param slice     spot slice value (y)
 * @param centre    spot centre value (x)
 * @param xOutput   Output: World x 
 * @param zOutput   Output: World z
 * @return kBool    kTRUE if valid, kFALSE if failed or invalid
 */
kVsFx(kBool) kL3dLutWriter_GetRanges(kL3dLutWriter writer, k32u slice, k64f centre, k64f* xOutput, k64f* zOutput);

/**
 * @brief Writes out LUT parameters used (Step Size, Intensity, Lut Size) to local sensor storage as a kdat6 file
 * 
 * @param writer        Handler to lut writer object
 * @param params        Pointer to output the camera default parameters
 * @param xArray        Pointer to output x LUT table
 * @param zArray        Pointer to output z LUT table
 * @param validArray    Pointer to output valid table
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriter_ExportSettings(kL3dLutWriter writer, const kRangeLutParams* params, const kRangeLutArray* xArray,
    const kRangeLutArray* zArray, const kRangeLutArray* validArray);

/**
 * @brief Writes out look up tables as csv to local sensor storage
 * 
 * @param writer    Handler to lut writer object
 * @param storage   Storage object from kNode
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriter_ExportLuts(kL3dLutWriter writer, kStorage storage);

/**
 * @brief Save the lut plot to local root directory as kdat6 file
 * 
 * @param writer    Handler to lut writer object
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriter_ExportLutPlot(kL3dLutWriter writer);

/**
 * @brief Create and return lut plot as kPlot object
 * 
 * This class does not own the kPlot object and expects the display class to destroy it
 * 
 * @param writer    Handler to lut writer object
 * @param output    pointer to returned kPlot
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriter_LutPlot(kL3dLutWriter writer, kPlot* output);

#include <kVision/L3d/kL3dLutWriter.x.h>

#endif /* #ifndef K_VISION_L3D_LUTWRITER_H */
