/** 
 * @file    kNodeDef.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NODE_DEF_X_H
#define K_FIRESYNC_NODE_DEF_X_H

typedef struct kCameraInfoParams
{
    kCameraOutputMode outputMode;
    kBool ramImageEnabled; 
    k64f pixelPeriod;
    kCameraReadout readoutMode;
    kBool interruptsEnabled;
    k32u maxColumns;
    k32u width;
    k32u height;
    k32u xSub;
    k32u ySub;
    kCameraSubsampling xSubMode;
    kCameraSubsampling ySubMode;
    const kPoint64f* kneePoints;
    kSize kneeCount;
    kAlgInfo algorithmConfig;
} kCameraInfoParams;

kDeclareEnumEx(kFs, kSystemChange, kValue)

kDeclareEnumEx(kFs, kSyncDomain, kValue)

kDeclareEnumEx(kFs, kEventControl, kValue)

kDeclareEnumEx(kFs, kEventEncoder, kValue)
kDeclareEnumEx(kFs, kEventMissStyle, kValue)
kDeclareValueEx(kFs, kEventStats, kValue)

kDeclareEnumEx(kFs, kCameraModel, kValue)
kDeclareEnumEx(kFs, kCameraControl, kValue)

kDeclareEnumEx(kFs, kCameraExtension, kValue)
kDeclareEnumEx(kFs, kCameraStepsStyle, kValue)
kDeclareEnumEx(kFs, kCameraOverrideType, kValue)
kDeclareValueEx(kFs, kCameraOverrideItem, kValue)
kDeclareEnumEx(kFs, kCameraExposure, kValue)
kDeclareEnumEx(kFs, kCameraPhaseDecoderFormat, kValue)

kDeclareEnumEx(kFs, kCameraPhaseDecoderType, kValue)
kDeclareEnumEx(kFs, kCameraPhaseDecoderLutType, kValue)
kDeclareEnumEx(kFs, kCameraOutputMode, kValue)
kDeclareEnumEx(kFs, kCameraPortMode, kValue)
kDeclareEnumEx(kFs, kCameraReadout, kValue)
kDeclareEnumEx(kFs, kCameraGain, kValue)
kDeclareEnumEx(kFs, kCameraSubsampling, kValue)
kDeclareEnumEx(kFs, kCameraAlgorithm, kValue)

/**
 * @struct  kCameraCapability
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera feature bitset (capabilities).
 */
typedef k32s kCameraCapability; 

/** @relates kCameraCapability @{ */
#define kCAMERA_CAPABILITY_NULL                     (0x00000000)      ///< No capabilities.
#define kCAMERA_CAPABILITY_RAM_IMAGE                (0x00000001)      ///< RAM Image.
#define kCAMERA_CAPABILITY_PRNU                     (0x00000002)      ///< PRNU.
#define kCAMERA_CAPABILITY_FPN                      (0x00000004)      ///< FPN.
#define kCAMERA_CAPABILITY_SPOT_SORT                (0x00000008)      ///< Spot Sorting.
#define kCAMERA_CAPABILITY_RANGE_LUT_8K             (0x00000010)      ///< Range LUT (8K entries).
#define kCAMERA_CAPABILITY_RANGE_LUT_16K            (0x00000020)      ///< Range LUT (16K entries).
#define kCAMERA_CAPABILITY_PHASE_DECODER            (0x00000040)      ///< Original phase decoder hardware acceleration (e.g., CFG 623)
#define kCAMERA_CAPABILITY_MAX_RANGES_4K            (0x00000080)      ///< Support for fixed maxFound values (4K).
#define kCAMERA_CAPABILITY_MAX_SORTED_RANGES_32K    (0x00000100)      ///< Support for sorted maxFound values.
#define kCAMERA_CAPABILITY_INVERT_X                 (0x00000200)      ///< Support for PL-based image x inversion.
#define kCAMERA_CAPABILITY_DVFPN                    (0x00000400)      ///< DVFPN.
#define kCAMERA_CAPABILITY_DYNAMIC_FPN              (0x00000800)      ///< Dynamic FPN extension is supported.
#define kCAMERA_CAPABILITY_XRESAMPLING_2K           (0x00001000)      ///< X-resampling points of 2K is supported.
#define kCAMERA_CAPABILITY_XRESAMPLING_4K           (0x00002000)      ///< X-resampling points of 4K is supported.
#define kCAMERA_CAPABILITY_XRESAMPLING_11K          (0x00004000)      ///< X-resampling points of 11K is supported.
#define kCAMERA_CAPABILITY_PL_DDR_VIDEO             (0x00008000)      ///< Uses PL DDR for video buffer space.
#define kCAMERA_CAPABILITY_DELTA_PHASE_DECODER      (0x00010000)      ///< Phase decoder + delta compression hardware acceleration (e.g., CFG 628)
#define kCAMERA_CAPABILITY_XRESAMPLING_16K          (0x00020000)      ///< X-resampling points of 16K is supported.
#define kCAMERA_CAPABILITY_EPS_DECODER              (0x00040000)      ///< New EPS decoder (e.g., CFG 637)
#define kCAMERA_CAPABILITY_AFW_ALG_FIFO_SIZE_16K    (0x00080000)      ///< Algorithm AFW FIFO size of 16K.
#define kCAMERA_CAPABILITY_AFW_ALG_FIFO_SIZE_32K    (0x00100000)      ///< Algorithm AFW FIFO size of 32K.
#define kCAMERA_CAPABILITY_MAX_RANGES_8K            (0x00200000)      ///< Support for fixed maxFound values (8K).
#define kCAMERA_CAPABILITY_XRESAMPLING_8K           (0x00400000)      ///< X-resampling points of 8K is supported.
#define kCAMERA_CAPABILITY_AFW_ALG_FIFO_SIZE_64K    (0x00800000)      ///< Algorithm AFW FIFO size of 64K.
#define kCAMERA_CAPABILITY_AFW_ALG_FIFO_SIZE_128K   (0x01000000)      ///< Algorithm AFW FIFO size of 128K.
#define kCAMERA_CAPABILITY_PL_SEQUENCING            (0x02000000)      ///< PL sequencing is supported.

/** @} */

/** 
 * Reads a kCameraCapability value from an XML item. 
 *
 * @public                  @memberof kCameraCapability
 * @param   capabilities    kCameraCapability value. 
 * @param   xml             XML object. 
 * @param   item            XML item. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kCameraCapability_Read(kCameraCapability* capabilities, kXml xml, kXmlItem item);

/** 
 * Writes a kCameraCapability value to an XML item. 
 *
 * @public                  @memberof kCameraCapability
 * @param   capabilities    kCameraCapability value. 
 * @param   xml             XML object. 
 * @param   item            XML item. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kCameraCapability_Write(kCameraCapability capabilities, kXml xml, kXmlItem item);

kInlineFx(kBool) kCameraCapability_HasSpotSorting(kCameraCapability capabilities)
{
    return (capabilities & kCAMERA_CAPABILITY_SPOT_SORT) != 0;
}

kInlineFx(kBool) kCameraCapability_HasRangeTranslation(kCameraCapability capabilities)
{
    return (capabilities & (kCAMERA_CAPABILITY_RANGE_LUT_8K | kCAMERA_CAPABILITY_RANGE_LUT_16K)) != 0;
}

kInlineFx(kBool) kCameraCapability_HasXResampling(kCameraCapability capabilities)
{
    return (capabilities & (kCAMERA_CAPABILITY_XRESAMPLING_2K | kCAMERA_CAPABILITY_XRESAMPLING_4K | kCAMERA_CAPABILITY_XRESAMPLING_8K | kCAMERA_CAPABILITY_XRESAMPLING_11K | kCAMERA_CAPABILITY_XRESAMPLING_16K)) != 0;
}

kInlineFx(kBool) kCameraCapability_HasPhaseDecoder(kCameraCapability capabilities)
{
    return (capabilities & 
        (kCAMERA_CAPABILITY_PHASE_DECODER | kCAMERA_CAPABILITY_DELTA_PHASE_DECODER | kCAMERA_CAPABILITY_EPS_DECODER)) != 0;
}

kDeclareEnumEx(kFs, kCameraCapability, kValue)

kDeclareEnumEx(kFs, kCameraSpotSort, kValue)
kDeclareEnumEx(kFs, kAcgSpotFormat, kValue)
kDeclareValueEx(kFs, kRangeLutCaps, kValue)
kDeclareValueEx(kFs, kRangeLutParams, kValue)
kDeclareValueEx(kFs, kRangeLutArray, kValue)
kDeclareEnumEx(kFs, kCameraFrameAlgType, kValue)
kDeclareValueEx(kFs, kCameraImageInfo, kValue)
kDeclareValueEx(kFs, kCameraListInfo, kValue)
kDeclareValueEx(kFs, kCameraProfileInfo, kValue)
kDeclareValueEx(kFs, kCameraBinCSumInfo, kValue)
kDeclareValueEx(kFs, kCameraFrameInfo, kValue)
kDeclareValueEx(kFs, kCameraStats, kValue)
kDeclareEnumEx(kFs, kCameraAccelerationTestMode, kValue)

kDeclareEnumEx(kFs, kCameraInputCounterSource, kValue)

#define kCAMERA_INPUT_COUNTER_SOURCE_NODE_INPUT_0_MASK          (0x0001)
#define kCAMERA_INPUT_COUNTER_SOURCE_NODE_INPUT_0_SHIFT         (0)

#define kCAMERA_INPUT_COUNTER_SOURCE_MASTER_INPUT_0_MASK        (0x0010)
#define kCAMERA_INPUT_COUNTER_SOURCE_MASTER_INPUT_0_SHIFT       (4)

#define kCAMERA_INPUT_COUNTER_SOURCE_CAMERA_0_MASK              (0x0004)
#define kCAMERA_INPUT_COUNTER_SOURCE_CAMERA_0_SHIFT             (2)

#define kCAMERA_INPUT_COUNTER_SOURCE_PROJECTOR_0_MASK           (0x0002)
#define kCAMERA_INPUT_COUNTER_SOURCE_PROJECTOR_0_SHIFT          (1)

#define kCAMERA_INPUT_COUNTER_SOURCE_GATE_MASK                  (0x8000)
#define kCAMERA_INPUT_COUNTER_SOURCE_GATE_SHIFT                 (15)

#define kCAMERA_INPUT_COUNTER_SOURCE_INPUT_COUNTER_MASK         (0x0300)
#define kCAMERA_INPUT_COUNTER_SOURCE_INPUT_COUNTER_SHIFT        (8)

typedef k32s kCameraBufferSpace;

#define kCAMERA_BUFFER_SPACE_NULL                       (0)
#define kCAMERA_BUFFER_SPACE_PS_DDR                     (1)
#define kCAMERA_BUFFER_SPACE_PL_DDR                     (2)

kDeclareEnumEx(kFs, kNodeClearOption, kValue)

kDeclareEnumEx(kFs, kRegisterPermission, kValue)

kDeclareEnumEx(kFs, kLightModel, kValue)

kDeclareEnumEx(kFs, kLightDiodeModel, kValue)

/**
 * @struct  kLightCapability
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents light feature bitset (capabilities).
 */
typedef k32s kLightCapability; 

/** @relates kLightCapability @{ */
#define kLIGHT_CAPABILITY_NULL                     (0x0000)      ///< No capabilities.
#define kLIGHT_CAPABILITY_MASK                     (0x0001)      ///< Light output masking supported. 
#define kLIGHT_CAPABILITY_INTENSITY                (0x0002)      ///< Light intensity adjustment supported. 
/** @} */

/** 
 * Reads a kLightCapability value from an XML item. 
 *
 * @public                  @memberof kLightCapability
 * @param   capabilities    kLightCapability value. 
 * @param   xml             XML object. 
 * @param   item            XML item. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kLightCapability_Read(kLightCapability* capabilities, kXml xml, kXmlItem item);

/** 
 * Writes a kLightCapability value to an XML item. 
 *
 * @public                  @memberof kLightCapability
 * @param   capabilities    kLightCapability value. 
 * @param   xml             XML object. 
 * @param   item            XML item. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kLightCapability_Write(kLightCapability capabilities, kXml xml, kXmlItem item);

kDeclareEnumEx(kFs, kLightCapability, kValue)

kDeclareEnumEx(kFs, kLightControl, kValue)
kDeclareEnumEx(kFs, kLightMask, kValue)

kDeclareEnumEx(kFs, kLightExtension, kValue)
kDeclareEnumEx(kFs, kLightStepsStyle, kValue)
kDeclareEnumEx(kFs, kLightOverrideType, kValue)
kDeclareValueEx(kFs, kLightOverrideItem, kValue)
kDeclareValueEx(kFs, kTempControlEx, kValue)
kDeclareValueEx(kFs, kLightStats, kValue)
kDeclareEnumEx(kFs, kLightPrimingMode, kValue)
kDeclareEnumEx(kFs, kLightCouplingMode, kValue)

kFsFx(kStatus) xkLightModel_LightDriverCurrentLimitInfo(kLightModel model, k32u currentLimit, kVersion revision, kInfo32u* info);

kDeclareEnumEx(kFs, kMemoryTrafficOp, kValue)

kDeclareEnumEx(kFs, kProjectorModel, kValue)
kDeclareEnumEx(kFs, kProjectorControl, kValue)
kDeclareEnumEx(kFs, kProjectorStepsStyle, kValue)

//Deprecated: removed from public API in FSS-686; not presently supported, but could be supported again in future
#define kPROJECTOR_CONTROL_SLAVED_TO_NODE_INPUT     (0x02)      ///< Projector output state slaved to node digital input. 
//kDeprecate(kPROJECTOR_CONTROL_SLAVED_TO_NODE_INPUT)     //remove on next pass?

//Deprecated: removed from public API in FSS-686; not presently supported, but could be supported again in future
#define kPROJECTOR_CONTROL_SLAVED_TO_MASTER_INPUT   (0x04)      ///< Projector output state slaved to master digital input.
//kDeprecate(kPROJECTOR_CONTROL_SLAVED_TO_MASTER_INPUT)   //remove on next pass?

kDeclareEnumEx(kFs, kProjectorExtension, kValue)
kDeclareEnumEx(kFs, kProjectorPatternMode, kValue)
kDeclareEnumEx(kFs, kProjectorPatternSource, kValue)
kDeclareValueEx(kFs, kProjectorStats, kValue)

kDeclareEnumEx(kFs, kEncoderDirection, kValue)

kDeclareEnumEx(kFs, kEncoderSnapshotSource, kValue)

kDeclareValueEx(kFs, kEventManagerType, kValue)

kDeclareEnumEx(kFs, kDigitalOutControl, kValue)

kDeclareValueEx(kFs, kDigitalOutStats, kValue)

kDeclareEnumEx(kFs, kAnalogOutControl, kValue)

kDeclareValueEx(kFs, kAnalogOutStats, kValue)

kDeclareEnumEx(kFs, kSerialOutControl, kValue)

kDeclareEnumEx(kFs, kSerialOutMode, kValue)
kDeclareValueEx(kFs, kSerialOutStats, kValue)

kDeclareEnumEx(kFs, kIoTestModel, kValue)
kDeclareEnumEx(kFs, kIoTestSerialMode, kValue)
kDeclareValueEx(kFs, kIoTestStats, kValue)

kDeclareEnumEx(kFs, kOrientationModel, kValue)
kDeclareEnumEx(kFs, kOrientationAxis, kValue)
kDeclareValueEx(kFs, kAngle3d64f, kValue)
kDeclareValueEx(kFs, kOrientationCalStatus, kValue)
kDeclareValueEx(kFs, kOrientationStats, kValue)
kDeclareValueEx(kFs, kOrientationOffsets, kValue)
kDeclareValueEx(kFs, kOrientationCalProfile, kValue)

kDeclareEnumEx(kFs, kTestJigModel, kValue)
kDeclareEnumEx(kFs, kTestJigCommand, kValue)
kDeclareValueEx(kFs, kTestJigStats, kValue)

kDeclareEnumEx(kFs, kPipeEngine, kValue)

/**
 * @internal
 * @struct  kBlockTypeInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents descriptive information about a block type. 
 */
typedef struct kBlockTypeInfo
{
    kText64 name;                   ///< Block type name. 
    kText256 description;           ///< Block type description.
} kBlockTypeInfo; 

kDeclareValueEx(kFs, kBlockTypeInfo, kValue)

/**
 * @internal
 * @struct  kPortInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents descriptive information about a port. 
 */
typedef struct kPortInfo
{
    k32u id;                        ///< Port id.
    kText32 name;                   ///< Port name.
} kPortInfo; 

kInlineFx(kStatus) kPortInfo_Init(kPortInfo* info, k32u id, const kChar* name)
{
    info->id = id;     

    kCheck(kStrCopy(info->name, sizeof(info->name),  name));

    return kOK; 
}

kDeclareValueEx(kFs, kPortInfo, kValue)

kDeclareEnumEx(kFs, kPortDirection, kValue)
kDeclareValueEx(kFs, kRoutePoint, kValue)
kDeclareValueEx(kFs, kRouteEntry, kValue)
kDeclareValueEx(kFs, kDataInSource, kValue)

kDeclareEnumEx(kFs, kActionTrigger, kValue)

/**
 * @struct  kActionTypeInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents descriptive information about an action type. 
 */
typedef struct kActionTypeInfo
{
    kText64 name;                   ///< Action type name. 
    kText256 description;           ///< Action type description.
} kActionTypeInfo; 

kDeclareValueEx(kFs, kActionTypeInfo, kValue)

kDeclareEnumEx(kFs, kSocType, kValue)

/**
 * @struct  kSpiDeviceType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents SPI device type.
 */
typedef k32s kSpiDeviceType; 

/** @relates kSpiDeviceType @{ */
#define kSPI_DEVICE_TYPE_NULL       (0)                 ///< Unknown SPI device type.
#define kSPI_DEVICE_TYPE_DP03       (1)                 ///< SPI bus for DP03.
#define kSPI_DEVICE_TYPE_OPTIONS    (1)                 ///< Bitset containing all kSpiDeviceType options.
/** @} */

kDeclareEnumEx(kFs, kSpiDeviceType, kValue)
kDeclareEnumEx(kFs, kControllerType, kValue)

#define kNODE_ID_LOCAL_MASK             (0x80000000)
#define kNODE_ID_LOCAL_SHIFT            (31)

#define kNODE_ID_SERIAL_NUMBER_MASK     (0x7FFFFFFF)
#define kNODE_ID_SERIAL_NUMBER_SHIFT    (0)

kDeclareValueEx(kFs, kNodeId, kValue)

kDeclareEnumEx(kFs, kNodeType, kValue)
kDeclareEnumEx(kFs, kAnalogOutModel, kValue)
kDeclareEnumEx(kFs, kDaughterboardModel, kValue)
kDeclareEnumEx(kFs, kNodeFirmwareType, kValue)
kDeclareEnumEx(kFs, kBootMode, kValue)
kDeclareEnumEx(kFs, kRestartReason, kValue)
kDeclareEnumEx(kFs, kLedMode, kValue)

/**
 * @struct  kBootConfig
 * @ingroup kFireSync
 * @brief   Represents boot mode configuration. 
 */
typedef struct kBootConfig
{
    kBootMode mode;                     ///< Desired boot mode.
    kBool waitForRescue;                ///< Wait for rescue request during boot (increases boot time, but can be useful for testing). 
    k64u waitForRescueTimeout;          ///< Timeout for wait for rescue request.
    kBool autoRun;                      ///< Boot automatically, or wait for debug intervention?
    kIpAddress serverAddress;           ///< For network boot, address of FTP server that has firmware. 
    kText256 programPath;               ///< For network boot, path to firmware on FTP server.
    kBool allowWatchdog;                ///< Allow watchdog to be enabled/disabled.
    kBool enableSshd;                   ///< Enable SSH daemon.
    kBool enableCrashLog;               ///< Enable crash logging.
} kBootConfig;

/**
 * Initializes a kBooConfig structure with default values.
 *
 * @public                  @memberof kBootMode
 * @param   config          kBootConfig value.
 * @param   waitForRescue   Initial value for waitForRescue field.
 * @return                  Operation status.
 */
kFsFx(kStatus) kBootConfig_Init(kBootConfig* config, kBool waitForRescue);

kDeclareValueEx(kFs, kBootConfig, kValue)

#define kNODE_DEFAULT_NET_BOOT_SERVER       (0x0A000001)    ///< Default IP address of network boot server (FTP). 

kFsFx(const kChar*) kDefaultNetBootPath(kControllerType controller); 

kDeclareEnumEx(kFs, kSyncSource, kValue)
kDeclareEnumEx(kFs, kNodeState, kValue)

/**
 * @internal
 * @struct  kNodeStateInfo
 * @extends kValue
 * @ingroup kFireSync
 */
typedef struct kNodeStateInfo
{
    kNodeState state;           ///< Node state. 
    k64u configRevision;        ///< Node configuration edit revision.
    k64u time;                  ///< Current FireSync synchronization time.
    k64s encoder;               ///< Current FireSync encoder value.
} kNodeStateInfo; 

kDeclareValueEx(kFs, kNodeStateInfo, kValue)

kDeclareEnumEx(kFs, kRegisterModule, kValue)

/**
 * @internal
 * @struct  kDiscoveryInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents discovery information for a single device. 
 */
typedef struct kDiscoveryInfo
{
    k32u id;                        ///< Device identifier (serial number). 
    kNodeType type;                 ///< Node type. 
    kVersion protocolVersion;       ///< FireSync protocol version. 
    kVersion platformVersion;       ///< FireSync platform firmware version (FS6+ only). 
    k64u upTime;                    ///< Time since node boot (microseconds). 
    kIpConfig ipConfig;             ///< IP network adapter configuration.          
    k16u controlPort;               ///< Control server port number (FS6+ only). 
    k16u healthPort;                ///< Health server port number (FS6+ only). 
    k16u dataPort;                  ///< Data server port number (FS6+ only).
    kNodeStateInfo nodeState;       ///< Information about current node state. 
    kSize nodeInterfaceIndex;       ///< Logical node interface index on which message was sent (relative to kNode_InterfaceCount).
    kSize nodeInterfaceCount;       ///< Total count of node interfaces (same as kNode_InterfaceCount).
    kSize nodeInterfaceReportCount; ///< Total count of node interfaces from which a discovery messages was received in this enumeration cycle.
    kSize localAdapterId;           ///< Unique numeric id of the network adapter that received discovery (kSIZE_NULL if not used or unknown).

} kDiscoveryInfo; 

kDeclareValueEx(kFs, kDiscoveryInfo, kValue)

/**
 * @internal
 * @struct  kDiscoverySetAddressInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents discovery set-address command information. 
 */
typedef struct kDiscoverySetAddressInfo
{
    k32u id;                        ///< Device identifier (serial number). 
    kIpConfig ipConfig;             ///< New address configuration.      
    kSize nodeInterfaceIndex;       ///< Logical node interface index on which message was sent (relative to kNode_InterfaceCount).
    kSize localAdapterId;           ///< Unique numeric id of the network adapter that received discovery (kSIZE_NULL if not used or unknown).
} kDiscoverySetAddressInfo; 

kDeclareValueEx(kFs, kDiscoverySetAddressInfo, kValue)

kDeclareValueEx(kFs, kNodeInfo, kValue)

kDeclareValueEx(kFs, kNodeStats, kValue)

kDeclareValueEx(kFs, kNodeVariableInfo, kValue)

kDeclareEnumEx(kFs, kGpioBankType, kValue)
kDeclareEnumEx(kFs, kGpioPin, kValue)

kDeclareEnumEx(kFs, kLed, kValue)

kDeclareValueEx(kFs, kMsgSource, kValue)

kInlineFx(kBool) kStampStatus_NodeInputState(kStampStatus status, kSize index)
{
    return kField_Extract_(status, kCAMERA_INPUT_COUNTER_SOURCE_NODE_INPUT_0_MASK << index,
                                   kCAMERA_INPUT_COUNTER_SOURCE_NODE_INPUT_0_SHIFT + index); 
}

kInlineFx(kBool) kStampStatus_MasterInputState(kStampStatus status, kSize index)
{
    return kField_Extract_(status, kCAMERA_INPUT_COUNTER_SOURCE_MASTER_INPUT_0_MASK << index,
                                   kCAMERA_INPUT_COUNTER_SOURCE_MASTER_INPUT_0_SHIFT + index);
}

kInlineFx(kBool) kStampStatus_ProjectorState(kStampStatus status, kSize index)
{
    return kField_Extract_(status, kCAMERA_INPUT_COUNTER_SOURCE_PROJECTOR_0_MASK << index,
                                   kCAMERA_INPUT_COUNTER_SOURCE_PROJECTOR_0_SHIFT + index);
}

kInlineFx(kBool) kStampStatus_CameraState(kStampStatus status, kSize index)
{
    return kField_Extract_(status, kCAMERA_INPUT_COUNTER_SOURCE_CAMERA_0_MASK << index,
                                   kCAMERA_INPUT_COUNTER_SOURCE_CAMERA_0_SHIFT + index);
}

kInlineFx(kBool) kStampStatus_GateState(kStampStatus status)
{
    return kField_Extract_(status, kCAMERA_INPUT_COUNTER_SOURCE_GATE_MASK,
                                   kCAMERA_INPUT_COUNTER_SOURCE_GATE_SHIFT);
}

kInlineFx(k32u) kStampStatus_InputCounter(kStampStatus status)
{
    return kField_Extract_(status, kCAMERA_INPUT_COUNTER_SOURCE_INPUT_COUNTER_MASK,
                                   kCAMERA_INPUT_COUNTER_SOURCE_INPUT_COUNTER_SHIFT);
}

#define kSTAMP_SUBFRAME_INDEX_LSB           (0)
#define kSTAMP_SUBFRAME_INDEX_MSB           (14)

#define kSTAMP_SUBFRAME_IS_FINAL_LSB        (15)
#define kSTAMP_SUBFRAME_IS_FINAL_MSB        (15)

kInlineFx(kStampSubframe) kStampSubframe_Create(k16u index, kBool isFinal)
{
    kStampSubframe subframe = 0; 
    
    kField_InsertNamedRange_(&subframe, kSTAMP_SUBFRAME_INDEX, index); 

    //inverted for backwards compatibility
    kField_InsertNamedRange_(&subframe, kSTAMP_SUBFRAME_IS_FINAL, !isFinal);    

    return subframe;
}

kInlineFx(k16u) kStampSubframe_Index(kStampSubframe subframe)
{
    return (k16u) kField_ExtractNamedRange_(subframe, kSTAMP_SUBFRAME_INDEX);
}

kInlineFx(kBool) kStampSubframe_IsFinal(kStampSubframe subframe)
{
    //inverted for backwards compatibility
    return (kBool) !kField_ExtractNamedRange_(subframe, kSTAMP_SUBFRAME_IS_FINAL);
}

kDeclareValueEx(kFs, kStamp, kValue)

kFsFx(kStatus) kStamp_Write5V0(kType type, const void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kStamp_Read5V0(kType type, void* values, kSize count, kSerializer serializer);

kFsFx(kStatus) kStamp_Write6V1(kType type, const void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kStamp_Read6V1(kType type, void* values, kSize count, kSerializer serializer);

kDeclareEnumEx(kFs, kHealthId, kValue)

kDeclareEnumEx(kFs, kTempProbeId, kValue)

kDeclareEnumEx(kFs, kHealthOption, kValue)

kDeclareValueEx(kFs, kHealthStat, kValue)

kFsFx(kStatus) kHealthStat_Write0(kType type, void* values, kSize count, kSerializer serializer); 
kFsFx(kStatus) kHealthStat_Read0(kType type, void* values, kSize count, kSerializer serializer); 
kFsFx(kStatus) kHealthStat_Write1(kType type, void* values, kSize count, kSerializer serializer); 
kFsFx(kStatus) kHealthStat_Read1(kType type, void* values, kSize count, kSerializer serializer); 

kDeclareValueEx(kFs, kProfileStat, kValue)

kFsFx(kStatus) kProfileStat_Write0(kType type, void* values, kSize count, kSerializer serializer); 
kFsFx(kStatus) kProfileStat_Read0(kType type, void* values, kSize count, kSerializer serializer); 
kFsFx(kStatus) kProfileStat_Write1(kType type, void* values, kSize count, kSerializer serializer); 
kFsFx(kStatus) kProfileStat_Read1(kType type, void* values, kSize count, kSerializer serializer); 

kDeclareValueEx(kFs, kLogItem, kValue)

kFsFx(kStatus) kLogItem_Write0(kType type, void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kLogItem_Read0(kType type, void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kLogItem_Write1(kType type, void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kLogItem_Read1(kType type, void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kLogItem_Write2(kType type, void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kLogItem_Read2(kType type, void* values, kSize count, kSerializer serializer);

kDeclareEnumEx(kFs, kInputEventSource, kValue)
kDeclareValueEx(kFs, kInputEvent, kValue)

kFsFx(kStatus) kInputEvent_Write(kType type, void* values, kSize count, kSerializer serializer);
kFsFx(kStatus) kInputEvent_Read(kType type, void* values, kSize count, kSerializer serializer);

kDeclareEnumEx(kFs, kStorageEntry, kValue);

#define kHW_DAUGHTERBOARD_INFO_START_MARKER     (0x46534442)        ///< "FSDB" stands for FireSync DaughterBoard

/**
 * @struct  kHwDaughterboardInfo
 * @ingroup kHardware-Platform
 * @brief   Represents device parameters specific to a daughterboard model.
 */
typedef struct kHwDaughterboardInfo
{
    k32u startMarker;   ///< Must equal kHW_DAUGHTERBOARD_INFO_START_MARKER.
    k32u version;       ///< kHwDaughterboardInfo structure version. 
                        ///<    0 : model(4 bytes), reserved(4)
                        ///<    1 : model(4 bytes), partNumber(4), batch(16), revision(4)
                        ///<    1 : model(4 bytes), partNumber(4), batch(16), revision(4), modelV2(4)
    k32s model;         ///< Daughterboard model, for backward compatibility.
    k32u partNumber;    ///< Part number.
    kText16 batch;      ///< Manufacturing batch number.
    kVersion revision;  ///< Revision number.
    k32s modelV2;       ///< Daughterboard model, used for version 2 or later.
} kHwDaughterboardInfo;

/**
 * @internal
 * @struct  kCameraFrameAlgType
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera frame algorithm type options.
 */
typedef k32s kCameraFrameAlgType; 

/** @relates kCameraFrameAlgType @{ */
#define kCAMERA_FRAME_ALG_TYPE_NULL             (0)     ///< Unknown frame type.
#define kCAMERA_FRAME_ALG_TYPE_LIST             (1)     ///< Simple list of data items (e.g., spots).
#define kCAMERA_FRAME_ALG_TYPE_PROFILE          (2)     ///< Range lookup data.
#define kCAMERA_FRAME_ALG_TYPE_BIN_CSUM         (3)     ///< BinCSum data.
#define kCAMERA_FRAME_ALG_TYPE_STRIPE           (4)     ///< Stripe data.
#define kCAMERA_FRAME_ALG_TYPE_PHASE            (5)     ///< Phase data.
#define kCAMERA_FRAME_ALG_TYPE_COMPRESSED_PHASE (6)     ///< Compressed phase data.
/** @} */

/**
 * @internal
 * @struct  kCameraImageInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents video image frame information. 
 */
typedef struct kCameraImageInfo
{
    k32u placeholder;                               ///< unused
} kCameraImageInfo; 

/**
 * @internal
 * @struct  kCameraListInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents simple list algorithm frame information. 
 */
typedef struct kCameraListInfo
{
    kType itemType;                                 ///< Item type.
    kSize itemCount;                                ///< Item count.
} kCameraListInfo; 

/**
 * @internal
 * @struct  kCameraProfileInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents range lookup frame information. 
 */
typedef struct kCameraProfileInfo
{
    kType pointType;                                ///< Point type.
    kSize pointCount;                               ///< Point count.
    kBool hasSlices;                                ///< Has slices.
    kSize intensityDataOffset;                      ///< Pre-calculated offset.
    kSize sliceDataOffset;                          ///< Pre-calculated offset.
    kBool emitSlices;                               ///< Emit slice data if present.
} kCameraProfileInfo; 

/**
 * @internal
 * @struct  kCameraBinCSumInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents Bin/CSum algorithm frame information. 
 */
typedef struct kCameraBinCSumInfo
{
    kType csumType;                                 ///< CSum type.
    kSize binHeight;                                ///< Binarized data height.
    kSize binWidth;                                 ///< Binarized data width.
    kSize binLength;                                ///< Binarized data length.
    kSize csumLength;                               ///< CSum data length.
} kCameraBinCSumInfo; 

/**
 * @internal
 * @struct  kCameraPhaseInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents phase frame information. 
 */
typedef struct kCameraPhaseInfo
{
    kType stripeType;                               ///< Stripe type (can be null for some formats). 
    kType phaseType;                                ///< Phase type. 
    kSize stripeTotalSize;                          ///< Total stripe frame size (bytes). Includes stamp + content + trailing.
    kSize stripeTrailingSize;                       ///< Stripe frame trailing size (bytes). 
    kSize phaseTotalSize;                           ///< Total phase frame size (bytes). Includes stamp + content + trailing.
    kSize phaseTrailingSize;                        ///< Phase frame trailing size (bytes). 
    kSize width;                                    ///< data width.
    kSize height;                                   ///< data height.
} kCameraPhaseInfo; 

/**
 * @internal
 * @struct  kCameraCompressedPhaseInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents compressed phase frame information. 
 */
typedef struct kCameraCompressedPhaseInfo
{
    kCameraPhaseDecoderFormat format;               ///< Stream format. 
    k32u phasePrediction;                           ///< Prediction value.
    k32u phaseDeltaBitDepth;                        ///< Phase delta bit depth setting.
    kSize packetSize;                               ///< Compressed stream packet size. 
    kSize subframeCapacity;                         ///< Max uncompressed length of subframe (final may be less).
} kCameraCompressedPhaseInfo; 

/**
 * @internal
 * @struct  kCameraFrameInfo
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents camera frame information. 
 */
typedef struct kCameraFrameInfo
{
    kBool imageEnabled;                     ///< Is image data enabled?
    kSize imageTotalSize;                   ///< Total image frame size (bytes). Includes stamp + content + trailing.
    kSize imageTrailingSize;                ///< Image frame trailing size (bytes). 

    kCameraFrameAlgType algType;            ///< Data frame type.
    kSize algTotalSize;                     ///< Total algorithm frame size (bytes). Includes stamp + content + trailing.
    kSize algTrailingSize;                  ///< Algorithm frame trailing size (bytes). 

    kSize maxBatchSize;                     ///< Maximum batch size.
    kSize plXSubShift;                      ///< PL x-subsampling amount, expressed as 2^n. 
    kSize plYSubShift;                      ///< PL y-subsampling amount, expressed as 2^n. 
    kSize imagerXSubShift;                  ///< Imager x-subsampling amount, expressed as 2^n. 
    kSize imagerYSubShift;                  ///< Imager y-subsampling amount, expressed as 2^n. 
    k32u width;                             ///< Imager width, after all subsampling. 
    k32u height;                            ///< Imager height, after all subsampling. 
    kType pixelType;                        ///< Pixel data type.
    kPixelFormat pixelFormat;               ///< Pixel format.
    kCfa cfa;                               ///< Imager color filter array type.
    k32u colorGains[2][2];                  ///< Color gains array.
    k32u stateId;                           ///< State Id.
    k64u totalLeadIn;                       ///< Total lead in (FS ns).
    k64u totalLeadOut;                      ///< Total lead out (FS ns).

    kCameraImageInfo image;                 ///< Image frame info.

    union 
    {
        kCameraListInfo list;                           ///< List algorithm frame info.
        kCameraProfileInfo profile;                     ///< Profile algorithm frame info.
        kCameraBinCSumInfo binCsum;                     ///< Bin/CSum algorithm frame info.
        kCameraPhaseInfo phase;                         ///< Phase decoder frame info.
        kCameraCompressedPhaseInfo compressedPhase;     ///< Compressed phase decoder info.
    } algInfo; 

} kCameraFrameInfo; 

#define kPL_FREQUENCY                (98304000)
#define kPL_CLOCK_TOLERANCE          (0.0001)
#define kPL_CLOCK_TOLERANCE_FACTOR   (1.0 + kPL_CLOCK_TOLERANCE)
#define kSYNC_TICKS_PER_FS_US        (3)
#define kPL_TICKS_PER_SYNC_TICK      (32)
#define kPL_TICKS_PER_FS_US          (kSYNC_TICKS_PER_FS_US * kPL_TICKS_PER_SYNC_TICK)
#define kNS_PER_FS_NS                (0.9765625)
#define kFS_NS_PER_NS                (1.024)
#define kPL_DEG_GRANULARITY_NS       (125) 
#define kPL_DEG_MAX_NS               ((1000 * (k64u)k32U_MAX) / kPL_TICKS_PER_FS_US)    //k32U_MAX pl ticks in fs ns
#define kPL_EVENT_GRANULARITY_NS     (1000) 
#define kPL_EVENT_MAX_NS             ((1000 * (k64u)k32U_MAX) / kSYNC_TICKS_PER_FS_US)  //k32U_MAX sync ticks in fs ns
#define kPL_EVENT_MIN_GAP_NS         (500)              //Minimum allowed gap between two events, recommended by HW team (fs ns). 
#define kDEFAULT_EXPOSURE_NS         (102400000)
#define kSYNC_BOOT_TIME              (15*1000*1000)     // Maximum time for network/master synchronization after boot. 

#define kPROJECTOR_TIMING_GRANULARITY_NS                (1000)

#define kPL_MAX_SEQUENCING_COUNT                        (16)                  //Maximum PL sequencing count

#define kPL_MAX_EXPOSURE_SLAVE_DELAY_NS                 (85250)               //Maximum SoE delay is ~83.327 us for 13-bit register (8191 pl ticks).
                                                                              //85.250 fs ns is used as maximum as it is the largest fs ns value without going over the actual maximum.

#define kPL_MAX_EXPOSURE_SLAVE_HOLD_NS                  (85250)               //Maximum EoE delay is ~83.327 us for 13-bit register (8191 pl ticks).
                                                                              //85.250 fs ns is used as maximum as it is the largest fs ns value without going over the actual maximum.

#define kENCODER_MIN_SIMULATION_RESOLUTION              (0)                   //Minimum resolution. 
#define kENCODER_MAX_SIMULATION_RESOLUTION              (0xFFFF)              //Maximum resolution. 
#define kENCODER_SIMULATION_RESOLUTION_GRANULARITY      (1)                   //Resolution granularity
#define kENCODER_DEFAULT_SIMULATION_RESOLUTION          (4096)                //Default simulator resolution              

#define kxFsTimeToTime_(FS_TIME)     ((FS_TIME) * 1000 / 1024)
#define kxTimeToFsTime_(TIME)        ((TIME) * 1024 / 1000)

#define kSyncTicksToFsUs_(TICKS)     ((TICKS) / kSYNC_TICKS_PER_FS_US)
#define kFsUsToSyncTicks_(FS_US)     ((FS_US) * kSYNC_TICKS_PER_FS_US)

#define kSyncTicksToFsNs_(TICKS)     ((TICKS) * 1000 / kSYNC_TICKS_PER_FS_US)
#define kFsNsToSyncTicks_(FS_NS)     ((FS_NS) * kSYNC_TICKS_PER_FS_US / 1000)

#define kPlTicksToFsUs_(TICKS)       ((TICKS) / kPL_TICKS_PER_FS_US)
#define kFsUsToPlTicks_(FS_US)       ((FS_US) * kPL_TICKS_PER_FS_US)

#define kPlTicksToFsNs_(TICKS)       ((TICKS) * 1000 / kPL_TICKS_PER_FS_US)
#define kFsNsToPlTicks_(FS_NS)       ((FS_NS) * kPL_TICKS_PER_FS_US / 1000)

template<typename T>
T kPhysicalToNominalDelay(T plDelay, T totalLeadIn)
{
    return plDelay + totalLeadIn;
}

template<typename T>
T kNominalToPhysicalDelay(T nominalDelay, T totalLeadIn)
{
    return nominalDelay - totalLeadIn;
}

template<typename T>
T kPhysicalToNominalExposure(T plExposure, T totalLeadIn, T totalLeadOut)
{
    return plExposure + totalLeadOut - totalLeadIn;
}
    
template<typename T>
T kNominalToPhysicalExposure(T nominalExposure, T totalLeadIn, T totalLeadOut)
{
    return nominalExposure + totalLeadIn - totalLeadOut;
}

template<typename T>
T kPhysicalToNominalGap(T plGap, T totalLeadOut)
{
    return plGap - totalLeadOut;
}

template<typename T>
T kNominalToPhysicalGap(T nominalGap, T totalLeadOut)
{
    return nominalGap + totalLeadOut;
}

kInlineFx(kBool) kLightControl_IsSlaved(kLightControl c)
{
    return c == kLIGHT_CONTROL_COUPLED_TO_NODE_INPUT || 
           c == kLIGHT_CONTROL_COUPLED_TO_MASTER_INPUT ||
           c == kLIGHT_CONTROL_COUPLED_TO_CAMERA || 
           c == kLIGHT_CONTROL_COUPLED_TO_PROJECTOR;
}

/*
 * Device access should only be used in a controlled factory environment.
 */
#if defined(K_FIRESYNC_INCLUDE_DEVICE_ACCESS)

#   define kFIRESYNC_DEVICE_ACCESS_KEY        k64U(0x4543495645445346)

#endif

/* 
* Deprecated (Stage 1): not recommended for further use, but not yet announced via kDeprecate
*/

//Deprecated: replace with right-hand values
#define kHEALTH_ID_LD12_TEMPERATURE                 (kHEALTH_ID_LASER_DRIVER_TEMPERATURE)
#define kHEALTH_ID_NET_LINK_STATUS                  (kHEALTH_ID_NET_INTERFACE_LINK_STATUS)
#define kHEALTH_ID_NET_CAPACITY                     (kHEALTH_ID_NET_INTERFACE_CAPACITY)
#define kHEALTH_ID_NET_RATE_TX                      (kHEALTH_ID_NET_INTERFACE_RATE_TX)
#define kHEALTH_ID_NET_RATE_RX                      (kHEALTH_ID_NET_INTERFACE_RATE_RX)
#define kHEALTH_ID_NET_BYTES_TX                     (kHEALTH_ID_NET_INTERFACE_BYTES_TX)
#define kHEALTH_ID_NET_BYTES_RX                     (kHEALTH_ID_NET_INTERFACE_BYTES_RX)
#define kHEALTH_ID_CAMERA_ERROR_HISTORY_FLAGS       (kHEALTH_ID_CAMERA_ERROR_FLAGS_PREVIOUS) 
#define kHEALTH_ID_CAMERA_ERROR_FLAG_ACCUM          (kHEALTH_ID_CAMERA_ERROR_FLAGS_ACCUM) 

#define xkHEALTH_ID_CAMERA_PL_RESET_ATTEMPT_COUNT           (10000)         // Count of PL reset attempts.
#define xkHEALTH_ID_CAMERA_PL_RESET_RETRY_COUNT             (10001)         // Count of PL reset retry.


//[Deprecated] Replace with kLIGHT_CONTROL_COUPLED_TO_NODE_INPUT
#define kLIGHT_CONTROL_SLAVED_TO_NODE_INPUT     (kLIGHT_CONTROL_COUPLED_TO_NODE_INPUT)   

//[Deprecated] Replace with kLIGHT_CONTROL_COUPLED_TO_MASTER_INPUT
#define kLIGHT_CONTROL_SLAVED_TO_MASTER_INPUT   (kLIGHT_CONTROL_COUPLED_TO_MASTER_INPUT) 

//[Deprecated] Replace with kLIGHT_CONTROL_COUPLED_TO_CAMERA
#define kLIGHT_CONTROL_SLAVED_TO_CAMERA         (kLIGHT_CONTROL_COUPLED_TO_CAMERA)       

//[Deprecated] Replace with kLIGHT_CONTROL_COUPLED_TO_PROJECTOR
#define kLIGHT_CONTROL_SLAVED_TO_PROJECTOR      (kLIGHT_CONTROL_COUPLED_TO_PROJECTOR)    

//[Deprecated] Replace with kCAMERA_CONTROL_COUPLED_TO_NODE_INPUT
#define kCAMERA_CONTROL_SLAVED_TO_NODE_INPUT        (kCAMERA_CONTROL_COUPLED_TO_NODE_INPUT)  

//[Deprecated] Replace with kCAMERA_CONTROL_COUPLED_TO_MASTER_INPUT
#define kCAMERA_CONTROL_SLAVED_TO_MASTER_INPUT      (kCAMERA_CONTROL_COUPLED_TO_MASTER_INPUT)

//[Deprecated] Replace with kCAMERA_CONTROL_COUPLED_TO_CAMERA
#define kCAMERA_CONTROL_SLAVED_TO_CAMERA            (kCAMERA_CONTROL_COUPLED_TO_CAMERA)      

//[Deprecated] Replace with kCAMERA_CONTROL_COUPLED_TO_PROJECTOR
#define kCAMERA_CONTROL_SLAVED_TO_PROJECTOR         (kCAMERA_CONTROL_COUPLED_TO_PROJECTOR)   

//[Deprecated] Replace with kDIGITAL_OUT_CONTROL_COUPLED_TO_NODE_INPUT
#define kDIGITAL_OUT_CONTROL_SLAVED_TO_NODE_INPUT   (kDIGITAL_OUT_CONTROL_COUPLED_TO_NODE_INPUT)

//[Deprecated] Replace with kDIGITAL_OUT_CONTROL_COUPLED_TO_PROJECTOR
#define kDIGITAL_OUT_CONTROL_SLAVED_TO_PROJECTOR    (kDIGITAL_OUT_CONTROL_COUPLED_TO_PROJECTOR) 

#endif
