/** 
 * @file    kUpgrader.h
 * @brief   Declares the kUpgrader class. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kFS_UPGRADER_H
#define kFS_UPGRADER_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/Data/kArchive.h>
#include <kFireSync/Upgrade/kUpgActionRunner.h>
#include <kFireSync/Upgrade/kUpgCompat5.h>
#include <kApi/Utils/kEvent.h>
#include <kFireSync/Client/kSystem.h>
#include <kFireSync/Client/kNode.h>
#include <kFireSync/Data/kPropertyList.h>

/**
 * Performs upgrade using a supplied upgrade package.
 *
 * @class       kUpgrader
 * @extends     kObject
 * @ingroup     kFireSync-Upgrade
 * @brief       Implements device upgrade facility.
 * 
 * The kUpgrader class can be used to upgrade one or more sensors or perform upgrade package validation.
 */
typedef kObject kUpgrader;

/** @cond */
kDeclareClassEx(kFs, kUpgrader, kObject)
kDeclareValueEx(kFs, kUpgraderNode, kValue)
/** @endcond */

/** 
 * Constructs an upgrader object.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Destination for the constructed object handle.
 * @param   alloc       Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_Construct(kUpgrader* upgrader, kAlloc alloc);

/** 
 * Adds a node and the associated properties. For upgrade package validation the node handle can be kNULL.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @param   node        Node handle (or kNULL for upgrade package validation). 
 * @param   properties  Map of node properties. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_AddNode(kUpgrader upgrader, kNode node, kPropertyList properties);

/** 
 * Returns the number of nodes associated with the upgrader object.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @return              Returns the number of nodes.
 */
kFsFx(kSize) kUpgrader_NodeCount(kUpgrader upgrader);

/** 
 * Clears the nodes associated with the upgrader.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_ClearNodes(kUpgrader upgrader);

/** 
 * Sets the ID of the local node.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @param   localNode   ID of the local node. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_SetLocalNode(kUpgrader upgrader, k32u localNode);

/** 
 * Skips validation when applying the upgrade package.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @param   skip        Sets the skip value. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_SkipValidation(kUpgrader upgrader, kBool skip);

/** 
 * Applies the upgrade package to all nodes. Use kUpgrader_AddNode to add nodes prior to calling this function.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @param   system      System object. 
 * @param   archive     Archive object. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_Run(kUpgrader upgrader, kSystem system, kArchive archive);

/** 
 * Runs the validation only, without actually applying the upgrade package. 
 *
 * Use kUpgrader_AddNode prior to calling this function and set the node argument to kNULL.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @param   archive     Archive object. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_Validate(kUpgrader upgrader, kArchive archive);

/** 
 * Runs the validation only, without actually applying the upgrade package. 
 *
 * Use kUpgrader_AddNode prior to calling this function and set the node argument to kNULL.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @param   fileName    Path to the archive file. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kUpgrader_ValidateFile(kUpgrader upgrader, const kChar* fileName);

/** 
 * Returns the current progress progress as value between 0 and 100.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @return              Returns the progress.
 */
kFsFx(k32u) kUpgrader_Progress(kUpgrader upgrader);

/** 
 * Allows to subscribe for log messages.
 *
 * The argument recieved by the event handler is of type kUpgLogMsg.
 *
 * @public              @memberof kUpgrader
 * @param   upgrader    Upgrader object. 
 * @return              Returns the even handle.
 */
kFsFx(kEvent) kUpgrader_LogEvent(kUpgrader upgrader);

#endif
