/** 
 * @file    kSvDataOut.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYC_SERVER_DATA_OUT_X_H
#define K_FIRESYC_SERVER_DATA_OUT_X_H

#define kSV_DATA_OUT_MAX_CONNECTIONS             (8)             ///< Maximum simultaneous connnections (includes app channels; at most one).

#define kSV_DATA_OUT_DEFAULT_PORT_COUNT          (1)             ///< Default number of output block ports. 
#define kSV_DATA_OUT_DEFAULT_MAX_QUEUE_SIZE      (1 << 23)       ///< Default maximum per-channel queue size (bytes). 
#define kSV_DATA_OUT_DEFAULT_MAX_QUEUE_COUNT     (1 << 12)       ///< Default maximum per-channel queue item count. 

#define kSV_DATA_OUT_QUIT_QUERY_PERIOD           (100000)        ///< Polling interval for quit (stop) status (us). 

#define kSV_DATA_OUT_MONITOR_PERIOD              (10000000)      ///< Period at which stall-monitor thread runs (us). 

#define kSV_DATA_OUT_TPUT_AVERAGE_WINDOW         (500000)        ///< Time averaging window for throughput calculations (us).
#define kSV_DATA_OUT_TPUT_AVERAGE_MIN_SAMPLE     (100000)        ///< Minimum time between throughput samples (us).

#define kSV_DATA_OUT_CLIENT_READ_BUFFER          (4096)          ///< Irrelevant to performance
#define kSV_DATA_OUT_SOCKET_READ_BUFFER          (-1)            ///< Irrelevant to performance

#if defined(K_TI_BIOS)
#   define kSV_DATA_OUT_DEFAULT_SOCKET_WRITE_BUFFER      (32*1024)       ///< Socket write buffer size, in bytes.
#elif defined(K_VX_KERNEL)
#   define kSV_DATA_OUT_DEFAULT_SOCKET_WRITE_BUFFER      (128*1024)      ///< Socket write buffer size, in bytes.
#else
#   define kSV_DATA_OUT_DEFAULT_SOCKET_WRITE_BUFFER      (256*1024)      ///< Socket write buffer size, in bytes.
#endif

typedef struct kSvDataOutClass
{
    kObjectClass base; 
    kSvNode node;                       //Parent/owner.
    kIpEndPoint localEndPoint;          //Local server IP endpoint.

    kSize portCount;                    //Count of output block ports. 
    kSize maxQueueSize;                 //Per-channel maximum output queue size (bytes). 
    kSize maxQueueCount;                //Per-channel maximum output queue count (items). 
    kSize socketWriteBufferSize;        //Per-channel socket write buffer size. 

    kThreadPriorityClass threadPriorityClass;   //Desired thread priority class for network streaming threads (low, med, high).
    k32s threadPriorityOffset;                  //Desired thread priority offset for network streaming threads.
    kBitArray threadAffinity;                   //Desired thread affinity for network streaming threads.

    kThread serverThread;               //Thread for accepting new connections.
    kTcpServer server;                  //TCP server to accept new connections.

    kPeriodic monitor;                  //Periodic callback to monitor stalled connection check and health update.

    kList connections;                  //List of connections -- kList<kSvDataOutChannel>.
    kList connectionLock; 

    kAtomic32s shouldQuit;              //Is the server stopping?

    kPxIoBlock block;                   //Output block to accept data from pipe. 

    kArrayList channelMap;              //Maps block port indices to output channel lists -- kArrayList<kArrayList<kSvDataOutChannel>>.
    kLock channelMapLock;               //Provides exclusive access to channel map. 

    kSvDataOutStats stats;              //Run-time statistics. 
    kLock statsLock;                    //Provides exclusive access to run-time statistics. 
    k64u channelBytesWritten;           //Cumulative bytes written, as reported by channels. 
    k64u channelBytesEnqueued;          //Cumulative bytes enqueued, as reported by channels. 
    k64u channelItemsEnqueued;          //Cumulative items enqueued, as reported by channels. 
    k64u channelDropCount;              //Cumulative messages dropped, as reported by channels. 

    kTimeAvg throughputAvg;             //Moving average stats for throughput. 

    kHealthProbe bytesWrittenProbe;     //Health probe for total bytes written by all channels.  
    kHealthProbe bytesEnqueuedProbe;    //Health probe for current bytes enqueued by all channels.  
    kHealthProbe itemsEnqueuedProbe;    //Health probe for current items enqueued by all channels.  
    kHealthProbe throughputProbe;       //Health probe for total throughput (bytes/sec). 
    kHealthProbe messageDropProbe;      //Health probe for total messages dropped by all channels. 
    kHealthProbe totalMessageDropProbe; //Health probe for total messages dropped by all channels since log start. 
} kSvDataOutClass; 

kDeclareClassEx(kFs, kSvDataOut, kObject)
        
kFsFx(kStatus) kSvDataOut_Init(kSvDataOut service, kType type, kSvNode node, kAlloc alloc);
kFsFx(kStatus) kSvDataOut_VRelease(kSvDataOut service);

kFsFx(kStatus) kSvDataOut_Parse(kSvDataOut service, kXml xml, kXmlItem item); 
kFsFx(kStatus) kSvDataOut_Format(kSvDataOut service, kXml xml, kXmlItem item); 

kFsFx(kSvNode) kSvDataOut_Node(kSvDataOut service); 

kFsFx(kStatus) kSvDataOut_ServerThreadEntry(kSvDataOut service); 

kFsFx(kStatus) kSvDataOut_OnMonitorPeriod(kSvDataOut service, kPeriodic timer); 

kFsFx(kStatus) kSvDataOut_AddNetChannel(kSvDataOut service, kTcpClient client);
kFsFx(kStatus) kSvDataOut_RemoveFirstNetChannel(kSvDataOut service); 
kFsFx(kStatus) kSvDataOut_RemoveInactiveChannels(kSvDataOut service); 

kFsFx(kStatus) kSvDataOut_SetSocketBufferSize(kSvDataOut service, kSize size);
kFsFx(kSize) kSvDataOut_SocketBufferSize(kSvDataOut service);

kFsFx(kStatus) kSvDataOut_SetMaxQueueSize(kSvDataOut service, kSize size); 
kFsFx(kSize) kSvDataOut_MaxQueueSize(kSvDataOut service); 

kFsFx(kStatus) kSvDataOut_SetMaxQueueCount(kSvDataOut service, kSize count); 
kFsFx(kSize) kSvDataOut_MaxQueueCount(kSvDataOut service); 

kFsFx(kStatus) kSvDataOut_SetThreadPriority(kSvDataOut service, kThreadPriorityClass priorityClass, k32s priorityOffset); 
kFsFx(kStatus) kSvDataOut_SetThreadAffinity(kSvDataOut service, kBitArray affinity); 

kFsFx(kStatus) kSvDataOut_UpdateBlock(kSvDataOut service); 

kFsFx(kStatus) kSvDataOut_OnData(kSvDataOut service, kPxPort port, kMsgInfo message); 

kFsFx(kStatus) kSvDataOut_RegisterChannelPorts(kSvDataOut service, kSvDataOutChannel channel, const k16u* ports, kSize portCount); 
kFsFx(kStatus) kSvDataOut_UnregisterChannelPorts(kSvDataOut service, kSvDataOutChannel channel); 

kFsFx(kStatus) kSvDataOut_OnHealthUpdate(kSvDataOut service, kObject sender, kPointer unused);

kFsFx(kStatus) kSvDataOut_AddChannelStats(kSvDataOut service, k64u newBytesWritten, k64u newMessagesDropped, k64u currentBytesEnqueued, k64u currentItemsEnqueued); 

kFsFx(kStatus) kSvDataOut_LogChannelMap(kSvDataOut service); 

#endif
