/** 
 * @file    kPxParallelPipe.x.h
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_PARALLEL_PIPE_X_H
#define K_FIRESYNC_PX_PARALLEL_PIPE_X_H

#include <kFireSync/Pipe/kPxPipe.h>
#include <kApi/Threads/kMsgQueue.h>

/*
 * kPxParallelPipeProcessItem
 */

/**
 * @internal
 * @struct  kPxParallelPipeProcessItem
 * @extends kValue
 * @ingroup kFireSync-Pipe
 * @brief   Represents a message item that has been received and enqueued by a kPxParallelPipeProcess object. 
 */
typedef struct kPxParallelPipeProcessItem
{
    kMsgInfo message;           ///< Message object. 
    kSize size;                 ///< Calculated size of message object. 
} kPxParallelPipeProcessItem;

kDeclareValueEx(kFs, kPxParallelPipeProcessItem, kValue)

/*
 * kPxParallelPipeProcess
 */

 /**
  * @internal
  * @class   kPxParallelPipeProcess
  * @extends kObject
  * @ingroup kFireSync-Pipe
  * @brief   Represents a single process (collection of tightly-coupled blocks) within a kPxParallelPipe object. 
  */
typedef kObject kPxParallelPipeProcess; 

typedef struct kPxParallelPipeProcessClass
{
    kObjectClass base;
    kPxParallelPipe pipe;                   //Owner.
    k32u processId;                         //Process group identifier. 
    kAlloc messageAlloc;                    //Message allocator. 
    kArrayList blocks;                      //List of blocks associated with this process -- kArrayList<kPxBlock>. 
    kLock lock;                             //Protects state shared with other threads (externalInputMessages, serviceRequested). 
    kQueue externalInputMessages;           //Messages received from outside of the pipe, but not yet processed -- kQueue<kPxParallelPipeProcessItem>. 
    kQueue internalInputMessages;           //Messages received from inside of the pipe, but not yet processed -- kQueue<kPxParallelPipeProcessItem>. 
    kBool serviceRequested;                 //Has this process requested service from the pipe (and not yet been serviced)?
    kQueue localMessages;                   //Locally-routed messages generated during process execution -- kQueue<kMsgInfo>. 
    k32u priority;                          //Normalized execution priority of this process [0...999]. 
} kPxParallelPipeProcessClass;

kDeclareClassEx(kFs, kPxParallelPipeProcess, kObject)

kFsFx(kStatus) kPxParallelPipeProcess_Construct(kPxParallelPipeProcess* process, kPxParallelPipe pipe, k32u processId, kAlloc allocator);
kFsFx(kStatus) kPxParallelPipeProcess_Init(kPxParallelPipeProcess process, kType type, kPxParallelPipe pipe, k32u processId, kAlloc alloc);
kFsFx(kStatus) kPxParallelPipeProcess_VRelease(kPxParallelPipeProcess process);

kFsFx(kStatus) kPxParallelPipeProcess_AddBlock(kPxParallelPipeProcess process, kPxBlock block);

kFsFx(kStatus) kPxParallelPipeProcess_Start(kPxParallelPipeProcess process);
kFsFx(kStatus) kPxParallelPipeProcess_Stop(kPxParallelPipeProcess process);

kFsFx(kStatus) kPxParallelPipeProcess_EnqueueExternalInput(kPxParallelPipeProcess process, kMsgInfo message);
kFsFx(kStatus) kPxParallelPipeProcess_EnqueueInternalInput(kPxParallelPipeProcess process, kMsgInfo message); 
kFsFx(kStatus) kPxParallelPipeProcess_EnqueueMsgOnQueue(kPxParallelPipeProcess process, kMsgInfo message, kQueue queue); 

kFsFx(kStatus) kPxParallelPipeProcess_RequestService(kPxParallelPipeProcess process);
kFsFx(kStatus) kPxParallelPipeProcess_RequestServiceOnQueue(kPxParallelPipeProcess process, kQueue queue);

kFsFx(kStatus) kPxParallelPipeProcess_Execute(kPxParallelPipeProcess process);
kFsFx(kStatus) kPxParallelPipeProcess_CompleteService(kPxParallelPipeProcess process);
kFsFx(kBool) kPxParallelPipeProcess_HasInput(kPxParallelPipeProcess process);

kFsFx(k32u) kPxParallelPipeProcess_Id(kPxParallelPipeProcess process);

kInlineFx(k32u) kPxParallelPipeProcess_Priority(kPxParallelPipeProcess process)
{
    kObj(kPxParallelPipeProcess, process);

    return obj->priority; 
}

kFsFx(kStatus) kPxParallelPipeProcess_OnBlockSend(kPxParallelPipeProcess process, kPxPort port, kMsgSet msg);
kFsFx(kStatus) kPxParallelPipeProcess_OnBlockSendSync(kPxParallelPipeProcess process, kPxPort port, kMsgSet msg); 
kFsFx(kStatus) kPxParallelPipeProcess_OnBlockSendAsync(kPxParallelPipeProcess process, kPxPort port, kMsgSet msg);

kFsFx(kStatus) kPxParallelPipeProcess_ClearInputQueue(kPxParallelPipeProcess process, kQueue queue);

kFsFx(kStatus) kPxParallelPipeProcess_CalculatePriority(kPxParallelPipeProcess process);
kFsFx(kStatus) kPxParallelPipeProcess_CalculateRouteLength(kPxParallelPipeProcess process, kPxBlock block, kPortDirection direction, kSize* length); 
kFsFx(kStatus) kPxParallelPipeProcess_CalculateRouteLengthHelper(kPxBlock block, kPortDirection direction, kArrayList processRoutes, kArrayList blockPath, kArrayList processPath);

/*
 * kPxParallelPipe
 */

#define kPX_PARALLEL_PIPE_QUIT_QUERY_PERIOD     (100000)          ///< Polling interval for quit (stop) status (us). 

typedef struct kPxParallelPipeClass
{
    kPxPipeClass base;

    kAlloc messageAlloc;                //Message allocator. 
    kArrayList blocks;                  //List of processing blocks -- kArrayList<kPxBlock>. 
    kArrayList routes;                  //List of routes -- kArrayList<kRouteEntry>. 
    kArrayList processes;               //List of processes -- kArrayList<kParallelPipeProcess>. 
    kArrayList threadPool;              //List of data processing threads -- kArrayList<kThread>. 
    kBool shouldQuit;                   //Thread quit flag. 
    kAtomic32s executionStatus;         //Pipe execution status.

    kList jobQueue;                     //Priority-queue of processes -- kList<kPxParallelPipeProcess>. 
    kLock jobQueueLock;                 //Provides exclusive access to job queue. 
    kSemaphore jobQueueSem;             //Job queue semaphore (counts processes in job queue). 

    kLock statLock;                     //Provides exclusive access to statistics (queueSize, queueCount). 
    kSize maxQueueSize;                 //Maximum message bytes that can be enqueued. 
    kSize queueSize;                    //Count of message bytes enqueued but not yet submitted for processing, across all processes. 
    kSize maxQueueCount;                //Maximum count of items that can be enqueued. 
    kSize queueCount;                   //Count of messages currently enqueued but not yet submitted for processing, across all processes. 

    kHealthProbe messageDropProbe;      //Health probe for message drops. 
    kHealthProbe queueSizeProbe;        //Health probe for bytes of data pending processing.
    kHealthProbe queueCountProbe;       //Health probe for messages of input data pending processing.
    kHealthProbe executionStatusProbe;  //Health probe for execution status.
    kHealthProbe totalMessageDropProbe; //Health probe for total message drops since start of the health log. 
    kHealthProbe executionErrorTotalProbe; //Health probe for total execution errors since start of the health log. 

} kPxParallelPipeClass; 

kDeclareClassEx(kFs, kPxParallelPipe, kPxPipe)
        
kFsFx(kStatus) kPxParallelPipe_Init(kPxParallelPipe pipe, kType type, k32u nodeId, kPxEnviron pipeEnviron, kAlloc alloc); 

kFsFx(kStatus) kPxParallelPipe_VRelease(kPxParallelPipe pipe);

kFsFx(kStatus) kPxParallelPipe_VSetMaxQueueSize(kPxParallelPipe pipe, kSize size); 
kFsFx(kStatus) kPxParallelPipe_VSetMaxQueueCount(kPxParallelPipe pipe, kSize count); 
kFsFx(kStatus) kPxParallelPipe_VClear(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_VAddBlocks(kPxParallelPipe pipe, const kPxBlock* blocks, kSize count);
kFsFx(kStatus) kPxParallelPipe_VAddRoutes(kPxParallelPipe pipe, const kRouteEntry* routes, kSize count); 
kFsFx(kStatus) kPxParallelPipe_VStart(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_VEngage(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_VStop(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_VPause(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_VResume(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_VReplay(kPxParallelPipe pipe, kMsgInfo message);
kFsFx(kStatus) kPxParallelPipe_VUpdateStats(kPxParallelPipe pipe);

kFsFx(kStatus) kPxParallelPipe_ResetQueueStats(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_ClearStats(kPxParallelPipe pipe);

kFsFx(kStatus) kPxParallelPipe_ClearConnections(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_EnumerateConnections(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_FindPort(kPxParallelPipe pipe, k32u blockId, k32u portId, kPxPort* port); 
kFsFx(kStatus) kPxParallelPipe_FindProcess(kPxParallelPipe pipe, k32u processId, kPxParallelPipeProcess* process);

kFsFx(kStatus) kPxParallelPipe_StartBlocks(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_EngageBlocks(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_StopBlocks(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_PauseBlocks(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_ResumeBlocks(kPxParallelPipe pipe); 

kFsFx(kStatus) kPxParallelPipe_StartProcessing(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_StopProcessing(kPxParallelPipe pipe);
kFsFx(kStatus) kPxParallelPipe_ProcessThread(kPxParallelPipe pipe);

kFsFx(kBool) kPxParallelPipe_CanAcceptMessage(kPxParallelPipe pipe); 

kFsFx(kStatus) kPxParallelPipe_RequestService(kPxParallelPipe pipe, kPxParallelPipeProcess process);
kFsFx(kStatus) kPxParallelPipe_OnIoBlockSend(kPxParallelPipe pipe, kPxPort port, kMsgSet msg);
kFsFx(void) kPxParallelPipe_RouteMessage(kPxParallelPipe pipe, kPxPort port, kMsgSet msg, kBool isExternal); 
kFsFx(kStatus) kPxParallelPipe_SubmitMessage(kPxParallelPipe pipe, kMsgInfo msgInfo, kBool isExternal);

kFsFx(kStatus) kPxParallelPipe_NotifyDrop(kPxParallelPipe pipe); 
kFsFx(kStatus) kPxParallelPipe_NotifyItemEnqueued(kPxParallelPipe pipe, kSize size);
kFsFx(kStatus) kPxParallelPipe_NotifyItemDequeued(kPxParallelPipe pipe, kSize size);

#endif
